package cn.geminis.core.util;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.DomDriver;
import org.springframework.util.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

/**
 * @author puddi
 */
public class XmlUtils {

    private static final XStream X_STREAM;

    static {
        X_STREAM = new XStream(new DomDriver());
    }

    public static String encode(Object obj, String rootName) {
        X_STREAM.alias(rootName, obj.getClass());
        return X_STREAM.toXML(obj);
    }

    public static <T> T decode(String xml) {
        return (T) X_STREAM.fromXML(xml);
    }

    private static Document readDocument(String xml) {
        var factory = DocumentBuilderFactory.newInstance();
        try (var stream = new ByteArrayInputStream(xml.getBytes())) {
            var builder = factory.newDocumentBuilder();
            return builder.parse(stream);
        } catch (IOException e) {
            throw new RuntimeException("关闭流错误", e);
        } catch (SAXException e) {
            throw new RuntimeException("解析xml错误", e);
        } catch (ParserConfigurationException e) {
            throw new RuntimeException("构造DocumentBuilder错误", e);
        }
    }

    private static String writeDocument(Document document) {
        var source = new DOMSource(document);
        try (var stream = new ByteArrayOutputStream()) {
            var result = new StreamResult(stream);
            var factory = TransformerFactory.newInstance();
            var trans = factory.newTransformer();
            trans.transform(source, result);
            return stream.toString();
        } catch (IOException | TransformerException e) {
            throw new RuntimeException("生成XML错误", e);
        }
    }

    private static Element getElement(Document document, String path) {
        var element = document.getDocumentElement();
        if (!StringUtils.isEmpty(path)) {
            var paths = path.split("\\.");
            for (var i = 0; i < paths.length; i++) {
                var elements = element.getElementsByTagName(paths[i]);
                if (elements.getLength() == 0) {
                    throw new RuntimeException("未找到XML节点：" + paths[i]);
                }
                element = (Element) elements.item(0);
            }
        }
        return element;
    }

    public static String setAttribute(String xml, String path, String attributeName, String value) {
        var document = readDocument(xml);
        var element = getElement(document, path);
        element.setAttribute(attributeName, value);
        return writeDocument(document);
    }

    public static String getAttribute(String xml, String path, String attributeName) {
        var document = readDocument(xml);
        var element = getElement(document, path);
        return element.getAttribute(attributeName);
    }

}
