package cn.geminis.core.util;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;
import java.util.UUID;

/**
 * @author Allen
 */
public class ByteUtils {

    private static final int DATA_LENGTH_LONG = 8;
    private static final int DATA_LENGTH_INT = 4;
    private static final int DATA_LENGTH_SHORT = 2;

    public static byte[] duplicate(byte[] data, int length) {
        var result = new byte[length];
        var index = 0;
        do {
            System.arraycopy(data, 0, result, index, index + data.length > length ? length - index : data.length);
            index += data.length;
        } while (index < length);
        return result;
    }

    public static byte[] resize(byte[] data, int length) {
        return Arrays.copyOf(data, length);
    }

    public static boolean compare(byte[] value1, byte[] value2) {
        if (value1 == null && value2 == null) {
            return true;
        }
        if (value1 != null && value2 == null) {
            return false;
        }
        if (value1 == null && value2 != null) {
            return false;
        }
        if (value1 == value2) {
            return true;
        }

        if (value1.length != value2.length) {
            return false;
        }

        for (var i = 0; i < value1.length; i++) {
            if (value1[i] != value2[i]) {
                return false;
            }
        }

        return true;
    }

    public static byte[] fromUUID(UUID uuid) {
        var bb = ByteBuffer.wrap(new byte[16]);
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        return bb.array();
    }

    public static UUID toUUID(byte[] bytes) {
        var bb = ByteBuffer.wrap(bytes);
        long firstLong = bb.getLong();
        long secondLong = bb.getLong();
        return new UUID(firstLong, secondLong);
    }

    public static String toHex(byte[] data) {
        if (Objects.isNull(data) || (data.length == 0)) {
            return null;
        }

        var stringBuilder = new StringBuilder();
        for (var i = 0; i < data.length; i++) {
            var v = data[i] & 0xFF;
            var hv = Integer.toHexString(v);
            if (hv.length() < 2) {
                stringBuilder.append(0);
            }
            stringBuilder.append(hv);
        }
        return stringBuilder.toString();
    }

    public static byte[] fromHex(String data) {
        if (Objects.isNull(data) || data.length() == 0) {
            return null;
        }

        var len = data.length();
        var b = new byte[len / 2];
        for (var i = 0; i < len; i += 2) {
            // 两位一组，表示一个字节,把这样表示的16进制字符串，还原成一个字节
            b[i / 2] = (byte) ((Character.digit(data.charAt(i), 16) << 4) + Character
                    .digit(data.charAt(i + 1), 16));
        }
        return b;
    }

    public static byte[] sub(byte[] data, int from, int length) {
        return Arrays.copyOfRange(data, from, from + length);
    }

    public static byte[] sub(byte[] data, int from) {
        return sub(data, from, data.length - from);
    }

    public static byte[] fromLong(long value) {
        var buffer = ByteBuffer.allocate(DATA_LENGTH_LONG);
        buffer.putLong(value);
        return buffer.array();
    }

    public static long toLong(byte[] data) {
        return toLong(data, 0);
    }

    public static long toLong(byte[] data, int index) {
        var buffer = ByteBuffer.wrap(data);
        return buffer.getLong(index);
    }

    public static byte[] fromInt(int value) {
        var buffer = ByteBuffer.allocate(DATA_LENGTH_INT);
        buffer.putInt(value);
        return buffer.array();
    }

    public static int toInt(byte[] data) {
        return toInt(data, 0);
    }

    public static int toInt(byte[] data, int index) {
        var buffer = ByteBuffer.wrap(data);
        return buffer.getInt(index);
    }

    public static byte[] fromShort(short value) {
        var buffer = ByteBuffer.allocate(DATA_LENGTH_SHORT);
        buffer.putShort(value);
        return buffer.array();
    }

    public static short toShort(byte[] data) {
        return toShort(data, 0);
    }

    public static short toShort(byte[] data, int index) {
        var buffer = ByteBuffer.wrap(data);
        return buffer.getShort(index);
    }


    public static short toShortBE(byte[] data) {
        var buffer = ByteBuffer.wrap(data);
        buffer.order(ByteOrder.BIG_ENDIAN);
        return buffer.getShort();
    }

    public static byte[] fromShortBE(short value) {
        var buffer = ByteBuffer.allocate(DATA_LENGTH_SHORT);
        buffer.order(ByteOrder.BIG_ENDIAN);
        buffer.putShort(value);
        return buffer.array();
    }

    public static byte[] reverse(byte[] value) {
        var data = new byte[value.length];
        for (var i = 0; i < value.length; i++) {
            data[i] = value[value.length - i - 1];
        }
        return data;
    }
}
