package cn.geminis.crypto.csp.soft.rsa;

import cn.geminis.core.util.ByteUtils;
import cn.geminis.core.util.FileUtils;
import cn.geminis.crypto.core.key.PrivateKey;
import cn.geminis.crypto.core.key.PublicKey;
import cn.geminis.crypto.csp.*;
import cn.geminis.crypto.csp.soft.SoftRandomGenerator;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.nist.NISTObjectIdentifiers;
import org.bouncycastle.asn1.oiw.OIWObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x9.ECNamedCurveTable;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECNamedDomainParameters;

import java.security.SecureRandom;

/**
 * @author Allen
 */
public class RsaCspFactory extends AbstractCspFactory {

    private static final String PRIVATE_KEY_FILE_NAME = "rsaPrivateKey.dat";
    private static final String PUBLIC_KEY_FILE_NAME = "rsaPublicKey.dat";
    private static final String PRIVATE_KEY_FILE_NAME_EC = "ecPrivateKey.dat";
    private static final String PUBLIC_KEY_FILE_NAME_EC = "ecPublicKey.dat";

    private PrivateKey privateKey;
    private PublicKey publicKey;
    private PrivateKey ecPrivateKey;
    private PublicKey ecPublicKey;
    private int moduleLength;
    private byte[] mainKey;

    public RsaCspFactory(String keyPath, String pin, String moduleLength) {
        // 私钥加密密钥为口令的摘要值
        AbstractDigest digest = this.createDigest();
        byte[] pinDigest = digest.digest(pin.getBytes());
        this.mainKey = ByteUtils.duplicate(pinDigest, 32);

        AbstractBlockCipher cipher = this.createBlockCipher();

        String pkFileName = keyPath + PUBLIC_KEY_FILE_NAME;
        String skFileName = keyPath + PRIVATE_KEY_FILE_NAME;
        String ecPkFileName = keyPath + PUBLIC_KEY_FILE_NAME_EC;
        String ecSkFileName = keyPath + PRIVATE_KEY_FILE_NAME_EC;
        this.moduleLength = Integer.parseInt(moduleLength);

        byte[] pkData = FileUtils.readFile(pkFileName);
        byte[] skData = FileUtils.readFile(skFileName);
        byte[] ecPkData = FileUtils.readFile(ecPkFileName);
        byte[] ecSkData = FileUtils.readFile(ecSkFileName);

        if (pkData == null || skData == null) {
            // 没有密钥，创建
            RsaKeyGenerator generator = new RsaKeyGenerator(this.moduleLength);
            var keypair = generator.generateKeyPair();
            pkData = keypair.getPublicKey().getEncoded();
            skData = keypair.getPrivateKey().getEncoded();

            skData = cipher.encrypt(skData);

            FileUtils.writeFile(pkFileName, pkData);
            FileUtils.writeFile(skFileName, skData);
        }

        if (ecPkData == null || ecSkData == null) {
            // 没有密钥，创建
            String name = "P-256";
            ASN1ObjectIdentifier oid = ECNamedCurveTable.getOID(name);
            X9ECParameters ecSpec = ECNamedCurveTable.getByName(name);
            ECNamedDomainParameters domainParameters = new ECNamedDomainParameters(oid, ecSpec.getCurve(), ecSpec.getG(), ecSpec.getN());
            ECKeyGenerationParameters generationParameters = new ECKeyGenerationParameters(domainParameters, new SecureRandom());
            ECKeyPairGenerator keyPairGenerator = new ECKeyPairGenerator();
            keyPairGenerator.init(generationParameters);

            AsymmetricCipherKeyPair keyPair = keyPairGenerator.generateKeyPair();
            ecPkData = new PublicKey(keyPair.getPublic()).getEncoded();
            ecSkData = new PrivateKey(keyPair.getPrivate()).getEncoded();
            ecSkData = cipher.encrypt(ecSkData);

            FileUtils.writeFile(ecPkFileName, ecPkData);
            FileUtils.writeFile(ecSkFileName, ecSkData);
        }

        try {
            skData = cipher.decrypt(skData);
            ecSkData = cipher.decrypt(ecSkData);
        } catch (Exception e) {
            throw new RuntimeException("解密PIN码保护的私钥错误", e);
        }

        this.privateKey = new PrivateKey(skData);
        this.publicKey = new PublicKey(pkData);
        this.ecPrivateKey = new PrivateKey(ecSkData);
        this.ecPublicKey = new PublicKey(ecPkData);

        register();
    }

    @Override
    public RandomGenerator createRandomGenerator() {
        return new SoftRandomGenerator();
    }

    @Override
    public String getDigestAlgOid() {
        return OIWObjectIdentifiers.idSHA1.toString();
    }

    @Override
    public String getDigestAlgName() {
        return "SHA1";
    }

    @Override
    public AbstractDigest createDigest() {
        return new Sha1Digest();
    }

    @Override
    public String getSignerAlgOid() {
        return PKCSObjectIdentifiers.sha1WithRSAEncryption.toString();
    }

    @Override
    public String getAsyncEncryptionAlgOid() {
        return PKCSObjectIdentifiers.rsaEncryption.toString();
    }

    @Override
    public String getSignerAlgName() {
        return "Sha1WithRSA";
    }

    @Override
    public AbstractSigner createSigner() {
        return new Sha1WithRsaSigner(publicKey, privateKey);
    }

    @Override
    public AbstractBlockCipher createBlockCipher() {
        return new AesBlockCipher(this.mainKey);
    }

    @Override
    public AbstractAsymmetricBlockCipher createAsymmetricBlockCipher() {
        return new RsaBlockCipher(this.publicKey, this.privateKey);
    }

    @Override
    public KeyPairGenerator createKeyPairGenerator() {
        return new RsaKeyGenerator(this.moduleLength);
    }

    @Override
    public AbstractAgreement createAgreement() {
        return new ECDHAgreement(this.ecPublicKey, this.ecPrivateKey);
    }

    @Override
    public AbstractMac createMac() {
        return new Sha1HMac();
    }

    @Override
    public String getBlockCipherAlgOid() {
        return NISTObjectIdentifiers.aes.toString();
    }

    @Override
    public String getKeyPairOid() {
        return "1.2.840.113549.1.1.1";
    }

    @Override
    public void close() {
    }
}
