package cn.geminis.crypto.core.util;

import cn.geminis.crypto.core.x509.X509Certificate;
import cn.geminis.crypto.csp.AbstractSigner;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cms.*;

import java.util.List;

/**
 * @author Allen
 */
public class PkcsUtils {

    /**
     * 编码P7B证书链格式
     *
     * @param certs 证书链证书
     * @return P7B格式证书链
     */
    public static byte[] encodeCertChain(List<X509Certificate> certs) {
        CMSSignedDataGenerator generator = new CMSSignedDataGenerator();
        try {
            CMSProcessableByteArray msg = new CMSProcessableByteArray("".getBytes());
            for (X509Certificate cert : certs) {
                generator.addCertificate(new X509CertificateHolder(cert.getBcCertificate()));
            }
            CMSSignedData signedData = generator.generate(msg);
            return signedData.getEncoded();
        } catch (Exception ex) {
            throw new RuntimeException("创建P7B证书链错误", ex);
        }
    }

    public static byte[] generatePkcs7(byte[] data, AbstractSigner signer, X509Certificate cert) {
        var generator = new CMSSignedDataGenerator();
        var builder = new SignerInfoGeneratorBuilder(
                ProviderUtils.getDigestCalculatorProvider(),
                signatureAlgorithm -> signatureAlgorithm
        );

        var content = new CMSProcessableByteArray(data);

        try {
            SignerInfoGenerator signerInfoGenerator = builder
                    .build(signer.createContentSigner(), new X509CertificateHolder(cert.getBcCertificate()));
            generator.addSignerInfoGenerator(signerInfoGenerator);
            CMSSignedData cmsSignedData = generator
                    .generate(content, false);
            return cmsSignedData.toASN1Structure().getEncoded(ASN1Encoding.DER);
        } catch (Exception e) {
            throw new RuntimeException("产生PKCS7格式数据错误", e);
        }
    }

    public static boolean verifyPkcs7(byte[] pkcs7, byte[] data, List<X509Certificate> certs) {
        try {
            CMSSignedData signedData = new CMSSignedData(
                    new CMSProcessableByteArray(data),
                    pkcs7);

            SignerInformation signerInformation = signedData
                    .getSignerInfos()
                    .iterator()
                    .next();

            return signerInformation.verify(
                    ProviderUtils.getSignerInformationVerifier(
                            X509CertificateUtils.findCert(certs,
                                    signerInformation.getSID())));
        } catch (Exception e) {
            throw new RuntimeException("验证PKCS7格式数据错误", e);
        }
    }

}
