package cn.futuai.open.encrypt.spring.cloud.gateway.filter.response;

import cn.futuai.open.encrypt.core.constants.ApiEncryptConstant;
import cn.futuai.open.encrypt.core.exception.ApiEncryptException;
import cn.futuai.open.encrypt.core.util.ApiChecker;
import cn.futuai.open.encrypt.core.util.ApiEncryptUtil;
import cn.futuai.open.encrypt.spring.cloud.gateway.config.property.GatewayApiEncryptProperties;
import cn.futuai.open.encrypt.spring.cloud.gateway.filter.AbstractGatewayFilter;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import java.nio.charset.StandardCharsets;
import javax.annotation.Resource;
import org.reactivestreams.Publisher;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.NettyWriteResponseFilter;
import org.springframework.cloud.gateway.filter.factory.rewrite.ModifyResponseBodyGatewayFilterFactory;
import org.springframework.cloud.gateway.filter.factory.rewrite.RewriteFunction;
import org.springframework.http.MediaType;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

/**
 * 返回加密过滤器
 * @author Jason Kung
 * @date 2023/10/10 17:12
 */
public class GatewayResponseEncryptFilter extends AbstractGatewayFilter {

    @Resource
    private ModifyResponseBodyGatewayFilterFactory encryptFilterFactory;

    @Resource
    private GatewayApiEncryptProperties gatewayApiEncryptProperty;

    @Override
    protected GatewayApiEncryptProperties getGatewayApiEncryptProperty() {
        return gatewayApiEncryptProperty;
    }

    @Override
    protected Mono<Void> doFilterInternal(ServerWebExchange exchange, GatewayFilterChain chain) {
        String requestUri = exchange.getRequest().getURI().getPath();

        // 检查是否在加密接口列表中
        if (ApiChecker.isPass(requestUri, gatewayApiEncryptProperty.getResponseEncrypt().getEnabled(),
                gatewayApiEncryptProperty.getResponseEncrypt().getCheckModel())) {
            return chain.filter(exchange);
        }

        return encryptFilterFactory.apply(
                new ModifyResponseBodyGatewayFilterFactory.Config().setRewriteFunction(byte[].class,
                        byte[].class, new ResponseEncryptRewriter())).filter(exchange, chain);
    }

    @Override
    public int getOrder() {
        return NettyWriteResponseFilter.WRITE_RESPONSE_FILTER_ORDER - 1;
    }

    static class ResponseEncryptRewriter implements RewriteFunction<byte[], byte[]> {

        @Override
        public Publisher<byte[]> apply(ServerWebExchange exchange, byte[] bytes) {
            if (!MediaType.APPLICATION_JSON.equals(exchange.getResponse().getHeaders().getContentType())) {
                return Mono.just(bytes);
            }
            return Mono.just(encrypt(exchange, bytes));
        }

        private byte[] encrypt(ServerWebExchange exchange, byte[] jsonBytes) {
            String aesKey = exchange.getAttribute(ApiEncryptConstant.AES_KEY);
            if (ArrayUtil.isEmpty(jsonBytes) || StrUtil.isBlank(aesKey)) {
                return jsonBytes;
            }

            String encryptResult;
            String requestUri = exchange.getRequest().getURI().getPath();
            String json = new String(jsonBytes);
            try {
                encryptResult = ApiEncryptUtil.aesEncrypt(json, aesKey);
            } catch (Exception e) {
                throw new ApiEncryptException(requestUri, json, aesKey, e);
            }
            return encryptResult.getBytes(StandardCharsets.UTF_8);
        }
    }
}