package cn.funnymap.lgis.vector.parser;

import cn.funnymap.lgis.exception.ShpParserValidationException;
import cn.funnymap.lgis.file.FileType;
import cn.funnymap.lgis.file.FileUtil;
import org.apache.commons.lang3.StringUtils;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Shapefile解析器处理内容
 *
 * @author jiaoxn
 */
class ShpProcessorParameter {
    // 可接受的文件类型：ZIP、SHP
    private static final List<FileType> ACCEPTABLE_FILE_TYPES = Arrays.asList(FileType.ZIP, FileType.SHP);

    private final Path inputFilePath;
    private final boolean isZip;
    private Path decompressionDirPath;
    private List<Path> shpFilePathList;
    private final CoordinateReferenceSystem targetCrs;

    public ShpProcessorParameter(boolean isZip, List<Path> shpFilePathList, Path inputFilePath,
                                 CoordinateReferenceSystem targetCrs) {
        this.isZip = isZip;
        this.shpFilePathList = shpFilePathList;
        this.inputFilePath = inputFilePath;
        this.targetCrs = targetCrs;
    }

    public static ShpProcessorParameter build(ShpParseParameter shpParseParameter) {
        Path inputFilePath = shpParseParameter.getFilePath();
        validateInputFileType(inputFilePath);
        boolean isZip = FileUtil.validateFileType(inputFilePath, FileType.ZIP);
        List<Path> shpFilePathList = isZip ? new ArrayList<>() : Collections.singletonList(inputFilePath);
        return new ShpProcessorParameter(isZip, shpFilePathList, inputFilePath, shpParseParameter.getTargetCrs());
    }
    private static void validateInputFileType(Path inputFilePath) {
        if (!isAcceptableFileType(inputFilePath)) {
            String exceptionMessage = String.format("仅接收%s格式的文件", acceptableFileTypes());
            throw new ShpParserValidationException(exceptionMessage);
        }
    }
    private static boolean isAcceptableFileType(Path path) {
        return FileUtil.validateFileType(path, ACCEPTABLE_FILE_TYPES);
    }
    private static String acceptableFileTypes() {
        List<String> acceptableFileTypeValueList =
                ACCEPTABLE_FILE_TYPES.stream().map(FileType::getValue).collect(Collectors.toList());
        return StringUtils.join(acceptableFileTypeValueList, "、");
    }

    public Path getInputFilePath() {
        return inputFilePath;
    }

    public List<Path> getShpFilePathList() {
        return shpFilePathList;
    }

    public void setShpFilePathList(List<Path> shpFilePathList) {
        this.shpFilePathList = shpFilePathList;
    }

    public boolean isZip() {
        return isZip;
    }

    public CoordinateReferenceSystem getTargetCrs() {
        return targetCrs;
    }

    public Path getDecompressionDirPath() {
        return decompressionDirPath;
    }

    public void setDecompressionDirPath(Path decompressionDirPath) {
        this.decompressionDirPath = decompressionDirPath;
    }
}
