package cn.funnymap.lgis.vector.parser;

import cn.funnymap.lgis.exception.ShpParserRuntimeException;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.geometry.jts.WKTWriter2;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.opengis.feature.simple.SimpleFeature;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Shapefile-WKT字符串解析器
 *
 * @author jiaoxn
 */
public class Shp2WktParser extends ShpParser<List<String>> {
    public Shp2WktParser(ShpParseParameter parseParameter) {
        super(parseParameter);
    }

    @Override
    public List<String> parseShapefile() {
        List<String> wktList = new ArrayList<>();

        try {
            for(Path path : processorParameter.getShpFilePathList()) {
                SimpleFeatureCollection featureCollection = this.readAndTransform(path);
                List<Geometry> geometryList = this.readGeometry(featureCollection);
                String wkt = this.geometry2Wkt(geometryList);
                wktList.add(wkt);
            }
        } catch (IOException ioException) {
            throw new ShpParserRuntimeException("Shapefile解析失败，" + ioException.getCause());
        } finally {
            this.deleteRelatedFile();
        }

        return wktList;
    }
    private List<Geometry> readGeometry(SimpleFeatureCollection simpleFeatureCollection) {
        return Arrays.stream(simpleFeatureCollection.toArray())
                .map(item -> (Geometry) ((SimpleFeature) item).getDefaultGeometry())
                .collect(Collectors.toList());
    }
    private String geometry2Wkt(List<Geometry> geometryList) {
        GeometryFactory geometryFactory = new GeometryFactory();
        GeometryCollection geometryCollection =
                geometryFactory.createGeometryCollection(geometryList.toArray(new Geometry[0]));
        WKTWriter2 wktWriter2 = new WKTWriter2();
        String geometryCollectionWkt = wktWriter2.write(geometryCollection);
        return extractWktContent(geometryCollectionWkt);
    }
    private String extractWktContent(String geometryCollection) {
        // 定义正则表达式以匹配 GEOMETRYCOLLECTION 括号之间的内容
        Pattern pattern = Pattern.compile("GEOMETRYCOLLECTION \\((.*)\\)");
        Matcher matcher = pattern.matcher(geometryCollection);

        if (matcher.find()) {
            return matcher.group(1);
        }

        return null;
    }
}
