package cn.funnymap.lgis.vector.converter.wkt2shp;

import cn.funnymap.lgis.vector.VectorConverterConsumer;
import cn.funnymap.lgis.vector.converter.wkt2shp.exception.Wkt2ShapefileConverterException;
import cn.funnymap.lgis.vector.converter.wkt2shp.exception.Wkt2ShapefileParameterValidationException;
import org.geotools.data.DataUtilities;
import org.geotools.data.Transaction;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.shapefile.ShapefileDataStoreFactory;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureSource;
import org.geotools.data.simple.SimpleFeatureStore;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.*;
import java.util.stream.Collectors;

/**
 * WKT字符串转Shapefile文件
 *
 * @author jiaoxn
 */
public abstract class WKT2ShapefileConverter
        implements VectorConverterConsumer<Path, CoordinateReferenceSystem, ConverterParam> {
    @Override
    public final void convert(Path targetFilePath, CoordinateReferenceSystem crs,
                              List<ConverterParam> converterParamList) {
        WKTContent wktContent = this.contentFromConverterParams(crs, converterParamList);
        this.validateWktContent(wktContent);
        this.saveToFile(wktContent, targetFilePath);
    }

    protected abstract Class<?> geometryType();

    private WKTContent contentFromConverterParams(CoordinateReferenceSystem crs,
                                                  List<ConverterParam> converterParamList) {
        List<String> wktList = converterParamList.stream().map(ConverterParam::getWkt).collect(Collectors.toList());
        SortedMap<String, Class<?>> attrDef = this.attrDefFromValue(converterParamList.get(0).getAttrValueList());
        List<SortedMap<String, Object>> values =
                converterParamList.stream().map(ConverterParam::getAttrValueList).collect(Collectors.toList());

        return new WKTContent(wktList, attrDef, values, crs);
    }
    private SortedMap<String, Class<?>> attrDefFromValue(SortedMap<String, Object> valueMap) {
        SortedMap<String, Class<?>> attrDef = new TreeMap<>();
        valueMap.forEach((attrName, value) -> attrDef.put(attrName, value.getClass()));
        return attrDef;
    }

    private void validateWktContent(WKTContent wktContent) {
        try {
            this.validateWktAndAttrValuesSize(wktContent);
            List<Geometry> geometryList = this.isStandardWktFormat(wktContent.getWktList());
            this.validateUnifiedGeometryType(geometryList);
            this.validateAttr(wktContent);
            wktContent.setGeometryList(geometryList);
        } catch (ParseException exception) {
            throw new Wkt2ShapefileParameterValidationException("WKT格式校验失败，请检查输入参数");
        }
    }
    private void validateWktAndAttrValuesSize(WKTContent wktContent) {
        if (wktContent.getWktList().size() != wktContent.getValueList().size()) {
            throw new Wkt2ShapefileParameterValidationException("WKT字符串个数与属性值的个数不一致");
        }
    }
    private List<Geometry> isStandardWktFormat(List<String> wktList) throws ParseException {
        List<Geometry> geometryList = new ArrayList<>();

        for (String wkt : wktList) {
            Geometry geometry = this.wkt2geometry(wkt);
            geometryList.add(geometry);
        }

        return geometryList;
    }
    private Geometry wkt2geometry(String wkt) throws ParseException {
        GeometryFactory geometryFactory = JTSFactoryFinder.getGeometryFactory();
        WKTReader reader = new WKTReader(geometryFactory);
        return reader.read(wkt);
    }
    private void validateUnifiedGeometryType(List<Geometry> geometryList) {
        for (Geometry geometry : geometryList) {
            if (!Objects.equals(geometry.getClass(), this.geometryType())) {
                throw new Wkt2ShapefileParameterValidationException("转换参数中包含多种几何类型");
            }
        }
    }
    private void validateAttr(WKTContent wktContent) {
        int attrNumber = wktContent.getAttribute().keySet().size();

        for (SortedMap<String, Object> valueMap : wktContent.getValueList()) {
            if (valueMap.keySet().size() != attrNumber) {
                throw new Wkt2ShapefileParameterValidationException("输入数据值的长度和属性的长度不一致");
            }

            valueMap.keySet()
                    .stream()
                    .filter(attrName -> valueMap.get(attrName).getClass() != wktContent.getAttribute().get(attrName))
                    .forEachOrdered(attrName -> {
                        throw new Wkt2ShapefileParameterValidationException(String.format("%s属性值类型不一致", attrName));
            });
        }
    }

    private void saveToFile(WKTContent wktContent, Path targetFilePath) {
        try {
            // 定义Feature Type
            SimpleFeatureType simpleFeatureType = this.createFeatureType(wktContent.getCrs(), wktContent.getAttribute());

            // 创建数据存储
            ShapefileDataStore shapefileDataStore = this.createDataStore(targetFilePath, simpleFeatureType);

            // 数据集合
            SimpleFeatureCollection simpleFeatureCollection = this.createFeatureCollection(wktContent, simpleFeatureType);

            // 数据写入到Shapefile文件
            this.writeData(shapefileDataStore, simpleFeatureCollection);
        } catch (IOException exception) {
            throw new Wkt2ShapefileConverterException("WKT转Shapefile文件报错了，" + exception.getMessage());
        }
    }

    private SimpleFeatureType createFeatureType(CoordinateReferenceSystem crs, SortedMap<String, Class<?>> attr) {
        SimpleFeatureTypeBuilder simpleFeatureTypeBuilder = new SimpleFeatureTypeBuilder();
        simpleFeatureTypeBuilder.setCRS(crs);
        simpleFeatureTypeBuilder.setName("shapefile");
        simpleFeatureTypeBuilder.add("the_geom", this.geometryType());

        // 定义属性
        attr.forEach(simpleFeatureTypeBuilder::add);

        return simpleFeatureTypeBuilder.buildFeatureType();
    }

    private ShapefileDataStore createDataStore(Path path, SimpleFeatureType simpleFeatureType) throws IOException {
        Map<String, Serializable> params = new HashMap<>();
        params.put(ShapefileDataStoreFactory.URLP.key, path.toUri().toURL());
        ShapefileDataStoreFactory dataStoreFactory = new ShapefileDataStoreFactory();

        ShapefileDataStore shapefileDataStore = (ShapefileDataStore) dataStoreFactory.createNewDataStore(params);
        shapefileDataStore.createSchema(simpleFeatureType);
        shapefileDataStore.setCharset(StandardCharsets.UTF_8);
        return shapefileDataStore;
    }

    private SimpleFeatureCollection createFeatureCollection(WKTContent wktContent, SimpleFeatureType featureType) {
        List<SimpleFeature> simpleFeatureList = new ArrayList<>();

        SimpleFeatureBuilder simpleFeatureBuilder = new SimpleFeatureBuilder(featureType);

        for (int i = 0; i < wktContent.getGeometryList().size(); i++) {
            simpleFeatureBuilder.add(wktContent.getGeometryList().get(i));
            for (Object attrValue : wktContent.getValueList().get(i).values()) {
                simpleFeatureBuilder.add(attrValue);
            }
            SimpleFeature simpleFeature = simpleFeatureBuilder.buildFeature(null);
            simpleFeatureList.add(simpleFeature);
        }

        return DataUtilities.collection(simpleFeatureList);
    }


    private void writeData(ShapefileDataStore shapefileDataStore, SimpleFeatureCollection simpleFeatureCollection) throws IOException {
        String typeName = shapefileDataStore.getTypeNames()[0];

        SimpleFeatureSource simpleFeatureSource = shapefileDataStore.getFeatureSource(typeName);

        if (simpleFeatureSource instanceof SimpleFeatureStore) {
            SimpleFeatureStore simpleFeatureStore = (SimpleFeatureStore) simpleFeatureSource;

            try (Transaction transaction = Transaction.AUTO_COMMIT) {
                simpleFeatureStore.setTransaction(transaction);

                try {
                    simpleFeatureStore.addFeatures(simpleFeatureCollection);
                    transaction.commit();
                } catch (Exception exception) {
                    transaction.rollback();
                }
            }
        }
    }
}
