package cn.funnymap.lgis.vector.converter.shp2geojson;

import cn.funnymap.lgis.file.FileUtil;
import cn.funnymap.lgis.vector.VectorConverterFunction;
import org.geotools.data.FileDataStore;
import org.geotools.data.FileDataStoreFinder;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.geojson.feature.FeatureJSON;

import java.io.IOException;
import java.io.StringWriter;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Shapefile转Geojson
 *
 * @author jiaoxn
 */
public class Shapefile2Geojson implements VectorConverterFunction<Path, List<String>> {
    /**
     * 读取Shapefile文件中的信息，将其转为GeoJSON，并以字符串的格式返回
     *
     * @param path .shp文件或者对应的ZIP文件夹
     */
    @Override
    public List<String> converter(Path path) {
        try {
            if (isZipFile(path)) {
                return this.converterZipFile2Geojson(path);
            } else if (isShpFile(path)) {
                return this.converterShpFile2Geojson(Collections.singletonList(path));
            }
        } catch (IOException exception) {
            throw new RuntimeException(exception);
        }

        return Collections.emptyList();
    }
    private boolean isZipFile(Path path) {
        return ".zip".equalsIgnoreCase(FileUtil.getFileExtension(path));
    }
    private List<String> converterZipFile2Geojson(Path zipFilePath) throws IOException {
        String dataName = FileUtil.getFileNameWithoutExtension(zipFilePath);

        if (dataName == null) {
            return Collections.emptyList();
        }

        Path tempZipDir = FileUtil.extractZipFileToTemp(zipFilePath);
        List<Path> shpFilePathList = FileUtil.getFileByFileExtension(tempZipDir, ".shp");

        if (shpFilePathList.isEmpty()) {
            return Collections.emptyList();
        }

        return this.converterShpFile2Geojson(shpFilePathList);
    }
    private boolean isShpFile(Path path) {
        return ".shp".equalsIgnoreCase(FileUtil.getFileExtension(path));
    }
    private List<String> converterShpFile2Geojson(List<Path> shpFilePathList) {
        List<String> geojsonList = new ArrayList<>();

        try {
            for (Path path : shpFilePathList) {
                // 读取Shapefile
                FileDataStore fileDataStore = FileDataStoreFinder.getDataStore(path.toFile());
                SimpleFeatureCollection featureCollection = fileDataStore.getFeatureSource().getFeatures();

                // 创建FeatureJSON对象
                StringWriter stringWriter = new StringWriter();
                FeatureJSON featureJson = new FeatureJSON();
                featureJson.writeFeatureCollection(featureCollection, stringWriter);

                // 转成JSON字符串
                String geojson = stringWriter.toString();
                geojsonList.add(geojson);
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return geojsonList;
    }
}
