package cn.funnymap.lgis.vector;

import cn.funnymap.lgis.util.WKTUtil;
import cn.funnymap.lgis.vector.excption.IncorrectFileFormatException;
import cn.funnymap.lgis.vector.excption.ReadShapefileAttributeException;
import cn.funnymap.lgis.vector.params.ShapefileAttributeFieldDef;
import org.geotools.data.DataStore;
import org.geotools.data.DataStoreFinder;
import org.geotools.data.FeatureSource;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.feature.FeatureCollection;
import org.locationtech.jts.geom.Geometry;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;

import java.io.IOException;
import java.nio.file.Path;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 获取Shapefile的属性信息
 *
 * @author jiaoxn
 */
public class ShapefileAttribute {
    private ShapefileAttribute() {}

    public static List<ShapefileAttributeFieldDef> attributeFieldDefFrom(ShapefileDataStore shapefileDataStore,
                                                                         boolean whetherExcludeSpatialField) throws IOException {
        FeatureSource<SimpleFeatureType, SimpleFeature> featureSource = shapefileDataStore.getFeatureSource();
        SimpleFeatureType simpleFeatureType = featureSource.getSchema();
        List<AttributeDescriptor> attributeDescriptorList = simpleFeatureType.getAttributeDescriptors();
        return attributeDescriptorList.stream()
                .filter(item -> whetherExcludeSpatialField && !"the_geom".equalsIgnoreCase(item.getName().toString()))
                .map(ShapefileAttribute::attributeDescriptor2Def)
                .collect(Collectors.toList());
    }
    private static ShapefileAttributeFieldDef attributeDescriptor2Def(AttributeDescriptor attributeDescriptor) {
        String fieldName = attributeDescriptor.getName().toString();
        String fieldType = attributeDescriptor.getType().getName().toString();
        return new ShapefileAttributeFieldDef(fieldName, fieldName, fieldType, null);
    }

    public static List<Map<String, Object>> fromShpFile(Path shpFilePath) {
        // 校验输入文件的格式是否为 .shp 文件
        validateFileFormat(shpFilePath);

        return getShpAttribute(shpFilePath);
    }
    private static void validateFileFormat(Path shpFilePath) {
        if (!shpFilePath.getFileName().toString().endsWith(".shp")) {
            throw new IncorrectFileFormatException("请选择.shp文件");
        }
    }
    private static List<Map<String, Object>> getShpAttribute(Path shpFilePath) {
        List<Map<String, Object>> attrList = new ArrayList<>();

        try {
            Map<String, Object> params = new HashMap<>();
            params.put("url", shpFilePath.toFile().toURI().toURL());

            // 使用DataStoreFinder来创建DataStore
            DataStore dataStore = DataStoreFinder.getDataStore(params);

            // 获取FeatureSource，它表示地理数据的一个集合
            String typeName = dataStore.getTypeNames()[0];
            FeatureSource<SimpleFeatureType, SimpleFeature> featureSource = dataStore.getFeatureSource(typeName);

            // 如果没有过滤器，则获取所有要素
            FeatureCollection<SimpleFeatureType, SimpleFeature> featureCollection = featureSource.getFeatures();

            // 使用迭代器遍历所有要素
            try (SimpleFeatureIterator iterator = ((SimpleFeatureCollection) featureCollection).features()) {
                while (iterator.hasNext()) {
                    SimpleFeature feature = iterator.next();
                    SimpleFeatureType featureType = feature.getFeatureType();

                    Map<String, Object> featureAttr = new LinkedHashMap<>();

                    // 遍历每个属性
                    for (int i = 0; i < featureType.getAttributeCount(); i++) {
                        String attributeName = featureType.getDescriptor(i).getLocalName();
                        Object attribute = feature.getAttribute(i);

                        if ("the_geom".equals(attributeName)) {
                            attribute = WKTUtil.geometry2wkt((Geometry) attribute);
                        }

                        featureAttr.put(attributeName, attribute);
                    }

                    attrList.add(featureAttr);
                }
            }

            dataStore.dispose();
        } catch (IOException e) {
            throw new ReadShapefileAttributeException(e.getMessage());
        }

        return attrList;
    }
}
