package cn.funnymap.lgis.vector;

import cn.funnymap.lgis.crs.CRSUtil;
import cn.funnymap.lgis.exception.ShapefileParseException;
import cn.funnymap.lgis.exception.ShapefileValidationException;
import cn.funnymap.lgis.file.FileType;
import cn.funnymap.lgis.file.FileUtil;
import cn.funnymap.lgis.vector.params.GeojsonResponse;
import cn.funnymap.lgis.vector.params.ShapefileAttributeFieldDef;
import com.alibaba.fastjson2.JSONObject;
import lombok.SneakyThrows;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.feature.DefaultFeatureCollection;
import org.geotools.geojson.feature.FeatureJSON;
import org.locationtech.jts.geom.Geometry;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import java.io.IOException;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * Shapefile操作类
 *
 * @author jiaoxn
 */
public class Shapefile extends VectorFile {
    private Path zipExtractDir;
    private List<Path> shpFileList;
    private final boolean isMultiShp;

    private static final List<FileType> OTHER_REQUIRED_FILE_TYPES =
            Arrays.asList(FileType.DBF, FileType.SHX, FileType.PRJ);

    public Shapefile(boolean isMultiShp) {
        this.isMultiShp = isMultiShp;
    }

    @Override
    protected void executeValidityVerification(Path path) {
        String fileExtension = FileUtil.getFileExtension(path);

        if (Objects.equals(fileExtension, FileType.ZIP.getValue())) {
            this.executeValidityVerificationWhenFileIsZip(path);
        } else if (Objects.equals(fileExtension, FileType.SHP.getValue())) {
            this.executeValidityVerificationWhenFileIsShp(path);
        } else {
            throw new ShapefileValidationException("请输入有效的Shapefile压缩包或者.shp文件");
        }
    }
    private void executeValidityVerificationWhenFileIsZip(Path path) {
        try {
            // 解压ZIP文件到临时文件夹
            this.zipExtractDir = FileUtil.extractZipFileToTemp(path);
            // 从解压的文件夹中嵌套读取.shp文件
            this.shpFileList = FileUtil.getFileByFileExtension(zipExtractDir, FileType.SHP.getValue());

            // 仅支持一个SHP文件
            if (!this.isMultiShp && shpFileList.size() > 1) {
                throw new ShapefileValidationException("Shapefile压缩包格式校验失败，请确保该文件仅包含一个.shp文件");
            }

            // 遍历.shp文件，在同一个父级文件夹中查找其他所需的文件，如果没有直接抛出异常
            for (Path shpPath : shpFileList) {
                this.checkOtherRequiredFiles(shpPath);
            }
        } catch (IOException ioException) {
            throw new ShapefileValidationException("Shapefile压缩包格式校验失败，" + ioException);
        }
    }
    private void executeValidityVerificationWhenFileIsShp(Path path) {
        try {
            this.checkOtherRequiredFiles(path);
        } catch (IOException ioException) {
            throw new ShapefileValidationException("Shapefile格式校验失败，" + ioException);
        }
    }
    private void checkOtherRequiredFiles(Path shpPath) throws IOException {
        // 获取父级文件夹
        Path parentDir = shpPath.getParent();
        String shpFilename = FileUtil.getFileNameWithoutExtension(shpPath);

        for (FileType requiredFileType : OTHER_REQUIRED_FILE_TYPES) {
            // 判断在父级文件中，是否存在同名的其他所需的类型文件
            String requiredFileName = String.format("%s%s", shpFilename, requiredFileType.getValue());
            List<Path> requiredFilePathList = FileUtil.getFileByFileName(parentDir, requiredFileName, false);

            if (requiredFilePathList.isEmpty()) {
                throw new ShapefileValidationException(String.format("Shapefile格式校验失败，%s文件不存在", requiredFileName));
            }
        }
    }

    @Override
    public String parseAsWktExecutor(CoordinateReferenceSystem targetCrs) {
        return "";
    }

    @Override
    public List<GeojsonResponse> parseAsGeojsonExecutor(CoordinateReferenceSystem targetCrs) {
        List<GeojsonResponse> geojsonResponseList = new ArrayList<>();

        try {
            for (Path shpFilePath : this.shpFileList) {
                ShapefileDataStore shapefileDataStore = this.createDataStore(shpFilePath);

                // 读取内容为GeoJSON
                SimpleFeatureCollection simpleFeatureCollection = this.readAndTransformData(shapefileDataStore, targetCrs);
                String geojson = this.simpleFeatureCollection2Json(simpleFeatureCollection);

                // 读取属性列的定义
                List<ShapefileAttributeFieldDef> shapefileAttributeFiledDef =
                        ShapefileAttribute.attributeFieldDefFrom(shapefileDataStore, true);

                GeojsonResponse geojsonResponse = new GeojsonResponse(JSONObject.parse(geojson),
                        shapefileAttributeFiledDef);
                geojsonResponseList.add(geojsonResponse);
            }

            return geojsonResponseList;
        } catch (IOException exception) {
            throw new ShapefileParseException("Shapefile压缩表解析失败");
        }
    }
    private ShapefileDataStore createDataStore(Path shpPath) throws MalformedURLException {
        ShapefileDataStore shapefileDataStore = new ShapefileDataStore(shpPath.toUri().toURL());
        // 设置编码【防止中文乱码】
        shapefileDataStore.setCharset(StandardCharsets.UTF_8);
        return shapefileDataStore;
    }
    private SimpleFeatureCollection readAndTransformData(ShapefileDataStore shapefileDataStore,
                                                         CoordinateReferenceSystem targetCrs) throws IOException {
        DefaultFeatureCollection featureCollection = new DefaultFeatureCollection();
        try (SimpleFeatureIterator iterator = shapefileDataStore.getFeatureSource().getFeatures().features()) {
            while (iterator.hasNext()) {
                SimpleFeature simpleFeature = iterator.next();
                Geometry geometry = this.transform(simpleFeature, targetCrs);
                simpleFeature.setDefaultGeometry(geometry);
                featureCollection.add(simpleFeature);
            }
        }

        return featureCollection;
    }
    private Geometry transform(SimpleFeature simpleFeature, CoordinateReferenceSystem targetCrs) {
        return CRSUtil.transform(
                (Geometry) simpleFeature.getDefaultGeometry(),
                simpleFeature.getType().getCoordinateReferenceSystem(),
                targetCrs);
    }
    private String simpleFeatureCollection2Json(SimpleFeatureCollection simpleFeatureCollection) throws IOException {
        StringWriter stringWriter = new StringWriter();
        FeatureJSON featureJson = new FeatureJSON();
        featureJson.writeFeatureCollection(simpleFeatureCollection, stringWriter);
        return stringWriter.toString();
    }

    @Override
    @SneakyThrows
    protected void deleteTempDir() {
        if (this.zipExtractDir != null) {
            FileUtil.delete(this.zipExtractDir);
        }
    }
}
