/*
 * Decompiled with CFR 0.152.
 */
package cn.funnymap.lgis.file;

import cn.funnymap.lgis.file.FileType;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.web.multipart.MultipartFile;

public class FileUtil {
    private FileUtil() {
    }

    public static long getDirectorySize(Path directory) throws IOException {
        long size = 0L;
        try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory);){
            for (Path file : directoryStream) {
                if (Files.isDirectory(file, new LinkOption[0])) {
                    size += FileUtil.getDirectorySize(file);
                    continue;
                }
                size += Files.size(file);
            }
        }
        return size;
    }

    public static String readableFileSize(long size) {
        if (size <= 0L) {
            return "0";
        }
        String[] units = new String[]{"B", "KB", "MB", "GB", "TB"};
        int digitGroups = (int)(Math.log10(size) / Math.log10(1024.0));
        return new DecimalFormat("#,###.##").format((double)size / Math.pow(1024.0, digitGroups)) + " " + units[digitGroups];
    }

    public static List<Path> getFileByFileExtension(Path targetDir, String ... extensions) throws IOException {
        ArrayList<Path> response = new ArrayList<Path>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(targetDir, entry -> {
            if (Files.isDirectory(entry, new LinkOption[0])) {
                response.addAll(FileUtil.getFileByFileExtension(entry, extensions));
            } else {
                String fileName = entry.getFileName().toString();
                for (String extension : extensions) {
                    if (!fileName.endsWith(extension)) continue;
                    return true;
                }
            }
            return false;
        });){
            for (Path file : stream) {
                response.add(file);
            }
        }
        return response;
    }

    public static List<Path> getFileByFileName(Path targetDir, final String fileName, boolean isNested) throws IOException {
        final ArrayList<Path> foundFiles = new ArrayList<Path>();
        if (!Files.isDirectory(targetDir, new LinkOption[0])) {
            return foundFiles;
        }
        if (isNested) {
            Files.walkFileTree(targetDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @NotNull
                public FileVisitResult visitFile(Path file, @Nullable BasicFileAttributes attrs) {
                    if (FileUtil.isTheSameFilename(file, fileName)) {
                        foundFiles.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(targetDir);){
                for (Path file : stream) {
                    if (!Files.isRegularFile(file, new LinkOption[0]) || !FileUtil.isTheSameFilename(file, fileName)) continue;
                    foundFiles.add(file);
                }
            }
        }
        return foundFiles;
    }

    public static String getFileExtension(Path path) {
        if (Files.isDirectory(path, new LinkOption[0])) {
            return null;
        }
        String fileName = path.getFileName().toString();
        int dotIndex = fileName.lastIndexOf(46);
        if (dotIndex != -1 && dotIndex < fileName.length() - 1) {
            return fileName.substring(dotIndex);
        }
        return null;
    }

    public static String getFileExtension(String fileName) {
        int dotIndex = fileName.lastIndexOf(46);
        if (dotIndex != -1 && dotIndex < fileName.length() - 1) {
            return fileName.substring(dotIndex);
        }
        return null;
    }

    public static String getFileNameWithoutExtension(Path path) {
        if (Files.isDirectory(path, new LinkOption[0])) {
            return null;
        }
        String fileName = path.getFileName().toString();
        int dotIndex = fileName.lastIndexOf(46);
        if (dotIndex != -1) {
            return fileName.substring(0, dotIndex);
        }
        return fileName;
    }

    public static Path extractZipFileToTemp(Path zipFilePath) throws IOException {
        if (FileUtil.validateFileType(zipFilePath, FileType.ZIP)) {
            String fileName = FileUtil.getFileNameWithoutExtension(zipFilePath);
            Path tempDir = Files.createTempDirectory(fileName, new FileAttribute[0]);
            FileUtil.extractZipFile(zipFilePath, tempDir);
            return tempDir;
        }
        return null;
    }

    public static void extractZipFile(Path zipFilePath, Path targetDir) throws IOException {
        if (!FileUtil.validateFileType(zipFilePath, FileType.ZIP)) {
            return;
        }
        try (ZipInputStream zipInputStream = new ZipInputStream(Files.newInputStream(zipFilePath, new OpenOption[0]));){
            ZipEntry zipEntry = zipInputStream.getNextEntry();
            while (zipEntry != null) {
                Path filePath = Paths.get(targetDir.toAbsolutePath().toString(), zipEntry.getName());
                if (zipEntry.isDirectory()) {
                    Files.createDirectories(filePath, new FileAttribute[0]);
                } else {
                    Files.createDirectories(filePath.getParent(), new FileAttribute[0]);
                    Files.copy(zipInputStream, filePath, new CopyOption[0]);
                }
                zipInputStream.closeEntry();
                zipEntry = zipInputStream.getNextEntry();
            }
        }
    }

    public static Path saveMultipartFileToTemp(MultipartFile multipartFile) throws IOException {
        if (multipartFile.isEmpty()) {
            return null;
        }
        String originalFilename = multipartFile.getOriginalFilename();
        if (originalFilename == null) {
            return null;
        }
        String fileSuffix = FileUtil.getFileExtension(originalFilename);
        if (fileSuffix == null) {
            return null;
        }
        String newFileName = UUID.randomUUID().toString();
        Path newFile = Files.createTempFile(newFileName, fileSuffix, new FileAttribute[0]);
        try {
            multipartFile.transferTo(newFile);
            return newFile;
        }
        catch (IOException ioException) {
            throw new RuntimeException("\u5185\u5b58\u4e2d\u521b\u5efaZIP\u6587\u4ef6\u51fa\u9519");
        }
    }

    public static boolean validateFileType(String fileName, FileType targetFileType) {
        String fileExtension = FileUtil.getFileExtension(fileName);
        return targetFileType.getValue().equalsIgnoreCase(fileExtension);
    }

    public static boolean validateFileType(Path path, FileType targetFileType) {
        String fileExtension = FileUtil.getFileExtension(path);
        return targetFileType.getValue().equalsIgnoreCase(fileExtension);
    }

    public static void delete(Path path) throws IOException {
        if (Files.isDirectory(path, new LinkOption[0])) {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @NotNull
                public FileVisitResult visitFile(Path file, @Nullable BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                @NotNull
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                @NotNull
                public FileVisitResult visitFileFailed(Path file, @NotNull IOException exc) throws IOException {
                    throw exc;
                }
            });
        } else {
            Files.delete(path);
        }
    }

    public static List<Path> getFileByFuzzyMatchingFilename(Path targetDir, final String fileName, boolean isNested) throws IOException {
        final ArrayList<Path> foundFiles = new ArrayList<Path>();
        if (!Files.isDirectory(targetDir, new LinkOption[0])) {
            return foundFiles;
        }
        if (isNested) {
            Files.walkFileTree(targetDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                @NotNull
                public FileVisitResult visitFile(Path file, @Nullable BasicFileAttributes attrs) {
                    if (FileUtil.isFilenameFuzzySame(file, fileName)) {
                        foundFiles.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(targetDir);){
                for (Path file : stream) {
                    if (!Files.isRegularFile(file, new LinkOption[0]) || !FileUtil.isFilenameFuzzySame(file, fileName)) continue;
                    foundFiles.add(file);
                }
            }
        }
        return foundFiles;
    }

    private static boolean isTheSameFilename(@NotNull Path path, @NotNull String targetName) {
        return Objects.equals(path.getFileName().toString(), targetName);
    }

    private static boolean isFilenameFuzzySame(@NotNull Path path, @NotNull String targetName) {
        return path.getFileName().toString().contains(targetName);
    }
}

