package cn.funnymap.lgis.response;

import cn.funnymap.lgis.response.exception.AbstractException;
import cn.funnymap.lgis.response.exception.common.BaseException;
import cn.funnymap.lgis.response.exception.common.ParamValidationException;
import cn.funnymap.lgis.response.status.code.BaseStatusEnum;
import cn.funnymap.lgis.response.status.code.ValidationExceptionStatusCodeEnum;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 实现ResponseBodyAdvice接口，统一异常拦截及统一响应
 *
 * @author jiaoxn
 */
@RestControllerAdvice(annotations = RestController.class)
@Slf4j
public class GlobalResponseAdvice implements ResponseBodyAdvice<Object> {
    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        return true;
    }

    @Override
    public Object beforeBodyWrite(Object body,
                                  MethodParameter returnType,
                                  MediaType selectedContentType,
                                  Class<? extends HttpMessageConverter<?>> selectedConverterType,
                                  ServerHttpRequest request,
                                  ServerHttpResponse response) {
        // 如果响应是流数据（对应下载的情况），直接返回
        if (body instanceof InputStreamResource) {
            return body;
        }

        // 如果响应是二进制数据（对应下载的情况），直接返回
        if (body instanceof byte[]) {
            return body;
        }

        // 如果响应体为空，直接响应成功
        if (body == null) {
            return ResponseStructure.success(null);
        }

        ResponseStructure<Object> responseStructure;

        // 如果是 POST 请求，业务状态码统一设置为 201
        if ("POST".equals(((ServletServerHttpRequest) request).getServletRequest().getMethod())) {
            responseStructure = ResponseStructure.created();
        } else {
            responseStructure = ResponseStructure.success();
        }

        responseStructure.setData(body);

        // 如果返回值是字符串类型，则用其替换 message
        if (body instanceof String) {
            return JSONObject.toJSONString(responseStructure);
        } else {
            responseStructure.setData(body);
        }

        return responseStructure;
    }

    /**
     * 拦截自定义异常
     */
    @ExceptionHandler({AbstractException.class})
    public ResponseEntity<ResponseStructure<String>> handleCustomException(AbstractException customException) {
        this.writeExceptionToLog(customException);

        ResponseStructure<String> responseStructure = ResponseStructure.fail(customException);
        return new ResponseEntity<>(responseStructure, customException.getHttpStatus());
    }

    /**
     * 拦截参数校验异常
     */
    @ExceptionHandler({ConstraintViolationException.class})
    public ResponseEntity<ResponseStructure<String>> handleCustomException(ConstraintViolationException exception) {
        List<String> msgList = exception.getConstraintViolations()
                .stream()
                .map(ConstraintViolation::getMessage)
                .collect(Collectors.toList());
        ParamValidationException paramValidationException =
                new ParamValidationException(ValidationExceptionStatusCodeEnum.ERR_WRONG_MULTIPART_FILE_TYPE,
                        String.join("；", msgList));

        this.writeExceptionToLog(paramValidationException);

        ResponseStructure<String> responseStructure = ResponseStructure.fail(paramValidationException);
        return new ResponseEntity<>(responseStructure, paramValidationException.getHttpStatus());
    }

    /**
     * 顶级异常捕获，当其他异常无法处理时选择使用
     */
    @ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler({Exception.class})
    public ResponseEntity<ResponseStructure<String>> handle(Exception exception) {
        BaseException baseException = new BaseException(BaseStatusEnum.ERR_UNKNOWN, exception.getMessage());

        this.writeExceptionToLog(baseException);

        ResponseStructure<String> responseStructure = ResponseStructure.fail(baseException);
        return new ResponseEntity<>(responseStructure, baseException.getHttpStatus());
    }

    private void writeExceptionToLog(AbstractException exception) {
        log.error(String.format("错误代码：%d，错误描述信息：%s", exception.getCode(), exception.getMessage()), exception);
    }
}
