package cn.fscode.commons.web.handler;

import cn.fscode.commons.tool.core.exception.BizException;
import com.alibaba.fastjson2.JSON;
import cn.fscode.commons.core.response.Resp;
import cn.fscode.commons.tool.core.exception.BaseException;
import org.hibernate.validator.internal.engine.path.PathImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 全局异常处理,throw new BusinessException(ResultEnum.FAILURE);要在控制层中使用，如果在服务层使用，不会
 * 走BusinessException方法处理，只会走Exception方法进行处理
 * 以下来自stackoverflow
 * @author shenguangyang
 */
@ControllerAdvice
@RestController
public class GlobalExceptionHandler {
    private static final Logger log = LoggerFactory.getLogger(GlobalExceptionHandler.class);
    @PostConstruct
    public void init() {
        log.info("init {}", this.getClass().getName());
    }
    /**
     * 请求资源不存在异常
     * @param req 请求
     * @param e 异常
     * @return 返回结果
     */
    @ExceptionHandler(value = {NoHandlerFoundException.class})
    public Object noHandlerFoundException(HttpServletRequest req, Exception e) {
        log.error("exception: {}, uri: {}", e.getMessage(), req.getRequestURI());
        return Resp.fail(HttpStatus.NOT_FOUND.value(), "Requested resource does not exist").wrap();
    }
    /**
     * 当调用接口时候如果没有传入某个参数就会报出当前异常
     */
    @ExceptionHandler(value = {MissingServletRequestParameterException.class})
    public Object missingServletRequestParameterException(HttpServletRequest req, MissingServletRequestParameterException e) {
        log.error("exception: {}, uri: {}", e.getMessage(), req.getRequestURI());
        return Resp.fail(e.getParameterName() + ": " + e.getMessage()).wrap();
    }

    /**
     * 方法参数类型不匹配
     */
    @ExceptionHandler(value = {MethodArgumentTypeMismatchException.class})
    public Object methodArgumentTypeMismatchException(HttpServletRequest req, Exception e) {
        log.error("exception: {}, uri: {}", e.getMessage(), req.getRequestURI());
        return Resp.fail("Method parameter types do not match").wrap();
    }


    /**
     * 参数解析失败
     */
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public Object httpMessageNotReadableException(HttpServletRequest request, Exception e) {
        log.error("exception: {}, uri: {}", e.getMessage(), request.getRequestURI());
        // 可能是请求体不是json格式
        return Resp.fail("Message not readable").wrap();
    }

    /**
     * 处理实体字段校验不通过异常
     * <p>ConstraintViolationException: 普通参数(非 java bean)校验出错时抛出 把校验注解写在参数上</p>
     * <p>MethodArgumentNotValidException：json请求体绑定到java bean上失败时抛出(参数验证失败)</p>
     * <p>BindException：表单提交请求参数绑定到java bean上失败时抛出 这种异常不能在参数对象上加@RequestBody (参数绑定失败)</p>
     * <p>@apiNote MethodArgumentNotValidException 继承 BindException</p>
     */
    @ExceptionHandler({BindException.class, MethodArgumentNotValidException.class})
    public Object bindException(Exception e, HttpServletRequest request) {
        StringBuilder errorMessageSb = new StringBuilder();
        // 错误信息map
        Map<String, String> errorMap = new HashMap<>(16);
        List<ObjectError> allErrors = e instanceof MethodArgumentNotValidException
                ? ((MethodArgumentNotValidException) e).getBindingResult().getAllErrors()
                : ((BindException) e).getAllErrors();
        // 拼接错误信息
        for (ObjectError oe : allErrors) {
            // 获取java bean字段上标注的错误信息
            if (oe instanceof FieldError) {
                String message = String.format("%s: %s", ((FieldError) oe).getField(), oe.getDefaultMessage());
                errorMessageSb.append(message).append("; ");
                errorMap.put(((FieldError) oe).getField(), oe.getDefaultMessage());
            } else {
                errorMap.put(oe.getObjectName(), oe.getDefaultMessage());
            }
        }

        log.error("exception: {}, uri: {}", JSON.toJSONString(errorMap),  request.getRequestURI());
        return Resp.fail(errorMessageSb.toString()).wrap();
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public Object constraintViolationException(ConstraintViolationException e, HttpServletRequest request) {
        log.error("exception: {}, uri: {}", e.getMessage(),  request.getRequestURI());
        // 错误信息
        StringBuilder sb = new StringBuilder();
        // 判断异常就是ConstraintViolationException：普通参数(非 java bean)校验出错时抛出 把校验注解写在参数上
        // 遍历校验失败的参数
        for (ConstraintViolation<?> cv : e.getConstraintViolations()) {
            String path = ((PathImpl) cv.getPropertyPath()).getLeafNode().getName();
            String message = String.format("%s: %s", path, cv.getMessage());
            sb.append(message).append("; ");
        }
        return Resp.fail(sb.toString()).wrap();
    }

    /**
     * 405 - Method Not Allowed
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public Object handleHttpRequestMethodNotSupportedException(HttpRequestMethodNotSupportedException e,
                                                                 HttpServletRequest request,
                                                                 HttpServletResponse response) {
        log.error("exception: {}, uri: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        String method = request.getMethod();
        return Resp.fail(HttpStatus.METHOD_NOT_ALLOWED.value(),"Request method " + method + " is not supported").wrap();
    }

    /**
     * 415 - Unsupported Media Type
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public Object handleHttpMediaTypeNotSupportedException(Exception e, HttpServletRequest request) {
        log.error("Unsupported Media Type, exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        return Resp.fail(HttpStatus.UNSUPPORTED_MEDIA_TYPE.value(), "Unsupported media type").wrap();
    }

    /**
     * 处理空指针的异常
     */
    @ExceptionHandler(value =NullPointerException.class)
    public Object exceptionHandler(HttpServletRequest request, Exception e){
        log.error("exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        return Resp.fail("Server error").wrap();
    }

    @ExceptionHandler(value = IllegalArgumentException.class)
    public Object illegalArgumentException(HttpServletRequest request, Exception e){
        log.error("exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        return Resp.fail(e.getMessage()).wrap();
    }
    
    /**
     * 业务异常
     */
    @ExceptionHandler(BizException.class)
    public Object bizException(BizException e, HttpServletRequest request) {
        log.error("exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        return Resp.fail(e.getCode(), e.getMessage()).wrap();
    }

    /**
     * 基本异常
     */
    @ExceptionHandler(BaseException.class)
    public Object baseException(BaseException e, HttpServletRequest request) {
        log.error("exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        return Resp.fail(e.getCode(), e.getMessage()).wrap();
    }

    /**
     * 声明要捕获的异常
     * @param request 请求体
     * @param e 异常
     * @return 返回的结果
     */
    @ExceptionHandler(Exception.class)
    public Object defaultExceptionHandler(Exception e, HttpServletRequest request) {
        log.error("exception: {}, url: {}", e.getMessage(), request.getRequestURI());
        log.debug(e.getMessage(), e);
        return Resp.fail("Server error").wrap();
    }
}
