package cn.fscode.commons.sms.manager;

import cn.fscode.commons.sms.manager.entity.AliyunSmsReps;
import cn.fscode.commons.sms.manager.entity.AliyunSmsReq;
import cn.fscode.commons.sms.properties.AliyunSmsProperties;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsRequest;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author shenguangyang
 */
public class AliyunSmsManager implements ISmsManager<AliyunSmsReq, AliyunSmsReps> {
    private static final Logger log = LoggerFactory.getLogger(AliyunSmsManager.class);
    /** 短信API产品名称（短信产品名固定，无需修改） */
    private static final String PRODUCT = "Dysmsapi";
    /** 短信API产品域名（接口地址固定，无需修改） */
    private static final String DOMAIN  = "dysmsapi.aliyuncs.com ";
    private static final String SUCCESS_CODE = "OK";
    private final AliyunSmsProperties aliyunSmsProperties;
    private IAcsClient acsClient;

    public AliyunSmsManager(AliyunSmsProperties aliyunSmsProperties) {
        this.aliyunSmsProperties = aliyunSmsProperties;
        this.init();
    }

    private void init() {
        String accessKeySecret = this.aliyunSmsProperties.getAccessKeySecret();
        String accessKeyId = this.aliyunSmsProperties.getAccessKeyId();
        // 初始化ascClient
        IClientProfile profile = DefaultProfile.getProfile("cn-hangzhou", accessKeyId,
                accessKeySecret);
        DefaultProfile.addEndpoint("cn-hangzhou", PRODUCT, "cn-hangzhou");
        this.acsClient = new DefaultAcsClient(profile);
    }

    @Override
    public AliyunSmsReps send(AliyunSmsReq req) {
        req.check();
        SendSmsRequest request = new SendSmsRequest();
        // 使用post提交
        request.setSysMethod(MethodType.POST);
        // 必填:待发送手机号。支持以逗号分隔的形式进行批量调用，批量上限为1000个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式；发送国际/港澳台消息时，接收号码格式为国际区号+号码，如“85200000000”
        request.setPhoneNumbers(req.getPhoneNumbers());
        // 必填:短信签名-可在短信控制台中找到
        request.setSignName(req.getSignName());
        // 必填:短信模板-可在短信控制台中找到，发送国际/港澳台消息时，请使用国际/港澳台短信模版
        request.setTemplateCode(req.getTemplateId());
        // 可选:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为
        // 友情提示:如果JSON中需要带换行符,请参照标准的JSON协议对换行符的要求,比如短信内容中包含\r\n的情况在JSON中需要表示成\\r\\n,否则会导致JSON在服务端解析失败
        // 参考：request.setTemplateParam("{\"变量1\":\"值1\",\"变量2\":\"值2\",\"变量3\":\"值3\"}")
        request.setTemplateParam(req.getTemplateParam().toJSONString());
        // 可选-上行短信扩展码(扩展码字段控制在7位或以下，无特殊需求用户请忽略此字段)
        request.setSmsUpExtendCode(req.getSmsUpExtendCode());
        // 可选:outId为提供给业务方扩展字段,最终在短信回执消息中将此值带回给调用者
        request.setOutId(req.getOutId());
        // 请求失败这里会抛ClientException异常
        SendSmsResponse sendSmsResponse;
        try {
            sendSmsResponse = acsClient.getAcsResponse(request);
            return AliyunSmsReps.builder()
                            .code(sendSmsResponse.getCode()).message(sendSmsResponse.getMessage())
                            .isSuccess(SUCCESS_CODE.equals(sendSmsResponse.getCode()))
                            .requestId(sendSmsResponse.getRequestId())
                    .build();
        } catch (ClientException e) {
            log.error("send sms fail, errorCode: {}, errorMessage: {}", e.getErrCode(), e.getMessage());
            return AliyunSmsReps.builder()
                    .code(e.getErrCode()).message(e.getMessage())
                    .isSuccess(false)
                    .requestId(e.getRequestId()).build();
        }
    }
}
