package cn.fscode.commons.database;

import cn.fscode.commons.core.constant.CommonsCoreConstants;
import cn.fscode.commons.database.sql.init.DatasourceConfig;
import cn.fscode.commons.tool.core.StringUtils;
import cn.fscode.commons.tool.core.exception.Assert;
import cn.fscode.commons.tool.core.exception.BizException;
import lombok.Getter;
import lombok.Setter;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.MutablePropertySources;

import java.util.*;

/**
 * @author shenguangyang
 */
@Getter
@Setter
@ConfigurationProperties(prefix = CommonsCoreConstants.PROPERTIES_PRE + "sql")
public class PcSqlProperties {
    private List<Init> inits;

    @Getter
    @Setter
    public static class Init {
        /**
         * 是否使能初始化db数据库
         */
        private Boolean enabled = Boolean.FALSE;

        /**
         * 如果采用了多数据源, 这里徐娅指定作用在哪个数据源上<br/>
         * 比如: 你配置了多个数据源, 然后你只想要其作用其中一个数据源上则可以指定 spring.datasource.dynamic.datasource 下
         * 的具体数据源key </br>
         *
         * 如果你没有使用多数据源请忽略该配置
         */
        private String useDatasource = "";

        /**
         * 指定初始化数据库的脚本(位于resources目录下的脚本)
         */
        private String sqlScript;
    }


    @Getter
    @Setter
    public static class Helper {
        private Environment environment;

        /**
         * application.yaml 中假如有如下配置 <br/>
         * <blockquote><pre>
         * test:
         *  - name: 12aeda
         *    age: 2
         * </pre></blockquote>
         *
         * 则 environment可以通过 test[0].name 和 test[0].age获取到值, 而这里的envIndex = "[0]"
         */
        private String envIndex = "";
        private String itemKeyPre;
        private static String KEY_PRE = CommonsCoreConstants.PROPERTIES_PRE + "sql.inits";

        public Helper(Environment environment, String envIndex) {
            this.environment = environment;
            this.envIndex = envIndex;
            this.itemKeyPre = CommonsCoreConstants.PROPERTIES_PRE + "sql.inits" + this.envIndex;
            Assert.notNull(environment, "environment is null");
            Assert.notNull(environment, "envIndex is null");
        }

        public static List<Helper> build(Environment environment) {
            List<Helper> helpers = new ArrayList<>();
            AbstractEnvironment aEnv = (AbstractEnvironment) environment;
            MutablePropertySources propertySources = aEnv.getPropertySources();
            propertySources.forEach(propertySource -> {
                if (propertySource instanceof MapPropertySource) {
                    MapPropertySource mps = (MapPropertySource) propertySource;
                    Set<String> keys = mps.getSource().keySet();
                    Map<String, String> temp = new HashMap<>();
                    for (String key : keys) {
                        if (key.startsWith(KEY_PRE)) {
                            String envIndex = key.replace(KEY_PRE, "").split("\\.")[0];
                            if (!temp.containsKey(envIndex)) {
                                temp.put(envIndex, "");
                                helpers.add(new Helper(environment, envIndex));
                            }
                        }
                    }
                }
            });
            return helpers;
        }

        public Boolean enabled() {
            return Boolean.parseBoolean(environment.getProperty(itemKeyPre + ".enabled", "false"));
        }

        public String getUseDatasource() {
            return environment.getProperty(itemKeyPre + ".use-datasource", "");
        }

        public String getSqlScript() {
            return environment.getProperty(itemKeyPre + ".sql-script", "");
        }

        public DatasourceConfig getDatasourceConfig() {
            DatasourceConfig.DatasourceConfigBuilder builder = DatasourceConfig.builder();
            String useDatasource = getUseDatasource();
            // 说明使用了动态数据源
            if (StringUtils.isNotEmpty(useDatasource)) {
                String driverClassName = environment.getProperty("spring.datasource.dynamic.datasource." + useDatasource + ".driver-class-name");
                if (Objects.isNull(driverClassName)) {
                    throw new BizException("sql init fail, use datasource [{}] not exist", useDatasource);
                }
                builder.driverClassName(driverClassName)
                        .username(environment.getProperty("spring.datasource.dynamic.datasource." + useDatasource + ".username"))
                        .password(environment.getProperty("spring.datasource.dynamic.datasource." + useDatasource + ".password"))
                        .jdbcUrl(environment.getProperty("spring.datasource.dynamic.datasource." + useDatasource + ".url"));
            } else {
                String driverClassName = environment.getProperty("spring.datasource.driver-class-name");
                if (Objects.isNull(driverClassName)) {
                    throw new BizException("sql init fail," +
                            " please config [spring.datasource.driver-class-name] or config [" + itemKeyPre + ".use-datasource]" + " to specify the data source");
                }
                builder.driverClassName(environment.getProperty("spring.datasource.driver-class-name"))
                        .jdbcUrl(environment.getProperty("spring.datasource.url"))
                        .username(environment.getProperty("spring.datasource.username"))
                        .password(environment.getProperty("spring.datasource.password"));
            }
            builder.sqlScript(getSqlScript());
            return builder.build();
        }
    }
}
