package cn.freelancy.sxtwl4j;

import cn.freelancy.sxtwl4j.bean.Day;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

import static cn.freelancy.sxtwl4j.util.NumberUtil.int2;

/**
 * @author Yawei Xi
 * @since 2018-8-9 14:53
 */
public class BaZi {

    /**
     * 推算十神用带数组，规则如下：
     * 金水木火土（阳）/金水木火土（阴）
     * 1 3 5 7 9     0 2 4 6 8
     * (1,3)(3,5)(5,7)(7,9)(9,1)(0,2)(2,4)(4,6)(6,8)(8,0)食
     * (0,3)(2,5)(4,7)(6,9)(8,1)(1,2)(3,4)(5,6)(7,8)(9,0)伤
     * (0,0)(1,1)(2,2)(3,3)(4,4)(5,5)(6,6)(7,7)(8,8)(9,9)比
     * (1,0)(0,1)(2,3)(3,2)(4,5)(5,4)(6,7)(7,6)(8,9)(9,8)劫
     * (3,1)(5,3)(7,5)(9,7)(1,9)(2,0)(4,2)(6,4)(8,6)(0,8)枭
     * (3,0)(5,2)(7,4)(9,6)(1,8)(2,1)(4,3)(6,5)(8,7)(0,9)印
     * (1,5)(5,9)(9,3)(3,7)(7,1)(0,4)(4,8)(8,2)(2,6)(6,0)才
     * (1,4)(5,8)(9,2)(3,6)(7,0)(0,5)(4,9)(8,3)(2,7)(6,1)财
     * (5,1)(9,5)(3,9)(7,3)(1,7)(4,0)(8,4)(2,8)(6,2)(0,6)杀
     * (4,1)(8,5)(2,9)(6,3)(0,7)(5,0)(9,4)(3,8)(7,2)(1,6)官
     */
    public static final char[][] SHI_SHEN = {
            {'比', '劫', '食', '伤', '才', '财', '杀', '官', '枭', '印'},
            {'劫', '比', '伤', '食', '财', '才', '官', '杀', '印', '枭'},
            {'枭', '印', '比', '劫', '食', '伤', '才', '财', '杀', '官'},
            {'印', '枭', '劫', '比', '伤', '食', '财', '才', '官', '杀'},
            {'杀', '官', '枭', '印', '比', '劫', '食', '伤', '才', '财'},
            {'官', '杀', '印', '枭', '劫', '比', '伤', '食', '财', '才'},
            {'才', '财', '杀', '官', '枭', '印', '比', '劫', '食', '伤'},
            {'财', '才', '官', '杀', '印', '枭', '劫', '比', '伤', '食'},
            {'食', '伤', '才', '财', '杀', '官', '枭', '印', '比', '劫'},
            {'伤', '食', '财', '才', '官', '杀', '印', '枭', '劫', '比'}
    };
    // 阳金水木火土，阴金水木火土
    public static final int[] tenElements = {1, 3, 5, 7, 9, 0, 2, 4, 6, 8};
    public static final Map<Character, Integer> getIndexByGanOrZhi = new HashMap<Character, Integer>() {
        {
            put('甲', tenElements[2]);
            put('乙', tenElements[7]);
            put('丙', tenElements[3]);
            put('丁', tenElements[8]);
            put('戊', tenElements[4]);
            put('己', tenElements[9]);
            put('庚', tenElements[0]);
            put('辛', tenElements[5]);
            put('壬', tenElements[1]);
            put('癸', tenElements[6]);
            put('子', tenElements[1]);
            put('丑', tenElements[9]);
            put('寅', tenElements[2]);
            put('卯', tenElements[7]);
            put('辰', tenElements[4]);
            put('巳', tenElements[8]);
            put('午', tenElements[3]);
            put('未', tenElements[9]);
            put('申', tenElements[0]);
            put('酉', tenElements[5]);
            put('戌', tenElements[4]);
            put('亥', tenElements[6]);
        }
    };
    private static final SimpleDateFormat SIMPLE_DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    /**
     * 年干
     */
    private char nianGan;
    /**
     * 年支
     */
    private char nianZhi;
    /**
     * 月干
     */
    private char yueGan;
    /**
     * 月支
     */
    private char yueZhi;
    /**
     * 日干
     */
    private char riGan;
    /**
     * 日支
     */
    private char riZhi;
    /**
     * 时干
     */
    private char shiGan;
    /**
     * 时支
     */
    private char shiZhi;
    /**
     * 真太阳时
     */
    private String trueSolarTime;
    /**
     * 纪时
     */
    private String jiShi;
    /**
     * 日标
     */
    private String riBiao;
    /**
     * 时标
     */
    private String shiBiao;
    /**
     * 年（纳音）
     */
    private String naYin4Year;
    /**
     * 月（纳音）
     */
    private String naYin4Month;
    /**
     * 日（纳音）
     */
    private String naYin4Day;
    /**
     * 时（纳音）
     */
    private String naYin4Hour;
    /**
     * 纳音
     */
    private String naYin;
    /**
     * 公历生日
     */
    private Date birthDay;
    /**
     * 年（公历）
     */
    private int solarYear;
    /**
     * 年（农历）
     */
    private int lunarYear;

    /**
     * 获取八字信息
     *
     * @param year      年
     * @param month     月
     * @param day       日
     * @param hour      时
     * @param minute    分
     * @param second    秒
     * @param longitude 经度
     * @param curTZ     时区 -8为北京时
     */
    public BaZi(int year, int month, int day, int hour, int minute, int second, double longitude, int curTZ) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(year, month, day, hour, minute, second);
        birthDay = calendar.getTime();

        solarYear = year;
        Lunar lunar = new Lunar();
        Day detailDay = lunar.getDay(year, month, day);
        lunarYear = null != detailDay ? detailDay.getLunarYear() : year;

        ChineseFestivals chineseFestivals = new ChineseFestivals();
        double t = hour + minute / 60.0 + second / 3600.0;
        double jd = JD.jd(Common.year2Ayear(year), month, day + t / 24);
        // 八字计算
        chineseFestivals.mingLiBaZi(jd + curTZ / 24.0 - Common.J2000, longitude / Common.radd, this);

        riBiao = "公历 " + year + "-" + month + "-" + day + " 儒略日数 " + int2(jd + 0.5) + " 距2000年首" + int2(jd + 0.5 - Common.J2000) + "日";
        shiBiao = "23　 01　 03　 05　 07　 09　 11　 13　 15　 17　 19　 21　 23";
    }

    /**
     * 通过日干和需要判断的字符推断出食神
     *
     * @param riGan 日干
     * @param other 需要判断的字符
     * @return 食神字符
     */
    public static char getShiShenByRiGanAndOther(char riGan, char other) {
        int ri = getIndexByGanOrZhi.get(riGan);
        int ot = getIndexByGanOrZhi.get(other);
        return SHI_SHEN[ri][ot];
    }

    private static int getIndexOfTheYear(Calendar calInput, String[] jqStrings) {
        int i;
        for (i = 0; i < jqStrings.length; i++) {
            Date date = getDateByStr(jqStrings[i]);
            if (date.getTime() >= calInput.getTimeInMillis()) {
                break;
            }
        }
        return i - 1;
    }

    /**
     * 通过日期字符串和格式化类型来获取日期对象
     *
     * @param dateStr 日期字符串
     * @return 日期对象
     */
    private static Date getDateByStr(String dateStr) {
        try {
            return SIMPLE_DATE_FORMAT.parse(dateStr);
        } catch (ParseException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    private static int indexOf(char[] array, char valueToFind) {
        return indexOf((char[]) array, (char) valueToFind, 0);
    }

    private static int indexOf(char[] array, char valueToFind, int startIndex) {
        if (array == null) {
            return -1;
        } else {
            if (startIndex < 0) {
                startIndex = 0;
            }

            for (int i = startIndex; i < array.length; ++i) {
                if (valueToFind == array[i]) {
                    return i;
                }
            }

            return -1;
        }
    }

    /**
     * 获取八字信息，如：戊戌庚申戊寅辛酉
     *
     * @return
     */
    public String getBaZi() {
        return new String(new char[]{nianGan, nianZhi, yueGan, yueZhi, riGan, riZhi, shiGan, shiZhi});
    }

    /**
     * 通过性别获取大运
     *
     * @param gender 性别：男(1), 女(0)
     * @return 大运列表
     */
    public List<String> getDaYunByGender(int gender) {
        // 默认数量为8
        int defaultSize = 8;
        return getDaYunByGender(gender, defaultSize);
    }

    /**
     * 通过性别获取大运
     *
     * @param gender 性别：男(1), 女(0)
     * @param size   数量
     * @return 大运列表
     */
    public List<String> getDaYunByGender(int gender, int size) {
        List<String> daYun = new ArrayList<>();

        int monthGan = indexOf(ChineseFestivals.GAN, yueGan);
        int monthZhi = indexOf(ChineseFestivals.ZHI, yueZhi);

        if (isPaiDaYunDirPositive(gender, nianGan)) {
            for (int i = 0; i < size; i++) {
                daYun.add(ChineseFestivals.GAN[(++monthGan) % 10] + "" + (ChineseFestivals.ZHI[(++monthZhi) % 12]));
            }
        } else {
            for (int i = 0; i < size; i++) {
                daYun.add(ChineseFestivals.GAN[((--monthGan) + 10) % 10] + "" + (ChineseFestivals.ZHI[((--monthZhi) + 12) % 12]));
            }
        }

        return daYun;
    }

    /**
     * 输入性别和年干，返回排大运的方向
     *
     * @param gender  性别
     * @param nianGan 年干
     * @return 正(1true), 负(false)
     */
    private boolean isPaiDaYunDirPositive(int gender, char nianGan) {
        return (isTianGanYang(nianGan) && gender == 1) || (!isTianGanYang(nianGan) && gender == 0);
    }

    /**
     * 判断某天干是否为“阳”
     *
     * @param tianGan 天干
     * @return 阳(true), 阴(false)
     */
    private boolean isTianGanYang(char tianGan) {
        return indexOf(ChineseFestivals.GAN, tianGan) % 2 == 0;
    }

    /**
     * 判断某地支是否为“阳”
     *
     * @param diZhi 地支
     * @return 阳(true), 阴(false)
     */
    private boolean isDiZhiYang(char diZhi) {
        return indexOf(ChineseFestivals.ZHI, diZhi) % 2 == 0;
    }

    /**
     * 获取空亡
     *
     * @return 空亡
     */
    public String getKongWang() {
        int gan = indexOf(ChineseFestivals.GAN, riGan);
        int zhi = indexOf(ChineseFestivals.ZHI, riZhi);
        int kong = (10 - gan - (12 - zhi) + 12) % 12;
        return ChineseFestivals.ZHI[kong] + "" + ChineseFestivals.ZHI[kong + 1];
    }

    /**
     * 获取起运日期
     *
     * @param gender 性别：男(1), 女(0)
     * @return 起运的日期
     */
    public Date getQiYunDate(int gender) {
        try {
            Calendar calendar = Calendar.getInstance();
            Calendar cal = Calendar.getInstance();
            cal.setTime(birthDay);
            long result;// 获取毫秒数

            String[] jie = getThisYearAndNextYearJieWithOutQi();
            int jieQiIndex = getIndexOfTheYear(cal, jie);

            if (isPaiDaYunDirPositive(gender, nianGan)) {
                calendar.setTime(SIMPLE_DATE_FORMAT.parse(jie[jieQiIndex + 1]));
                result = calendar.getTimeInMillis() - cal.getTimeInMillis();
            } else {
                calendar.setTime(SIMPLE_DATE_FORMAT.parse(jie[jieQiIndex]));
                result = cal.getTimeInMillis() - calendar.getTimeInMillis();
            }
            result = (result / 7200000L) / 3;

            cal.add(Calendar.MONTH, (int) result);
            return cal.getTime();
        } catch (ParseException e) {
            e.printStackTrace();
            return new Date();
        }
    }

    /**
     * 获取当年的节气中的单数的日期
     *
     * @return 24节气中的奇数的日期数组
     */
    private String[] getThisYearAndNextYearJieWithOutQi() {
        Lunar lunar = new Lunar();
        List<String> jieQi = lunar.get24JieQiByYear(lunarYear);
        jieQi.addAll(lunar.get24JieQiByYear(lunarYear + 1));
        List<String> jie = new ArrayList<>();
        for (String s : jieQi) {
            if (jieQi.indexOf(s) % 2 == 0) {
                jie.add(s);
            }
        }
        return jie.subList(0, 12).toArray(new String[]{});
    }

    public char getNianGan() {
        return nianGan;
    }

    public void setNianGan(char nianGan) {
        this.nianGan = nianGan;
    }

    public char getNianZhi() {
        return nianZhi;
    }

    public void setNianZhi(char nianZhi) {
        this.nianZhi = nianZhi;
    }

    public char getYueGan() {
        return yueGan;
    }

    public void setYueGan(char yueGan) {
        this.yueGan = yueGan;
    }

    public char getYueZhi() {
        return yueZhi;
    }

    public void setYueZhi(char yueZhi) {
        this.yueZhi = yueZhi;
    }

    public char getRiGan() {
        return riGan;
    }

    public void setRiGan(char riGan) {
        this.riGan = riGan;
    }

    public char getRiZhi() {
        return riZhi;
    }

    public void setRiZhi(char riZhi) {
        this.riZhi = riZhi;
    }

    public char getShiGan() {
        return shiGan;
    }

    public void setShiGan(char shiGan) {
        this.shiGan = shiGan;
    }

    public char getShiZhi() {
        return shiZhi;
    }

    public void setShiZhi(char shiZhi) {
        this.shiZhi = shiZhi;
    }

    public String getTrueSolarTime() {
        return trueSolarTime;
    }

    public void setTrueSolarTime(String trueSolarTime) {
        this.trueSolarTime = trueSolarTime;
    }

    public String getJiShi() {
        return jiShi;
    }

    public void setJiShi(String jiShi) {
        this.jiShi = jiShi;
    }

    public String getRiBiao() {
        return riBiao;
    }

    public void setRiBiao(String riBiao) {
        this.riBiao = riBiao;
    }

    public String getShiBiao() {
        return shiBiao;
    }

    public void setShiBiao(String shiBiao) {
        this.shiBiao = shiBiao;
    }

    public String getNaYin4Year() {
        return naYin4Year;
    }

    public void setNaYin4Year(String naYin4Year) {
        this.naYin4Year = naYin4Year;
    }

    public String getNaYin4Month() {
        return naYin4Month;
    }

    public void setNaYin4Month(String naYin4Month) {
        this.naYin4Month = naYin4Month;
    }

    public String getNaYin4Day() {
        return naYin4Day;
    }

    public void setNaYin4Day(String naYin4Day) {
        this.naYin4Day = naYin4Day;
    }

    public String getNaYin4Hour() {
        return naYin4Hour;
    }

    public void setNaYin4Hour(String naYin4Hour) {
        this.naYin4Hour = naYin4Hour;
    }

    public String getNaYin() {
        return naYin;
    }

    public void setNaYin(String naYin) {
        this.naYin = naYin;
    }

    public Date getBirthDay() {
        return birthDay;
    }

    public void setBirthDay(Date birthDay) {
        this.birthDay = birthDay;
    }

    public int getSolarYear() {
        return solarYear;
    }

    public void setSolarYear(int solarYear) {
        this.solarYear = solarYear;
    }

    public int getLunarYear() {
        return lunarYear;
    }

    public void setLunarYear(int lunarYear) {
        this.lunarYear = lunarYear;
    }
}
