package cn.fraudmetrix.riskservice.ruledetail.parse;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class LazyLoadMap<K, V> {

    private static final Log   logger        = LogFactory.getLog(LazyLoadMap.class);

    private Map<K, V>          map           = new ConcurrentHashMap<K, V>();
    private Map<K, SimpleLock> creating      = new ConcurrentHashMap<K, SimpleLock>();
    private Object             CREATE_FAILED = new Object();

    public abstract V create(K key);

    public V get(K key) {
        V r = getMayFailed(key);
        if (r == CREATE_FAILED) return null;
        return r;
    }

    private V getMayFailed(K key) {
        // 如果已经有了，直接返回即可
        if (map.containsKey(key)) {
            return map.get(key);
        }

        // 如果还没有，则需要创建
        return createKey(key);
    }

    private V createKey(K key) {
        // 如果没有正在创建，就创建一个
        createKeyNonExists(key);

        // 如果正在创建，就等它创建好之后一起返回
        SimpleLock lock = creating.get(key);
        if (lock == null) return map.get(key);
        lock.waitUnlock();
        return map.get(key);
    }

    private V createNoException(K key) {
        try {
            V result = create(key);
            if (result == null) {
                return (V) CREATE_FAILED;
            }
            return result;
        } catch (Throwable t) {
            logger.warn("Failed to create key: " + key, t);
            return (V) CREATE_FAILED;
        }
    }

    private void createKeyNonExists(K key) {
        SimpleLock lock;
        synchronized (this) {
            if (map.containsKey(key)) return;
            if (creating.containsKey(key)) return;
            lock = new SimpleLock();
            creating.put(key, lock);
        }
        V created = createNoException(key);
        synchronized (this) {
            map.put(key, created);
            lock.unlock();
            creating.remove(key);
        }
    }
}
