package cn.fraudmetrix.riskservice.observer;

import cn.fraudmetrix.riskservice.Version;
import cn.fraudmetrix.riskservice.observer.object.InOutTime;
import cn.fraudmetrix.riskservice.observer.object.InOutTimeStats;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.message.BasicNameValuePair;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.http.util.EntityUtils;

/**
 * 观察者.被观察者是  {@link cn.fraudmetrix.riskservice.RiskServiceClient}
 * 搜集客户端数据,并将其统计结果发送到服务器
 * @author kui.yuan@fruadmetrix.cn
 */
public class ClientDataObserver implements Observer {
    private final HttpClient httpClient;
    private final String url;
    private final String partnerCode;
    private ScheduledExecutorService peridFlushService = new ScheduledThreadPoolExecutor(1);
    private boolean isDead = false;
    InOutTimeStats inOutTimeStats = new InOutTimeStats();

    public ClientDataObserver(int flushIntervalInSeconds, final HttpClient httpClient, final String url, String partnerCode) {
        this.httpClient = httpClient;
        this.url = url;
        this.partnerCode = partnerCode;
        peridFlushService.scheduleAtFixedRate(new Runnable() {
            public void run() {
                flush();
            }
        }, flushIntervalInSeconds, flushIntervalInSeconds, TimeUnit.SECONDS);
    }

    public HttpEntity buildEntity(InOutTimeStats inOutTimeStats) {
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
        nameValuePairs.add(new BasicNameValuePair("averageOutTimeMillis", String.valueOf(inOutTimeStats.getAvgOutTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("maxOutTimeMillis", String.valueOf(inOutTimeStats.getMaxOutTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("minOutTimeMills", String.valueOf(inOutTimeStats.getMinOutTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("averageInTimeMillis", String.valueOf(inOutTimeStats.getAvgInTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("maxInTimeMillis", String.valueOf(inOutTimeStats.getMaxInTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("minInTimeMillis", String.valueOf(inOutTimeStats.getMinInTimeMillis())));
        nameValuePairs.add(new BasicNameValuePair("count", String.valueOf(inOutTimeStats.getCount())));
        nameValuePairs.add(new BasicNameValuePair("clientVersion", Version.getVersion()));
        nameValuePairs.add(new BasicNameValuePair("partnerCode", this.partnerCode));
        return new UrlEncodedFormEntity(nameValuePairs, Charset.defaultCharset());
    }

    public void update(Observable riskClient, Object inOutTime) {
        if (!isDead) {
            InOutTime clientStats = null;
            if (inOutTime instanceof InOutTime) {
                clientStats = (InOutTime) inOutTime;
            } else {
                return;
            }
            innerUpdate(clientStats);
        }

    }

    private void innerUpdate(InOutTime clientStats) {
        inOutTimeStats.take(clientStats.getInTimeMillis(), clientStats.getOutTimeMillis());
    }

    public void flush() {
        if (inOutTimeStats.getCount() <= 0) {
            return;
        }
        HttpPost httpPost = new HttpPost(url);
        httpPost.setEntity(buildEntity(inOutTimeStats));
        ClientDataObserver.this.inOutTimeStats = new InOutTimeStats();
        HttpResponse response = null;
        try {
            response = httpClient.execute(httpPost);
        } catch (IOException ignored) {
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {
                } finally {
                    httpPost.releaseConnection();
                }
            }
        }
    }

}
