/*
 * Copyright 2015 FraudMetrix.cn All right reserved. This software is the
 * confidential and proprietary information of FraudMetrix.cn ("Confidential
 * Information"). You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license agreement you entered
 * into with FraudMetrix.cn.
 */
package cn.fraudmetrix.riskservice;

import cn.fraudmetrix.riskservice.object.Environment;
import cn.fraudmetrix.riskservice.object.ProxyModel;
import cn.fraudmetrix.riskservice.object.RiskResult;
import cn.fraudmetrix.riskservice.observer.ClientDataObserver;
import cn.fraudmetrix.riskservice.observer.object.InOutTime;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.Asserts;
import org.apache.http.util.EntityUtils;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Observable;

/**
 * <p>
 * 风险决策服务客户端，先调用getInstance获取唯一实例，然后调用实例的execute方法访问风险决策接口。
 * </p>
 * <p>
 * 特别注意：
 * </p>
 * <p>
 * 多次调用getInstance获取的是同一实例，但只有第一次调用会按照给定参数进行初始化。
 * </p>
 * <p>
 * 第二次及之后调用时传入的参数将不会产生任何效果。
 * </p>
 *
 * @author ming.ma@fruadmetrix.cn
 */
public class RiskServiceClient extends Observable {

    private static final Log               log             = LogFactory.getLog(RiskServiceClient.class);

    /**
     * 风险决策接口地址
     */
    private String                         apiUrl;

    /**
     * URLencode编码默认使用的字符集
     */
    private static final String            DEFAULT_CHARSET = "utf-8";

    /**
     * 合作方代码
     */
    private NameValuePair                  partnerCode;

    /**
     * 参数URLencode编码所使用的字符集
     */
    private String                         charset;

    /**
     *  请求参数整体加密秘钥
     */
    private String                         appDataSecrectKey;

    private CloseableHttpClient            httpClient;
    private static final RiskServiceClient instance        = new RiskServiceClient();
    private static boolean                 inited;
    /**
     * ssl证书位置
     */
    private String                         sslCertPath;

    private RiskServiceClient(){
    }

    private void init(String partnerCode, Environment environment, int connectTimeout, int readTimeout,
                      int maxConnection, String charset, ProxyModel proxyModel, boolean heartbeatSwitch, String appDataSecrectKey) {
        synchronized (instance) {
            if (inited) {
                // throw new
                // IllegalStateException("You've already got a client instance, just use it. DO NOT call RiskServiceClient#getInstance() again!");
                return;
            }
            inited = true;
        }
        Asserts.notBlank(partnerCode, "partnerCode");
        Asserts.check(connectTimeout >= 500, "connectTimeout must >= 500ms.");
        Asserts.check(readTimeout >= 500, "readTimeout must >= 500ms.");
        Asserts.check(maxConnection > 0, "maxConnection must > 0.");

        if(sslCertPath != null && !"".equals(sslCertPath)){
            System.setProperty("javax.net.ssl.trustStore", sslCertPath);
        }
        if(appDataSecrectKey != null){
            this.appDataSecrectKey = appDataSecrectKey;
        }
        if (null == environment) environment = Environment.PRODUCT;
        this.apiUrl = environment.getApiUrl();

        this.partnerCode = new BasicNameValuePair("partner_code", partnerCode);
        this.charset = null == charset ? DEFAULT_CHARSET : charset;
        // 设置连接池
        PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager();
        // 设置连接池大小
        connMgr.setMaxTotal(maxConnection);
        connMgr.setDefaultMaxPerRoute(maxConnection);

        // 设置请求配置
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        // 设置连接超时
        configBuilder.setConnectTimeout(connectTimeout);
        // 设置读取超时
        configBuilder.setSocketTimeout(readTimeout);
        // 设置从连接池获取连接实例的超时
        configBuilder.setConnectionRequestTimeout(500);
        // 在提交请求之前 测试连接是否可用
        configBuilder.setStaleConnectionCheckEnabled(true);

        if(proxyModel != null){
            configBuilder.setProxy(proxyModel.getProxy());
        }
        RequestConfig requestConfig = configBuilder.build();

        httpClient = HttpClients.custom().setSSLSocketFactory(createSSLConnSocketFactory()).setConnectionManager(connMgr).setDefaultRequestConfig(requestConfig).build();
        if (heartbeatSwitch){
            Thread heartbeatThread = new Thread(new Runnable() {

                @Override
                public void run() {
                    while (true) {
                        RiskServiceClient.this.execute("heartbeat", "heartbeat", null);
                        try {
                            Thread.sleep(3000);
                        } catch (InterruptedException e) {
                            log.error(e);
                            break;
                        }
                    }
                }
            }, "RiskServiceClient Heartbeat Thread");
            heartbeatThread.setDaemon(true);
            heartbeatThread.start();
        }

        // 仅统计生产环境
        if (environment == Environment.PRODUCT) {
            // 增加客户端调用统计的观察者
            ClientDataObserver observer = new ClientDataObserver(
                                                                 60,
                                                                 httpClient,
                                                                 "https://api.fraudmetrix.cn/clientDataCollectService.json",
                                                                 partnerCode);
            this.addObserver(observer);
        }
    }

    private void init(String partnerCode, Environment environment, int connectTimeout, int readTimeout,
                      int maxConnection, String charset, ProxyModel proxyModel, boolean heartbeatSwitch) {
        init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, proxyModel, heartbeatSwitch, null);
    }

    private void init(String partnerCode, Environment environment, int connectTimeout, int readTimeout,
                      int maxConnection, String charset) {
        init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, null, true);
    }

    /**
     * 获取RiskServiceClient实例 默认配置使用生产环境API地址，url编码使用utf-8字符集 connectTimeout及readTimeout均为1000ms，最大并发连接50个
     *
     * @param partnerCode 合作方代码
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode) {
        return getInstance(partnerCode, null);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment) {
        return getInstance(partnerCode, environment, (String)null);
    }

    public static RiskServiceClient getInstance(String partnerCode, Environment environment,ProxyModel proxyModel) {
        return getInstance(partnerCode, environment, 1000, 1000, 50, null, proxyModel);
    }

    public static RiskServiceClient getInstance(String partnerCode, Environment environment,ProxyModel proxyModel, String appDataSecrectKey) {
        return getInstance(partnerCode, environment, 1000, 1000, 50, null, proxyModel,true, appDataSecrectKey);
    }

    /**
     *
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param heartbeatSwitch 心跳开关
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, boolean heartbeatSwitch) {
        return getInstance(partnerCode, environment, 1000, 1000, 50, null, null, heartbeatSwitch);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, String charset) {
        return getInstance(partnerCode, environment, 1000, 1000, 50, charset);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, null);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, String charset) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, 50, charset);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, int maxConnection) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, null);
    }

    /**
     * 获取RiskServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, int maxConnection, String charset) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, null);
    }


    /**
     * 获取RiskServiceClient实例
     *
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param appDataSecrectKey 加密秘钥，缺省值
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, String appDataSecrectKey, int connectTimeout,
                                                int readTimeout, int maxConnection) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, DEFAULT_CHARSET, null, true, appDataSecrectKey);
    }


    /**
     * 获取RiskServiceClient实例
     *
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @param proxyModel 代理类
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, int maxConnection, String charset,ProxyModel proxyModel) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, proxyModel, true);
    }

    /**
     *
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @param proxyModel 代理类
     * @param heartbeatSwitch 心跳开关
     * @return RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, int maxConnection, String charset,ProxyModel proxyModel,boolean heartbeatSwitch) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, proxyModel, heartbeatSwitch, null);
    }


    /**
     *
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @param proxyModel 代理类
     * @param heartbeatSwitch 心跳开关
     * @param appDataSecrectKey 加密秘钥
     * @return RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                                int readTimeout, int maxConnection, String charset,ProxyModel proxyModel,boolean heartbeatSwitch, String appDataSecrectKey) {
        instance.init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, proxyModel, heartbeatSwitch, appDataSecrectKey);
        return instance;
    }
    private HttpEntity buildRequestEntity(final String secretKey, final String eventId, final Map<String, String> params) {
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
        nameValuePairs.add(partnerCode);
        nameValuePairs.add(new BasicNameValuePair("secret_key", secretKey));
        nameValuePairs.add(new BasicNameValuePair("event_id", eventId));
        if (null != params) {
            for (String key : params.keySet()) {
                nameValuePairs.add(new BasicNameValuePair(key, params.get(key)));
            }
        }
        return new UrlEncodedFormEntity(nameValuePairs, Charset.forName(this.charset));
    }

    /**
     * 调用风险决策服务，获取决策结果
     *
     * @param secretKey 应用密钥
     * @param eventId 事件ID
     * @param params 业务参数
     * @return 风险决策结果
     */
    public RiskResult execute(final String secretKey, final String eventId, final Map<String, String> params) {
        RiskResult riskResult = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        HttpEntity entity;
        try {
            httpPost.setEntity(buildRequestEntity(secretKey, eventId, params));
            long requestStartTime = System.currentTimeMillis();
            response = httpClient.execute(httpPost);
            long requestEndTime = System.currentTimeMillis();

            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                log.warn("execute failed, response status: " + statusCode);
                return null;
            }
            entity = response.getEntity();
            if (entity == null) {
                log.warn("execute failed, response output is null!");
                return null;
            }
            String result = EntityUtils.toString(entity, "utf-8");
            result = result.trim();
            log.debug("execute result: " + result);
            riskResult = JSON.parseObject(result, RiskResult.class);
            try {
                long outCost = requestEndTime - requestStartTime;
                int inTime = riskResult.getSpend_time().intValue();
                InOutTime inOutTime = new InOutTime(inTime, outCost);
                setChanged();
                notifyObservers(inOutTime);
            } catch (Exception ignored) {
            }
        } catch (Exception e) {
            if (!"heartbeat".equals(secretKey)) {
                log.error("execute throw exception, details: ", e);
                riskResult = RiskResult.failedWithReason("000:" + e.toString());
            }
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {
                } finally {
                    httpPost.releaseConnection();
                }
            }
        }
        return riskResult;
    }


    /**
     * 加密调用风险决策服务，获取决策结果
     *
     * @param secretKey 应用密钥
     * @param eventId 事件ID
     * @param params 业务参数
     * @return 风险决策结果
     */
    public RiskResult executeWithSecret(final String secretKey, final String eventId, final Map<String, String> params) {
        Asserts.notBlank(appDataSecrectKey, "appDataSecrectKey不能为空");
        RiskResult riskResult = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        String partnerCodeValue = partnerCode.getValue();
        httpPost.addHeader("X-Partner-Code", partnerCodeValue);
        params.put("partner_code", partnerCodeValue);
        params.put("secret_key", secretKey);
        params.put("event_id", eventId);

        String base64 = encrypt(JSONObject.toJSONString(params), appDataSecrectKey);
        if(base64 == null){
            log.error("请检查秘钥是否合法,参数加密失败");
            riskResult = RiskResult.failedWithReason("000:" + "参数加密失败");
            return riskResult;
        }
        CloseableHttpResponse response = null;
        HttpEntity entity;
        try {
            httpPost.setEntity(new StringEntity(base64, ContentType.APPLICATION_OCTET_STREAM));
            long requestStartTime = System.currentTimeMillis();
            response = httpClient.execute(httpPost);
            long requestEndTime = System.currentTimeMillis();

            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                log.warn("execute failed, response status: " + statusCode);
                return null;
            }
            entity = response.getEntity();
            if (entity == null) {
                log.warn("execute failed, response output is null!");
                return null;
            }
            String result = EntityUtils.toString(entity, "utf-8");
            result = decrypt(result, appDataSecrectKey);
            if(result == null){
                log.error("请检查秘钥和参数是否合法,请求结果解密失败");
                riskResult = RiskResult.failedWithReason("000:" + "请求结果解析失败");
            }
            else{
                result = result.trim();
                log.debug("execute result: " + result);
                riskResult = JSON.parseObject(result, RiskResult.class);
                try {
                    long outCost = requestEndTime - requestStartTime;
                    int inTime = riskResult.getSpend_time().intValue();
                    InOutTime inOutTime = new InOutTime(inTime, outCost);
                    setChanged();
                    notifyObservers(inOutTime);
                } catch (Exception ignored) {
                }
            }

        } catch (Exception e) {
            if (!"heartbeat".equals(secretKey)) {
                log.error("execute throw exception, details: ", e);
                riskResult = RiskResult.failedWithReason("000:" + e.toString());
            }
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {
                } finally {
                    httpPost.releaseConnection();
                }
            }
        }
        return riskResult;
    }

    public String encrypt(String sSrc, String sKey) {
        // 判断Key是否正确
        if (sKey == null) {
            return null;
        }
        // 判断Key是否为16位
        if (sKey.length() != 16) {
            return null;
        }
        try {
            byte[] raw = sKey.getBytes("utf-8");
            SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");// "算法/模式/补码方式"
            cipher.init(Cipher.ENCRYPT_MODE, skeySpec);
            byte[] encrypted = cipher.doFinal(sSrc.getBytes("utf-8"));
            return Base64.encodeBase64String(encrypted);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    public String decrypt(String sSrc, String sKey) {
        // 判断Key是否正确
        if (sKey == null) {
            return null;
        }
        // 判断Key是否为16位
        if (sKey.length() != 16) {
            return null;
        }
        try {
            byte[] raw = sKey.getBytes("utf-8");
            SecretKeySpec skeySpec = new SecretKeySpec(raw, "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(Cipher.DECRYPT_MODE, skeySpec);
            byte[] encrypted = Base64.decodeBase64(sSrc);
            byte[] original = cipher.doFinal(encrypted);
            return new String(original, "utf-8");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }


    private SSLConnectionSocketFactory createSSLConnSocketFactory() {
        SSLConnectionSocketFactory sslsf = null;
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {

                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }

            }).build();
            sslsf = new SSLConnectionSocketFactory(sslContext, new X509HostnameVerifier() {

                @Override
                public boolean verify(String arg0, SSLSession arg1) {
                    return true;
                }

                @Override
                public void verify(String host, SSLSocket ssl) throws IOException {
                }

                @Override
                public void verify(String host, X509Certificate cert) throws SSLException {
                }

                @Override
                public void verify(String host, String[] cns, String[] subjectAlts) throws SSLException {
                }

            });
        } catch (GeneralSecurityException e) {
            log.error(e);
        }
        return sslsf;
    }



    public static class RiskServiceClientBuilder{

        private Environment environment;
        private ProxyModel proxyModel;
        private int  connectTimeout = 1000;
        private int readTimeout = 1000;
        private int maxConnection = 50;
        private String charset;
        private boolean heartbeatSwitch;
        private String  appDataSecrectKey;
        private String partnerCode;
        private String sslCertPath;

        /**
         * @param partnerCode 合作方
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder partnerCode(String partnerCode){
            this.partnerCode = partnerCode;
            return this;
        }

        /**
         * @param  environment 对接环境
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder environment(Environment environment){
            this.environment = environment;
            return this;
        }

        /**
         * @param proxyModel http代理
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder proxyModel(ProxyModel proxyModel){
            this.proxyModel = proxyModel;
            return this;
        }

        /**
         * @param connectTimeout 连接超时时间
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder connectTimeout(int connectTimeout){
            this.connectTimeout = connectTimeout;
            return this;
        }

        /**
         * @param  readTimeout 读取超时时间
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder readTimeout(int readTimeout){
            this.readTimeout = readTimeout;
            return this;
        }

        /**
         * @param  maxConnection 最大连接数
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder maxConnection(int maxConnection){
            this.maxConnection = maxConnection;
            return this;
        }

        /**
         * @param charset 请求编码
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder charset(String charset){
            this.charset = charset;
            return this;
        }

        /**
         * @param appDataSecrectKey 加密秘钥
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder appDataSecrectKey(String appDataSecrectKey){
            this.appDataSecrectKey = appDataSecrectKey;
            return this;
        }

        /**
         * @param  heartbeatSwitch 是否保持心跳
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder heartbeatSwitch(boolean heartbeatSwitch){
            this.heartbeatSwitch = heartbeatSwitch;
            return this;
        }

        /**
         * @param sslCertPath 证书位置
         * @return RiskServiceClientBuilder 构造器
         */
        public RiskServiceClientBuilder sslCertPath(String sslCertPath){
            this.sslCertPath = sslCertPath;
            return this;
        }

        /**
         * 构造RiskServiceClient示例
         * @return RiskServiceClient RiskServiceClient实例
         */
        public RiskServiceClient build(){
            if(sslCertPath != null){
                instance.sslCertPath = sslCertPath;
            }
            instance.appDataSecrectKey = this.appDataSecrectKey;
            instance.init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset, proxyModel, heartbeatSwitch);
            return instance;
        }

    }

}
