/*
 * Copyright 2015 FraudMetrix.cn All right reserved. This software is the
 * confidential and proprietary information of FraudMetrix.cn ("Confidential
 * Information"). You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license agreement you entered
 * into with FraudMetrix.cn.
 */
package cn.fraudmetrix.riskservice;

import cn.fraudmetrix.riskservice.object.Environment;
import cn.fraudmetrix.riskservice.ruledetail.parse.Format;
import cn.fraudmetrix.riskservice.ruledetail.parse.PublicFormatProvider;
import com.alibaba.fastjson.JSON;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.Asserts;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;

/**
 * <p>
 * 规则详情接口客户端，先调用getInstance获取唯一实例，然后调用实例的execute方法访问风险决策接口。
 * </p>
 * <p>
 * 特别注意：
 * </p>
 * <p>
 * 多次调用getInstance获取的是同一实例，但只有第一次调用会按照给定参数进行初始化。
 * </p>
 * <p>
 * 第二次及之后调用时传入的参数将不会产生任何效果。
 * </p>
 *
 * @author peichao.cai@fraudmetrix.cn
 */
public class RuleDetailClient extends Observable {

    private static final Log              log             = LogFactory.getLog(RuleDetailClient.class);

    /**
     * 风险决策接口地址
     */
    private String                        apiUrl;

    /**
     * URLencode编码默认使用的字符集
     */
    private static final String           DEFAULT_CHARSET = "utf-8";

    /**
     * 合作方代码
     */
    private String                        partnerCode;

    /**
     * 参数URLencode编码所使用的字符集
     */
    private String                        charset;

    private CloseableHttpClient           httpClient;
    private static final RuleDetailClient instance        = new RuleDetailClient();
    private static boolean                inited;

    private RuleDetailClient(){
    }

    private synchronized void init(String partnerCode, Environment environment, int connectTimeout, int readTimeout,
                                   int maxConnection, String charset) {
        // 确保只初始化一次
        if (!inited) {
            synchronized (instance) {
                if (!inited) {
                    inited = true;
                }
            }
        }

        // 检查参数
        Asserts.notBlank(partnerCode, "partnerCode");
        Asserts.check(connectTimeout >= 500, "connectTimeout must >= 500ms.");
        Asserts.check(readTimeout >= 500, "readTimeout must >= 500ms.");
        Asserts.check(maxConnection > 0, "maxConnection must > 0.");

        // 保存参数
        if (null == environment) environment = Environment.PRODUCT;
        this.apiUrl = environment.getRuleDetailUrl();
        this.partnerCode = partnerCode;
        this.charset = null == charset ? DEFAULT_CHARSET : charset;

        // 构建HttpClient
        PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager();
        connMgr.setMaxTotal(maxConnection);
        connMgr.setDefaultMaxPerRoute(maxConnection);
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        configBuilder.setConnectTimeout(connectTimeout);
        configBuilder.setSocketTimeout(readTimeout);
        configBuilder.setConnectionRequestTimeout(500);
        configBuilder.setStaleConnectionCheckEnabled(true);
        RequestConfig requestConfig = configBuilder.build();

        // 使用自定义的证书校验逻辑
        httpClient = HttpClients.custom().setSSLSocketFactory(createSSLConnSocketFactory()).setConnectionManager(connMgr).setDefaultRequestConfig(requestConfig).build();

        // 保持心跳
        Thread heartbeatThread = new Thread(new Runnable() {

            @Override
            public void run() {
                while (true) {
                    RuleDetailClient.this.executeHeartbeat();
                    try {
                        Thread.sleep(3000);
                    } catch (InterruptedException e) {
                        log.error(e);
                        break;
                    }
                }
            }
        }, "RuleDetailClient Heartbeat Thread");
        heartbeatThread.setDaemon(true);
        heartbeatThread.start();
    }

    public static RuleDetailClient getInstance(String partnerCode) {
        instance.init(partnerCode, Environment.PRODUCT, 1000, 1000, 50, "utf8");
        return instance;
    }

    public static RuleDetailClient getInstance(String partnerCode, Environment environment) {
        instance.init(partnerCode, environment, 1000, 1000, 50, "utf8");
        return instance;
    }

    public static RuleDetailClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                               int readTimeout) {
        instance.init(partnerCode, environment, connectTimeout, readTimeout, 50, "utf8");
        return instance;
    }

    public static RuleDetailClient getInstance(String partnerCode, Environment environment, int connectTimeout,
                                               int readTimeout, int maxConnection, String charset) {
        instance.init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset);
        return instance;
    }

    private String buildQueryString(final Map<String, String> params) {
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
        if (null != params) {
            for (String key : params.keySet()) {
                nameValuePairs.add(new BasicNameValuePair(key, params.get(key)));
            }
        }
        return URLEncodedUtils.format(nameValuePairs, Charset.forName(this.charset));
    }

    public void executeHeartbeat() {
        Map<String, String> params = new HashMap<String, String>();
        params.put("partner_code", "heartbeat");
        params.put("partner_key", "heartheat");
        try {
            postException(params);
        } catch (Exception ex) {
            // 忽略任何错误
        }
    }

    /**
     * 调用风险决策服务，获取决策结果
     * 
     * @param partnerKey 合作方密钥
     * @param sequenceId SequenceID，通过riskService接口可以获得
     * @return 风险决策结果
     */
    public RuleDetailResult execute(final String partnerKey, final String sequenceId) {
        // 检查参数
        if (partnerKey == null) throw new RuntimeException("partnerKey必填");
        if (sequenceId == null) throw new RuntimeException("sequenceId必填");

        // 参数检查通过，拼接参数Map
        Map<String, String> params = new HashMap<String, String>();
        params.put("partner_code", partnerCode);
        params.put("partner_key", partnerKey);
        params.put("sequence_id", sequenceId);

        // 调用同盾的接口，如果出错，则返回success=false的bean
        String result = post(params);
        if (result == null) return error("错误1");
        if (result.isEmpty()) return error("错误2");

        // 调用成功则进行解析，解析失败则返回错误
        JSON json;
        try {
            json = JSON.parseObject(result);
            if (json == null) return error("错误3");
        } catch (Exception ex) {
            log.warn("Failed to parse json", ex);
            return error("错误4");
        }

        // json解析成功，然后解析详情格式，解析失败则返回错误
        try {
            Format format = PublicFormatProvider.getInstance().getFormat(RuleDetailResult.class);
            return (RuleDetailResult) format.decode(json);
        } catch (Exception ex) {
            log.warn("Failed to decode RuleDetail");
            return error("错误5");
        }
    }

    private RuleDetailResult error(String message) {
        RuleDetailResult result = new RuleDetailResult();
        result.setSuccess(false);
        result.setReasonCode("000");
        result.setReasonDesc(message);
        return result;
    }

    private String post(Map<String, String> params) {
        try {
            String result = postException(params);
            log.debug("execute result: " + result);
            return result;
        } catch (Exception e) {
            log.warn("Failed to execute http post", e);
            return null;
        }
    }

    private String postException(Map<String, String> params) throws Exception {
        String queryString = buildQueryString(params);
        HttpGet httpGet = new HttpGet(apiUrl + "?" + queryString);
        CloseableHttpResponse response = null;
        HttpEntity entity;
        try {
            response = httpClient.execute(httpGet);

            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                throw new Exception("execute failed, response status: " + statusCode);
            }
            entity = response.getEntity();
            if (entity == null) {
                throw new Exception("execute failed, response output is null!");
            }
            String result = EntityUtils.toString(entity, "utf-8");
            result = result.trim();
            return result;
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {
                }
            }
        }
    }

    private SSLConnectionSocketFactory createSSLConnSocketFactory() {
        SSLConnectionSocketFactory sslsf = null;
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {

                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }

            }).build();
            sslsf = new SSLConnectionSocketFactory(sslContext, new X509HostnameVerifier() {

                @Override
                public boolean verify(String arg0, SSLSession arg1) {
                    return true;
                }

                @Override
                public void verify(String host, SSLSocket ssl) throws IOException {
                }

                @Override
                public void verify(String host, X509Certificate cert) throws SSLException {
                }

                @Override
                public void verify(String host, String[] cns, String[] subjectAlts) throws SSLException {
                }

            });
        } catch (GeneralSecurityException e) {
            log.error(e);
        }
        return sslsf;
    }

}
