/*
 * Copyright 2015 FraudMetrix.cn All right reserved. This software is the
 * confidential and proprietary information of FraudMetrix.cn ("Confidential
 * Information"). You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license agreement you entered
 * into with FraudMetrix.cn.
 */
package cn.fraudmetrix.riskservice;

import cn.fraudmetrix.riskservice.object.Environment;
import cn.fraudmetrix.riskservice.object.RiskResult;
import com.alibaba.fastjson.JSON;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.Asserts;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


/**
 * <p>风险决策服务客户端，先调用getInstance获取唯一实例，然后调用实例的execute方法访问风险决策接口。</p>
 *
 * <p>特别注意：</p>
 *      多次调用getInstance获取的是同一实例，但只有第一次调用会按照给定参数进行初始化。<br/>
 *      第二次及之后调用时传入的参数将不会产生任何效果。<br/>
 *
 * @author ming.ma@fruadmetrix.cn
 */
public class RiskServiceClient {

    private static final Log log = LogFactory.getLog(RiskServiceClient.class);

    /**
     * 风险决策接口地址
     */
    private String apiUrl;

    /**
     * URLencode编码默认使用的字符集
     */
    private static final String DEFAULT_CHARSET = "utf-8";

    /**
     * 合作方代码
     */
    private NameValuePair partnerCode;

    /**
     * 参数URLencode编码所使用的字符集
     */
    private String charset;

    private CloseableHttpClient httpClient;
    private static final RiskServiceClient instance = new RiskServiceClient();
    private static boolean inited;

    private RiskServiceClient() {}

    private void init(String partnerCode, Environment environment, int connectTimeout, int readTimeout, int maxConnection, String charset) {
        synchronized (instance) {
            if(inited) {
//                throw new IllegalStateException("You've already got a client instance, just use it. DO NOT call RiskServiceClient#getInstance() again!");
                return;
            }
            inited = true;
        }
        Asserts.notBlank(partnerCode, "partnerCode");
        Asserts.check(connectTimeout >= 500, "connectTimeout must >= 500ms.");
        Asserts.check(readTimeout >= 500, "readTimeout must >= 500ms.");
        Asserts.check(maxConnection > 0, "maxConnection must > 0.");

        if(null == environment) environment = Environment.PRODUCT;
        this.apiUrl = environment.getApiUrl();

        this.partnerCode = new BasicNameValuePair("partner_code", partnerCode);
        this.charset = null == charset ? DEFAULT_CHARSET : charset;
        // 设置连接池
        PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager();
        // 设置连接池大小
        connMgr.setMaxTotal(maxConnection);
        connMgr.setDefaultMaxPerRoute(maxConnection);

        // 设置请求配置
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        // 设置连接超时
        configBuilder.setConnectTimeout(connectTimeout);
        // 设置读取超时
        configBuilder.setSocketTimeout(readTimeout);
        // 设置从连接池获取连接实例的超时
        configBuilder.setConnectionRequestTimeout(500);
        // 在提交请求之前 测试连接是否可用
        configBuilder.setStaleConnectionCheckEnabled(true);
        RequestConfig requestConfig = configBuilder.build();
        httpClient = HttpClients
                .custom()
                .setSSLSocketFactory(createSSLConnSocketFactory())
                .setConnectionManager(connMgr)
                .setDefaultRequestConfig(requestConfig)
                .build();
        new Thread(new Runnable() {
            @Override
            public void run() {
                while(true) {
                    RiskServiceClient.this.execute("heartbeat", "heartbeat", null);
                    try {
                        Thread.sleep(3000);
                    } catch (InterruptedException e) {
                        log.error(e);
                        break;
                    }
                }
            }
        }, "RiskServiceClient Heartbeat Thread").start();
    }

    /**
     * 获取RiskServiceClient实例
     * 默认配置使用生产环境API地址，url编码使用utf-8字符集
     * connectTimeout及readTimeout均为1000ms，最大并发连接50个
     *
     * @param partnerCode 合作方代码
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode) {
        return getInstance(partnerCode, null);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment) {
        return getInstance(partnerCode, environment, null);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, String charset) {
        return getInstance(partnerCode, environment, 1000, 1000, 50, charset);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout, int readTimeout) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, null);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout, int readTimeout, String charset) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, 50, charset);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout, int readTimeout, int maxConnection) {
        return getInstance(partnerCode, environment, connectTimeout, readTimeout, maxConnection, null);
    }

    /**
     * 获取RiskServiceClient实例
     * @param partnerCode 合作方代码
     * @param environment 环境，缺省值{@link cn.fraudmetrix.riskservice.object.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @return 默认RiskServiceClient实例
     */
    public static RiskServiceClient getInstance(String partnerCode, Environment environment, int connectTimeout, int readTimeout, int maxConnection, String charset) {
        instance.init(partnerCode, environment, connectTimeout, readTimeout, maxConnection, charset);
        return instance;
    }

    private HttpEntity buildRequestEntity(final String secretKey, final String eventId, final Map<String, String> params) {
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
        nameValuePairs.add(partnerCode);
        nameValuePairs.add(new BasicNameValuePair("secret_key", secretKey));
        nameValuePairs.add(new BasicNameValuePair("event_id", eventId));
        if(null != params) {
            for (String key : params.keySet()) {
                nameValuePairs.add(new BasicNameValuePair(key, params.get(key)));
            }
        }
        return new UrlEncodedFormEntity(nameValuePairs, Charset.forName(this.charset));
    }

    /**
     * 调用风险决策服务，获取决策结果
     * @param secretKey 应用密钥
     * @param eventId 事件ID
     * @param params 业务参数
     * @return 风险决策结果
     */
    public RiskResult execute(final String secretKey, final String eventId, final Map<String, String> params){
        RiskResult riskResult = null;
        HttpPost httpPost = new HttpPost(apiUrl);
        CloseableHttpResponse response = null;
        HttpEntity entity;
        try {
            httpPost.setEntity(buildRequestEntity(secretKey, eventId, params));
            response = httpClient.execute(httpPost);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                log.warn("execute failed, response status: " + statusCode);
                return null;
            }
            entity = response.getEntity();
            if (entity == null) {
                log.warn("execute failed, response output is null!");
                return null;
            }
            String result = EntityUtils.toString(entity, "utf-8");
            log.debug("execute result: " + result);
            riskResult = JSON.parseObject(result, RiskResult.class);
        } catch (Exception e) {
            log.error("execute throw exception, details: ", e);
            riskResult = RiskResult.failedWithReason("000:" + e.toString());
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {}
            }
        }
        return riskResult;
    }

    private SSLConnectionSocketFactory createSSLConnSocketFactory() {
        SSLConnectionSocketFactory sslsf = null;
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {

                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }

            }).build();
            sslsf = new SSLConnectionSocketFactory(sslContext, new X509HostnameVerifier() {

                @Override
                public boolean verify(String arg0, SSLSession arg1) {
                    return true;
                }

                @Override
                public void verify(String host, SSLSocket ssl) throws IOException {
                }

                @Override
                public void verify(String host, X509Certificate cert) throws SSLException {
                }

                @Override
                public void verify(String host, String[] cns, String[] subjectAlts) throws SSLException {
                }

            });
        } catch (GeneralSecurityException e) {
            log.error(e);
        }
        return sslsf;
    }

}
