/*
 * Copyright 2015 FraudMetrix.cn All right reserved. This software is the
 * confidential and proprietary information of FraudMetrix.cn ("Confidential
 * Information"). You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license agreement you entered
 * into with FraudMetrix.cn.
 */
package cn.fraudmetrix.cloudservice;

import cn.fraudmetrix.cloudservice.annotation.method.HttpMethod;
import cn.fraudmetrix.cloudservice.constant.Environment;
import cn.fraudmetrix.cloudservice.constant.ReasonCode;
import cn.fraudmetrix.cloudservice.request.Request;
import cn.fraudmetrix.cloudservice.response.Response;
import cn.fraudmetrix.cloudservice.utils.BeanUtils;
import cn.fraudmetrix.cloudservice.utils.HttpUtils;
import com.alibaba.fastjson.JSON;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.util.Asserts;
import org.apache.http.util.EntityUtils;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import java.io.IOException;
import java.net.URI;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;

/**
 * <p>
 * 同盾云服务客户端，先调用getInstance获取唯一实例，然后调用实例的execute方法访问云服务接口。
 * </p>
 * <p>
 * 特别注意：
 * </p>
 * 多次调用getInstance获取的是同一实例，但只有第一次调用会按照给定参数进行初始化。<br/>
 * 第二次及之后调用时传入的参数将不会产生任何效果。<br/>
 *
 * @author jianhao.dai@fraudmetrix.cn
 */
public class CloudServiceClient {

    private static final Log                logger          = LogFactory.getLog(CloudServiceClient.class);

    /**
     * 云服务接口地址
     */
    private String                          apiUrl;

    /**
     * URLencode编码默认使用的字符集
     */
    private static final String             DEFAULT_CHARSET = "utf-8";

    /**
     * 合作方标识
     */
    private String                          partnerCode;
    /**
     * 合作方密钥
     */
    private String                          partnerKey;

    /**
     * 参数URLencode编码所使用的字符集
     */
    private String                          charset;

    private CloseableHttpClient             httpClient;
    private static final CloudServiceClient instance        = new CloudServiceClient();
    private static boolean                  inited;

    private CloudServiceClient(){
    }

    private void init(String partnerCode, String partnerKey, Environment environment, int connectTimeout,
                      int readTimeout, int maxConnection, String charset) {
        synchronized (instance) {
            if (inited) {
                // throw new
                // IllegalStateException("You've already got a client instance, just use it. DO NOT call CloudServiceClient#getInstance() again!");
                return;
            }
            inited = true;
        }
        Asserts.notBlank(partnerCode, "partnerCode");
        Asserts.notBlank(partnerKey, "partnerCode");
        Asserts.check(connectTimeout >= 500, "connectTimeout must >= 500ms.");
        Asserts.check(readTimeout >= 500, "readTimeout must >= 500ms.");
        Asserts.check(maxConnection > 0, "maxConnection must > 0.");

        if (null == environment) environment = Environment.PRODUCT;
        this.apiUrl = environment.getApiUrl();

        this.partnerCode = partnerCode;
        this.partnerKey = partnerKey;
        this.charset = null == charset ? DEFAULT_CHARSET : charset;
        // 设置连接池
        PoolingHttpClientConnectionManager connMgr = new PoolingHttpClientConnectionManager();
        // 设置连接池大小
        connMgr.setMaxTotal(maxConnection);
        connMgr.setDefaultMaxPerRoute(maxConnection);

        // 设置请求配置
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        // 设置连接超时
        configBuilder.setConnectTimeout(connectTimeout);
        // 设置读取超时
        configBuilder.setSocketTimeout(readTimeout);
        // 设置从连接池获取连接实例的超时
        configBuilder.setConnectionRequestTimeout(500);
        // 在提交请求之前 测试连接是否可用
        configBuilder.setStaleConnectionCheckEnabled(true);
        RequestConfig requestConfig = configBuilder.build();
        httpClient = HttpClients.custom().setSSLSocketFactory(createSSLConnSocketFactory()).setConnectionManager(connMgr).setDefaultRequestConfig(requestConfig).build();
    }

    /**
     * 获取CloudServiceClient实例 默认配置使用生产环境API地址，url编码使用utf-8字符集 connectTimeout及readTimeout均为1000ms，最大并发连接50个
     *
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey) {
        return getInstance(partnerCode, partnerKey, null);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment) {
        return getInstance(partnerCode, partnerKey, environment, null);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @param charset urlencode使用的字符集
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment,
                                                 String charset) {
        return getInstance(partnerCode, partnerKey, environment, 1000, 1000, 50, charset);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment,
                                                 int connectTimeout, int readTimeout) {
        return getInstance(partnerCode, partnerKey, environment, connectTimeout, readTimeout, null);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param charset urlencode使用的字符集
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment,
                                                 int connectTimeout, int readTimeout, String charset) {
        return getInstance(partnerCode, partnerKey, environment, connectTimeout, readTimeout, 50, charset);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment,
                                                 int connectTimeout, int readTimeout, int maxConnection) {
        return getInstance(partnerCode, partnerKey, environment, connectTimeout, readTimeout, maxConnection, null);
    }

    /**
     * 获取CloudServiceClient实例
     * 
     * @param partnerCode 合作方代码
     * @param partnerKey 合作方密钥
     * @param environment 环境，缺省值{@link cn.fraudmetrix.cloudservice.constant.Environment#PRODUCT}
     * @param connectTimeout 连接超时 缺省值1000ms
     * @param readTimeout 读取超时 缺省值1000ms
     * @param maxConnection 最大并发连接数 缺省值50
     * @param charset urlencode使用的字符集
     * @return 默认CloudServiceClient实例
     */
    public static CloudServiceClient getInstance(String partnerCode, String partnerKey, Environment environment,
                                                 int connectTimeout, int readTimeout, int maxConnection, String charset) {
        instance.init(partnerCode, partnerKey, environment, connectTimeout, readTimeout, maxConnection, charset);
        return instance;
    }

    private <T extends Response> T getFailureResponse(Class<T> clazz, ReasonCode reasonCode) {
        try {
            T response = clazz.newInstance();
            response.setSuccess(Boolean.FALSE);
            response.setReasonCode(reasonCode.getCode());
            response.setReasonDesc(reasonCode.getDesc());
            return response;
        } catch (IllegalAccessException | InstantiationException e) {
            return null;
        }
    }

    /**
     * 调用云服务服务，获取请求结果
     * 
     * @param request 请求参数
     * @param clazz 泛型类型
     * @param <T> 泛型
     * @return 云服务结果
     */
    public <T extends Response> T execute(Request request, Class<T> clazz) {
        request.setPartnerCode(partnerCode);
        request.setPartnerKey(partnerKey);

        Map<BeanUtils.AnnotationValue, Object> params = BeanUtils.parseParam(request);
        HttpUtils.HttpParam httpParam = HttpUtils.diversion(params);

        String path = BeanUtils.parsePath(request);
        URI uri = HttpUtils.buildRequestUri(apiUrl, path, httpParam);

        HttpMethod method = BeanUtils.parseMethod(request);
        switch (method.value()) {
            case HttpMethod.GET:
                return doHttpRequest(new HttpGet(uri), httpParam, clazz);
            case HttpMethod.DELETE:
                return doHttpRequest(new HttpDelete(uri), httpParam, clazz);
            case HttpMethod.POST:
                return doHttpRequestWithEntity(new HttpPost(uri), httpParam, clazz);
            case HttpMethod.PUT:
                return doHttpRequestWithEntity(new HttpPut(uri), httpParam, clazz);
            default:
                return getFailureResponse(clazz, ReasonCode.METHOD_NOT_ALLOWED);
        }
    }

    public <T extends Response> T doHttpRequestWithEntity(HttpEntityEnclosingRequestBase request,
                                                          HttpUtils.HttpParam httpParam, Class<T> clazz) {
        request.setEntity(HttpUtils.buildRequestEntity(httpParam, this.charset));
        return doHttpRequest(request, httpParam, clazz);
    }

    public <T extends Response> T doHttpRequest(HttpUriRequest request, HttpUtils.HttpParam httpParam, Class<T> clazz) {
        request.setHeaders(HttpUtils.buildRequestHeader(httpParam));
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(request);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK) {
                throw new HttpException("execute failed, response status: " + statusCode);
            }
            HttpEntity entity = response.getEntity();
            if (entity == null) {
                throw new HttpException("execute failed, response output is null!");
            }
            String result = EntityUtils.toString(entity, "utf-8");
            return JSON.parseObject(result, clazz);
        } catch (IOException | HttpException e) {
            logger.error("execute throw exception, details: ", e);
            return getFailureResponse(clazz, ReasonCode.AUTH_ERROR);
        } finally {
            if (response != null) {
                try {
                    EntityUtils.consume(response.getEntity());
                } catch (IOException ignored) {
                }
            }
        }
    }

    private SSLConnectionSocketFactory createSSLConnSocketFactory() {
        SSLConnectionSocketFactory sslsf = null;
        try {
            SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustStrategy() {

                public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                    return true;
                }

            }).build();
            sslsf = new SSLConnectionSocketFactory(sslContext, new X509HostnameVerifier() {

                @Override
                public boolean verify(String arg0, SSLSession arg1) {
                    return true;
                }

                @Override
                public void verify(String host, SSLSocket ssl) throws IOException {
                }

                @Override
                public void verify(String host, X509Certificate cert) throws SSLException {
                }

                @Override
                public void verify(String host, String[] cns, String[] subjectAlts) throws SSLException {
                }

            });
        } catch (GeneralSecurityException e) {
            logger.error(e);
        }
        return sslsf;
    }

}
