/* ----------------------------------------------------------------------------
 * Copyright (c) Guangzhou Fox-Tech Co., Ltd. 2020-2024. All rights reserved.
 *
 *     This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 * --------------------------------------------------------------------------- */

package cn.foxtech.common.utils.serialport;

import cn.foxtech.common.utils.serialport.linux.SerialPortLinux;
import cn.foxtech.common.utils.serialport.win32.SerialPortWin32;
import com.sun.jna.Platform;

/**
 * 串口接口：方便跨平台的各自串口类实现
 */
public interface ISerialPort {
    static ISerialPort newInstance() {
        if (Platform.isWindows()) {
            return new SerialPortWin32();
        } else if (Platform.isLinux()) {
            return new SerialPortLinux();
        } else {
            return null;
        }
    }

    public String getName();

    /**
     * 串口是否打开
     *
     * @return 是否打开
     */
    boolean isOpen();

    /**
     * 打开串口
     *
     * @param name LINUX下串口名为ttyS0这样格式的数据，WINDOWS下串口名为COM1这样的格式
     * @return 操作是否成功
     */
    boolean open(String name);


    /**
     * 设置串口参数
     *
     * @param baudRate 速率
     * @param databits 数据位
     * @param stopbits 停止位
     * @param parity   校验位
     * @return 是否成功
     */
    boolean setParam(Integer baudRate, String parity, Integer databits, Integer stopbits);

    /**
     * 发送数据
     *
     * @param data    缓冲区
     * @return 实际发送的数据长度
     */
    int sendData(byte[] data);

    /**
     * 接收数据
     *
     * @param data     准备发送的数据库
     * @param mTimeout 最大超时等待时间，单位毫秒
     * @return 接收到数据
     */
    int recvData(byte[] data, long mTimeout);

    /**
     * 异步模式需要的单纯读数据
     * @param readBuffer 缓存
     * @return 返回的数据
     */
    public int readData(byte[] readBuffer);

    /**
     * 读取串口数据
     *
     * @param recvBuffer      缓存
     * @param minPackInterval 两组数据报文之间，最小的时间间隔
     * @param maxPackInterval 两组数据报文之间，最大的时间间隔
     * @return 报文长度
     */
    public int recvData(byte[] recvBuffer, long minPackInterval, long maxPackInterval);

    /**
     * 清空缓冲区
     *
     * @return 操作是否成功
     */
    boolean clearRecvFlush();

    /**
     * 清空缓冲区
     *
     * @return 操作是否成功
     */
    boolean clearSendFlush();

    /**
     * 关闭串口
     */
    boolean close();
}
