package io.polaris.framework;

import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;

import io.polaris.framework.core.consts.PropKeys;
import io.polaris.framework.core.context.AppCtx;
import io.polaris.framework.core.context.env.properties.Rest;
import io.polaris.framework.http.restfull.CommonRestController;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.servlet.ServletComponentScan;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.client.DefaultResponseErrorHandler;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartResolver;

/**
 * @author Qt
 * @since Jun 20, 2024
 */
@Slf4j
@Configuration
@AutoConfigureAfter(BasicComponentAutoConfiguration.class)
@ServletComponentScan(
	basePackageClasses = {
		BasePackage.class
	}
)
@Import({
})
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
public class CommonRestAutoConfiguration {

	@Bean
	@ConditionalOnProperty(name = PropKeys.REST_ENABLED, havingValue = "true", matchIfMissing = true)
	public CommonRestController commonRest(MultipartResolver multipartResolver) {
		return new CommonRestController(multipartResolver);
	}


	@Bean
	@ConditionalOnMissingBean
	public RestTemplate restTemplate() {
		StringHttpMessageConverter c = new StringHttpMessageConverter(StandardCharsets.UTF_8);
		RestTemplate restTemplate = new RestTemplate();
		restTemplate.getMessageConverters().set(1, c);
		restTemplate = new RestTemplateBuilder().additionalMessageConverters(restTemplate.getMessageConverters()).uriTemplateHandler(restTemplate.getUriTemplateHandler()).build();
		restTemplate.setRequestFactory(clientHttpRequestFactory());
		List<ClientHttpRequestInterceptor> interceptors = new ArrayList<>(restTemplate.getInterceptors());
		restTemplate.setInterceptors(interceptors);
		restTemplate.setErrorHandler(new DefaultResponseErrorHandler());
		return restTemplate;
	}

	@Bean
	@ConditionalOnMissingBean
	public HttpComponentsClientHttpRequestFactory clientHttpRequestFactory() {
		try {
			HttpClientBuilder httpClientBuilder = HttpClientBuilder.create();
			SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, (arg0, arg1) -> true).build();
			httpClientBuilder.setSSLContext(sslContext);
			HostnameVerifier hostnameVerifier = NoopHostnameVerifier.INSTANCE;
			SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContext, hostnameVerifier);
			Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
				.register("http", PlainConnectionSocketFactory.getSocketFactory())
				.register("https", sslConnectionSocketFactory).build();// 注册http和https请求
			// 开始设置连接池

			Rest.Template template = AppCtx.getPlatformProperties().getRest().getTemplate();

			PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);
			connectionManager.setMaxTotal(template.getMaxTotal()); // 最大连接数500
			connectionManager.setDefaultMaxPerRoute(template.getDefaultMaxPerRoute()); // 同路由并发数100
			httpClientBuilder.setConnectionManager(connectionManager);
			//httpClientBuilder.setRetryHandler(new DefaultHttpRequestRetryHandler(3, true)); // 重试次数

			RequestConfig requestConfig = RequestConfig.custom()
				.setSocketTimeout(template.getSocketTimeout()) //服务器返回数据(response)的时间，超过抛出read timeout
				.setConnectTimeout(template.getConnectTimeout()) //连接上服务器(握手成功)的时间，超出抛出connect timeout
				.setConnectionRequestTimeout(template.getConnectionRequestTimeout()
				)//从连接池中获取连接的超时时间，超时间未拿到可用连接，会抛出org.apache.http.conn.ConnectionPoolTimeoutException: Timeout waiting for connection from pool
				.build();

			httpClientBuilder.setDefaultRequestConfig(requestConfig);

			return new HttpComponentsClientHttpRequestFactory(httpClientBuilder.build());
		} catch (KeyManagementException | NoSuchAlgorithmException | KeyStoreException e) {
			log.error("", e);
		}
		return null;
	}

}
