package cn.fntop.service;

import cn.fntop.config.CustomJacksonConverterFactory;
import cn.fntop.config.OpenApiProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.theokanning.openai.OpenAiError;
import com.theokanning.openai.OpenAiHttpException;
import io.reactivex.Single;
import lombok.Getter;
import lombok.Setter;
import lombok.SneakyThrows;
import okhttp3.ConnectionPool;
import okhttp3.Interceptor;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import retrofit2.HttpException;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;

import java.io.IOException;
import java.net.Proxy;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

public class OpenApiService implements OpenApiServiceSubject {

    protected static final String BASE_URL = "http://127.0.0.1:8080/";
    protected static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(10);
    protected static final ObjectMapper mapper = defaultObjectMapper();
    @Getter
    @Setter
    protected static OpenApi api;
    @Getter
    @Setter
    protected static ExecutorService executorService;
    @Getter
    @Setter
    protected static OpenApiServiceSubject openService;

    /**
     * 初始化实例并存入上下文
     * 默认方法
     *
     * @param properties
     * @param cls
     * @param subject
     * @return
     */
    @SneakyThrows
    public static OpenApiServiceSubject getInstance(OpenApiProperties properties, Class<? extends OpenApi> cls, Class<? extends OpenApiServiceSubject> subject) {
        return custom(properties, cls, subject, null, null, null, null);
    }

    /**
     * 自定义实例方法，上下文：{@link OpenApiContext}
     *
     * @param properties      @see {@link OpenApiProperties}
     * @param cls             implements {@link OpenApi}
     * @param subject         extends {@link OpenApiService}
     * @param timeout
     * @param interceptor     implements {@link Interceptor}
     * @param proxy           new {@link Proxy}(Proxy.Type.HTTP, new InetSocketAddress(config.getProxyDomain(), config.getProxyPort()))
     * @param executorService
     * @return
     */
    @SneakyThrows
    public static OpenApiServiceSubject custom(OpenApiProperties properties, Class<? extends OpenApi> cls, Class<? extends OpenApiServiceSubject> subject, Duration timeout, Interceptor interceptor, Proxy proxy, ExecutorService executorService) {
        if (OpenApiContext.get(subject) != null) {
            return OpenApiContext.get(subject);
        }
        ObjectMapper mapper = OpenApiService.defaultObjectMapper();
        OkHttpClient client = OpenApiService.defaultClient(timeout, interceptor, proxy).newBuilder().build();
        Retrofit retrofit1 = OpenApiService.defaultRetrofit(client, mapper, properties.getOpenGateway());
        OpenApiService.api = retrofit1.create(cls);
        OpenApiServiceSubject instance = subject.newInstance();
        OpenApiContext.set(subject, instance);
        OpenApiService.executorService = executorService;
        return instance;
    }

//    public OpenApiService(final OpenApi api) {
//        OpenApiService.api = api;
//        OpenApiService.executorService = null;
//    }
//    public OpenApiService(OpenApi api, final ExecutorService executorService) {
//        OpenApiService.api = api;
//        OpenApiService.executorService = executorService;
//    }

    public Object get(String suffix) {
        return execute(api.get(suffix));
    }

    public Object get(String suffix, String path) {
        return execute(api.get(suffix, path));
    }

    public Object getOnQuery(String suffix, String query) {
        return execute(api.get(suffix, query, true));
    }

    public Object post(String suffix, Object obj) {
        return execute(api.post(suffix, obj));
    }

    public Object post(String suffix, Object obj, boolean isFile, String folder, MultipartBody.Part... file) {

        if (isFile) {
            return execute(api.post(suffix, folder, file));
        }
        return execute(api.post(suffix, obj));
    }

    /**
     * Calls the Open AI api, returns the response, and parses error messages if the request fails
     */
    public static <T> T execute(Single<T> apiCall) {
        try {
            return apiCall.blockingGet();
        } catch (HttpException e) {
            try {
                if (e.response() == null || e.response().errorBody() == null) {
                    throw e;
                }
                String errorBody = e.response().errorBody().string();
                OpenAiError error = mapper.readValue(errorBody, OpenAiError.class);
                throw new OpenAiHttpException(error, e, e.code());
            } catch (IOException ex) {
                // couldn't parse OpenAI error
                throw e;
            }
        }
    }

    /**
     * Shuts down the OkHttp ExecutorService.
     * The default behaviour of OkHttp's ExecutorService (ConnectionPool)
     * is to shut down after an idle timeout of 60s.
     * Call OpenApiService method to shut down the ExecutorService immediately.
     */
    public void shutdownExecutor() {
        Objects.requireNonNull(OpenApiService.executorService, "executorService must be set in order to shut down");
        OpenApiService.executorService.shutdown();
    }

    public static ObjectMapper defaultObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        return mapper;
    }

    public static OkHttpClient defaultClient(Duration timeout) {
        return defaultClient(timeout, null, null);
    }

    /**
     * @param timeout     超时
     * @param interceptor 自定义拦截器
     * @param proxy       自定义代理 new {@link Proxy}({@link Proxy.Type}, new InetSocketAddress("你的代理ip", "代理端口"))
     * @return
     */
    public static OkHttpClient defaultClient(Duration timeout, Interceptor interceptor, Proxy proxy) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder().connectionPool(new ConnectionPool(5, 1, TimeUnit.SECONDS));
        if (timeout == null) {
            builder.readTimeout(DEFAULT_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS);
        } else {
            builder.readTimeout(timeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        if (interceptor != null) {
            builder.addInterceptor(interceptor);
        }
        builder.proxy(proxy);
        return builder.build();
    }

    public static Retrofit defaultRetrofit(OkHttpClient client, ObjectMapper mapper) {
        return new Retrofit.Builder().baseUrl(BASE_URL).client(client).addConverterFactory(CustomJacksonConverterFactory.create(mapper)).addCallAdapterFactory(RxJava2CallAdapterFactory.create()).build();
    }

    public static Retrofit defaultRetrofit(OkHttpClient client, ObjectMapper mapper, String gateway) {
        if (gateway == null) {
            try {
                throw new Exception("网关地址不能为空");
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        Retrofit.Builder builder = new Retrofit.Builder().client(client).baseUrl(gateway).addConverterFactory(CustomJacksonConverterFactory.create(mapper)).addCallAdapterFactory(RxJava2CallAdapterFactory.create());
        return builder.build();
    }

}
