package cn.fntop.service;

import cn.fntop.config.CustomJacksonConverterFactory;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.theokanning.openai.OpenAiError;
import com.theokanning.openai.OpenAiHttpException;
import io.reactivex.Single;
import lombok.SneakyThrows;
import okhttp3.ConnectionPool;
import okhttp3.Interceptor;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import retrofit2.HttpException;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.TimeUnit;

public class OpenApiService implements OpenApiServiceSubject {

    protected static final String BASE_URL = "http://127.0.0.1:8080/";
    protected static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(10);
    protected static final ObjectMapper mapper = defaultObjectMapper();

    protected static OpenApi api;
    //    protected final ExecutorService executorService;
    protected volatile static OpenApiServiceSubject INSTANCE;

    /**
     * 双检查锁单例
     *
     * @param gateway
     * @return
     */
    @SneakyThrows
    public static OpenApiServiceSubject getInstance(String gateway, Class<? extends OpenApi> cls, Class<? extends OpenApiServiceSubject> subject) {
        if (INSTANCE == null) {
            synchronized (OpenApiService.class) {
                if (INSTANCE == null) {
                    ObjectMapper mapper = OpenApiService.defaultObjectMapper();
                    OkHttpClient client = OpenApiService.defaultClient(Duration.ofMinutes(1))
                            .newBuilder()
                            .build();
                    Retrofit retrofit1 = OpenApiService.defaultRetrofit(client, mapper, gateway);
                    OpenApi api = retrofit1.create(cls);
                    OpenApiService.api = api;
                    INSTANCE = subject.newInstance();
                }
            }
        }
        return INSTANCE;
    }

//    public OpenApiService(final OpenApi api) {
//        this.api = api;
//        this.executorService = null;
//    }
//    public OpenApiService(final OpenApi api, final ExecutorService executorService) {
//        this.api = api;
//        this.executorService = executorService;
//    }

    public Object get(String suffix) {
        return execute(api.get(suffix));
    }

    public Object get(String suffix, String path) {
        return execute(api.get(suffix, path));
    }

    public Object getOnQuery(String suffix, String query) {
        return execute(api.get(suffix, query, true));
    }

    public Object post(String suffix, Object obj) {
        return execute(api.post(suffix, obj));
    }

    public Object post(String suffix, Object obj, boolean isFile, String folder, MultipartBody.Part... file) {

        if (isFile) {
            return execute(api.post(suffix, folder, file));
        }
        return execute(api.post(suffix, obj));
    }

    /**
     * Calls the Open AI api, returns the response, and parses error messages if the request fails
     */
    public static <T> T execute(Single<T> apiCall) {
        try {
            return apiCall.blockingGet();
        } catch (HttpException e) {
            try {
                if (e.response() == null || e.response().errorBody() == null) {
                    throw e;
                }
                String errorBody = e.response().errorBody().string();
                OpenAiError error = mapper.readValue(errorBody, OpenAiError.class);
                throw new OpenAiHttpException(error, e, e.code());
            } catch (IOException ex) {
                // couldn't parse OpenAI error
                throw e;
            }
        }
    }

    /**
     * Shuts down the OkHttp ExecutorService.
     * The default behaviour of OkHttp's ExecutorService (ConnectionPool)
     * is to shut down after an idle timeout of 60s.
     * Call this method to shut down the ExecutorService immediately.
     */
//    public void shutdownExecutor() {
//        Objects.requireNonNull(this.executorService, "executorService must be set in order to shut down");
//        this.executorService.shutdown();
//    }


    public static ObjectMapper defaultObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        return mapper;
    }

    public static OkHttpClient defaultClient(Duration timeout) {
        return new OkHttpClient.Builder().connectionPool(new ConnectionPool(5, 1, TimeUnit.SECONDS)).readTimeout(timeout.toMillis(), TimeUnit.MILLISECONDS).build();
    }

    public static OkHttpClient defaultClient(Duration timeout, Interceptor interceptor) {
        return new OkHttpClient.Builder().addInterceptor(interceptor).connectionPool(new ConnectionPool(5, 1, TimeUnit.SECONDS)).readTimeout(timeout.toMillis(), TimeUnit.MILLISECONDS).build();
    }

    public static Retrofit defaultRetrofit(OkHttpClient client, ObjectMapper mapper) {
        return new Retrofit.Builder().baseUrl(BASE_URL).client(client).addConverterFactory(CustomJacksonConverterFactory.create(mapper)).addCallAdapterFactory(RxJava2CallAdapterFactory.create()).build();
    }

    public static Retrofit defaultRetrofit(OkHttpClient client, ObjectMapper mapper, String url) {
        return new Retrofit.Builder().baseUrl(url).client(client).addConverterFactory(CustomJacksonConverterFactory.create(mapper)).addCallAdapterFactory(RxJava2CallAdapterFactory.create()).build();
    }
}
