package com.smart.panorama;

import android.content.Context;
import android.graphics.BitmapFactory;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.google.gson.reflect.TypeToken;
import com.smart.panorama.callback.LoginCallback;
import com.smart.panorama.callback.OSSCallback;
import com.smart.panorama.callback.UploadCallback;
import com.smart.panorama.callback.WorkCallback;
import com.smart.panorama.model.BaseResult;
import com.smart.panorama.model.BaseWorkResult;
import com.smart.panorama.model.FileImage;
import com.smart.panorama.model.LoginInfo;
import com.smart.panorama.model.LoginRequest;
import com.smart.panorama.model.ProcessWork;
import com.smart.panorama.model.PublishWork;
import com.smart.panorama.model.QueryWork;
import com.smart.panorama.model.WorkInfo;
import com.smart.panorama.model.WorkList;
import com.smart.panorama.model.WorkRecord;
import com.smart.panorama.oss.OSSUploadUtils;
import com.smart.panorama.oss.UploadFile;
import com.smart.panorama.util.JsonUtils;
import com.smart.panorama.util.RedirectInterceptor;
import com.zhy.http.okhttp.OkHttpUtils;
import com.zhy.http.okhttp.callback.StringCallback;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;

public class PanoPublishUtils {

    private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static Handler handler = new Handler(Looper.getMainLooper());
    private static final String BASE_URL = "http://panorama.m2m88.com";
    private static final int maxRetry = 3;//最大重试次数
    private static int retryNum;//假如设置为3次重试的话，则最大可能请求4次（默认1次+3次重试）
    private static LoginInfo mLoginInfo;
    private static boolean isDealing;

    static {
        OkHttpClient okHttpClient = new OkHttpClient.Builder()
                .addInterceptor(new RedirectInterceptor())
                .connectTimeout(10000L, TimeUnit.MILLISECONDS)
                .readTimeout(10000L, TimeUnit.MILLISECONDS)
                //其他配置
                .build();

        OkHttpUtils.initClient(okHttpClient);
    }

    public static void oauthLogin(LoginRequest request, LoginCallback listener) {

        if (request == null) {
            if (listener != null) {
                listener.onFailed("参数不能为空");
            }
            return;
        }

        if (TextUtils.isEmpty(request.getAccessKey()) || TextUtils.isEmpty(request.getPhone())) {

            if (listener != null) {
                listener.onFailed("accessKey或者phone不能为空");
            }
            return;
        }


        Map<String, Object> map = new HashMap<>();
        map.put("accessKey", request.getAccessKey());
        map.put("phone", request.getPhone());
        map.put("name", request.getName());
        map.put("linkman", request.getLinkman());
        map.put("address", request.getAddress());


        OkHttpUtils
                .postString()
                .url("http://39.98.58.224:8080/panorama/api/v1/user/oauthLogin")
                .content(JsonUtils.toJson(map))
                .mediaType(MediaType.parse("application/json; charset=utf-8"))
                .build()
                .execute(new StringCallback() {
                    @Override
                    public void onError(Call call, Exception e, int id) {

                        if (listener != null) {
                            listener.onFailed(e.getMessage());
                        }
                    }

                    @Override
                    public void onResponse(String response, int id) {

                        BaseResult<LoginInfo> baseResult = JsonUtils.fromJson(response, new TypeToken<BaseResult<LoginInfo>>() {
                        }.getType());

                        if (baseResult == null) {
                            if (listener != null) {
                                listener.onFailed("系统错误");
                            }
                            return;
                        }

                        if (baseResult.isSuccess()) {

                            LoginInfo loginInfo = baseResult.getData();

                            if (loginInfo.getType() != 3) {
                                if (listener != null) {
                                    listener.onFailed("该账户用户类型不能在手机端登录");
                                }
                                return;
                            }

                            if (loginInfo.getStatus() == 0) {
                                if (listener != null) {
                                    listener.onFailed("账户状态不正常");
                                }
                                return;
                            }

                            try {
                                Date nowTime = new Date();
                                Date expireTime = sdf.parse(loginInfo.getExpire_time());

                                if (nowTime.after(expireTime)) {
                                    if (listener != null) {
                                        listener.onFailed("账号已过期，请续费后再上传全景！");
                                    }
                                } else {
                                    mLoginInfo = loginInfo;
                                    if (listener != null) {
                                        listener.onSuccess(loginInfo);
                                    }
                                }

                            } catch (Exception e) {
                                if (listener != null) {
                                    listener.onFailed(e.getMessage());
                                }
                            }
                        } else {

                            if (listener != null) {
                                listener.onFailed(baseResult.getMessage());
                            }

                        }
                    }
                });

    }

    public static void uploadPano(WorkInfo workInfo, List<String> uploadFiles, UploadCallback listener) {
        uploadPano(getContext(), workInfo, uploadFiles, listener);
    }

    public static void uploadPano(Context context, WorkInfo workInfo, List<String> uploadFiles, UploadCallback listener) {

        if (uploadFiles == null || uploadFiles.isEmpty()) {
            callFailed(listener, "全景图片路径列表不能为空");
            return;
        }

        List<FileImage> tmp = new ArrayList<>();

        for (String path : uploadFiles) {

            File file = new File(path);
            FileImage fileImage = new FileImage();
            fileImage.setFileUrl(path);
            fileImage.setPhotoName(getFilePrefix(file.getName()));
            tmp.add(fileImage);
        }

        uploadPano(context, tmp, workInfo, listener);

    }

    public static void uploadPano(Context context, List<FileImage> uploadFiles, WorkInfo workInfo, UploadCallback listener) {

        if (isDealing) {
            callFailed(listener, "请等待前面的作品上传处理完成");
            return;
        }

        if (mLoginInfo == null) {
            callFailed(listener, "请先登录");
            return;
        }


        if (workInfo == null) {
            callFailed(listener, "全景作品信息不能为空");
            return;
        }

        if (TextUtils.isEmpty(workInfo.getWorkName())) {
            callFailed(listener, "全景作品名称不能为空");
            return;
        }

        if (workInfo.getWorkName().length() > 32) {
            callFailed(listener, "全景作品名称不能大于32个字符");
            return;
        }

        if (uploadFiles == null || uploadFiles.isEmpty()) {
            callFailed(listener, "全景图片路径列表不能为空");
            return;
        }

        if (uploadFiles.size() > 16) {
            callFailed(listener, "一次最多上传16张全景图片");
            return;
        }

        isDealing = true;

        if (listener != null) {
            listener.onProgress("开始上传全景图片");
        }

        List<UploadFile> finalList = new ArrayList<>();

        for (FileImage fileImage : uploadFiles) {

            File file = new File(fileImage.getFileUrl());

            if (TextUtils.isEmpty(fileImage.getPhotoName())) {
                fileImage.setPhotoName(getFilePrefix(file.getName()));
            }

            if (!file.exists()) {
                callFailed(listener, "全景图片列表中有不存在的路径");
                return;
            }

            if (!file.getName().toLowerCase().endsWith(".jpg")) {

                callFailed(listener, "全景图片列表中有非全景图片");
                return;
            }

            BitmapFactory.Options opts = new BitmapFactory.Options();
            //只请求图片宽高，不解析图片像素(请求图片属性但不申请内存，解析bitmap对象，该对象不占内存)
            opts.inJustDecodeBounds = true;
            BitmapFactory.decodeFile(fileImage.getFileUrl(), opts);

            if (opts.outHeight == 0) {
                callFailed(listener, "请检查是否有读取存储权限");
                return;
            }

            if ((opts.outWidth / opts.outHeight) != 2) {
                callFailed(listener, "全景图片列表中存在非2:1的全景图片");
                return;
            }


            UploadFile uploadFile = new UploadFile();
            uploadFile.setLocalPath(fileImage.getFileUrl());
            uploadFile.setFileName(fileImage.getPhotoName());
            uploadFile.setFileSize(file.length() / 1024);
            uploadFile.setObjectKey(OSSUploadUtils.genObjectKey(String.valueOf(mLoginInfo.getId()), file));
            finalList.add(uploadFile);

        }


        OSSUploadUtils.getInstance(context)
                .setListener(new OSSCallback() {
                    @Override
                    public void onComplete(List<UploadFile> data) {

                        if (data.isEmpty()) {
                            callFailed(listener, "全景图片上传失败");
                            return;
                        }

                        retryNum = 0;
                        addPano(mLoginInfo.getToken(), workInfo, data, listener);

                    }

                    @Override
                    public void onFailed(String errorMsg) {

                        callFailed(listener, errorMsg);
                    }

                    @Override
                    public void onProgress(String message) {

                        if (listener != null) {
                            listener.onProgress(message);
                        }
                    }
                })
                .uploadFiles(finalList);
    }

    private static void addPano(String token, WorkInfo workInfo, List<UploadFile> data, UploadCallback listener) {

        if (listener != null) {
            listener.onProgress("全景图片上传完成");
        }

        if (workInfo == null) {
            workInfo = new WorkInfo();
        }

        if (workInfo.getWorkName() == null) {
            workInfo.setWorkName("全景作品");
        }

        if (workInfo.getCustomerName() == null) {
            workInfo.setCustomerName("");
        }

        if (workInfo.getCustomerPhone() == null) {
            workInfo.setCustomerPhone("");
        }

        if (workInfo.getDetailAddress() == null) {
            workInfo.setDetailAddress("");
        }

        PublishWork publishWork = new PublishWork();
        publishWork.setName(workInfo.getWorkName());
        publishWork.setCustomer_name(workInfo.getCustomerName());
        publishWork.setCustomer_phone(workInfo.getCustomerPhone());
        publishWork.setDetail_address(workInfo.getDetailAddress());
        publishWork.setDir_id(0);
        publishWork.setDir_name("根目录");
        publishWork.setCate(13);
        publishWork.setCate_name("水电");
        publishWork.setIndustry(0);
        publishWork.setIndustry_name("");
        publishWork.setIs_public(1);
        publishWork.setIs_pay_work(0);
        publishWork.setRegion(0);
        publishWork.setRegion_name("");
        publishWork.setPic_tags(new ArrayList<Integer>());
        List<PublishWork.ImgsBean> imgs = new ArrayList<>();
        for (UploadFile uf : data) {
            PublishWork.ImgsBean img = new PublishWork.ImgsBean();
            img.setImgname(uf.getFileName());
            img.setImgsrc(uf.getObjectKey());
            img.setFilesize(uf.getFileSize());
            imgs.add(img);
        }
        publishWork.setImgs(imgs);
        publishWork.setMimgs(new ArrayList<PublishWork.ImgsBean>());
        WorkInfo finalWorkInfo = workInfo;
        OkHttpUtils
                .postString()
                .url(BASE_URL + "/api/add/pic")
                .addHeader("Token", token)
                .content(JsonUtils.toJson(publishWork))
                .mediaType(MediaType.parse("application/json; charset=utf-8"))
                .build()
                .execute(new StringCallback() {
                    @Override
                    public void onError(Call call, Exception e, int id) {

                        if (needTry(e)) {

                            if (retryNum < maxRetry) {
                                retryNum++;
                                addPano(token, finalWorkInfo, data, listener);
                                if (listener != null) {
                                    listener.onProgress("网络状况不佳，正在重试...");
                                }
                            } else {
                                callFailed(listener, e.getMessage());
                            }

                        } else {
                            callFailed(listener, e.getMessage());
                        }
                    }

                    @Override
                    public void onResponse(String response, int id) {

                        BaseWorkResult<Object> result = JsonUtils.fromJson(response, new TypeToken<BaseWorkResult<Object>>() {
                        }.getType());

                        if (result == null) {
                            callFailed(listener, "全景发布异常1");
                            return;
                        }

                        if (result.isSuccess()) {

                            retryNum = 0;
                            checkPano(token, finalWorkInfo, listener);

                        } else {

                            if (TextUtils.isEmpty(result.getMsg())) {
                                callFailed(listener, result.getMessage());
                            } else {
                                callFailed(listener, result.getMsg());
                            }
                        }
                    }
                });

    }

    private static void checkPano(String token, WorkInfo info, UploadCallback listener) {

        if (listener != null) {
            listener.onProgress("全景云端处理中...");
        }

        OkHttpUtils
                .post()
                .url(BASE_URL + "/api/work/asyncList")
                .addHeader("Token", token)
                .build()
                .execute(new StringCallback() {
                    @Override
                    public void onError(Call call, Exception e, int id) {

                        if (needTry(e)) {

                            if (retryNum < maxRetry) {
                                retryNum++;
                                checkPano(token, info, listener);
                                if (listener != null) {
                                    listener.onProgress("网络状况不佳，正在重试...");
                                }
                            } else {
                                callFailed(listener, e.getMessage());
                            }

                        } else {
                            callFailed(listener, e.getMessage());
                        }
                    }

                    @Override
                    public void onResponse(String response, int id) {

                        BaseWorkResult<List<ProcessWork>> result = JsonUtils.fromJson(response, new TypeToken<BaseWorkResult<List<ProcessWork>>>() {
                        }.getType());

                        if (result == null) {
                            callFailed(listener, "全景发布异常2");
                            return;
                        }

                        if (result.isSuccess()) {

                            List<ProcessWork> works = result.getData();

                            if (!works.isEmpty()) {

                                boolean found = false;
                                for (ProcessWork work : works) {

                                    if (info.getCustomerName().equals(work.getCustomer_name()) &&
                                            info.getCustomerPhone().equals(work.getCustomer_phone()) &&
                                            info.getWorkName().equals(work.getName()) &&
                                            info.getDetailAddress().equals(work.getDetail_address())) {

                                        found = true;
                                        break;
                                    }
                                }

                                if (found) {
                                    handler.postDelayed(new Runnable() {
                                        @Override
                                        public void run() {
                                            checkPano(token, info, listener);
                                        }
                                    }, 3000);
                                } else {

                                    retryNum = 0;
                                    getPanoWork(token, info, listener);
                                }

                            } else {

                                retryNum = 0;
                                getPanoWork(token, info, listener);
                            }

                        } else {

                            if (TextUtils.isEmpty(result.getMsg())) {
                                callFailed(listener, result.getMessage());
                            } else {
                                callFailed(listener, result.getMsg());
                            }

                        }

                    }
                });
    }

    private static void getPanoWork(String token, WorkInfo info, UploadCallback listener) {

        QueryWork queryWork = new QueryWork();
        queryWork.setPid(0);
        queryWork.setName("");
        queryWork.setIs_public(new ArrayList<>());
        QueryWork.OrderBean orderBean = new QueryWork.OrderBean();
        orderBean.setCreate_time("desc");
        queryWork.setOrder(orderBean);
        queryWork.setCurrent_page(1);
        queryWork.setPer_page(10);

        OkHttpUtils
                .postString()
                .url(BASE_URL + "/api/work/dir")
                .addHeader("Token", token)
                .content(JsonUtils.toJson(queryWork))
                .mediaType(MediaType.parse("application/json; charset=utf-8"))
                .build()
                .execute(new StringCallback() {
                    @Override
                    public void onError(Call call, Exception e, int id) {

                        if (needTry(e)) {

                            if (retryNum < maxRetry) {
                                retryNum++;
                                getPanoWork(token, info, listener);
                                if (listener != null) {
                                    listener.onProgress("网络状况不佳，正在重试...");
                                }
                            } else {
                                callFailed(listener, e.getMessage());
                            }

                        } else {
                            callFailed(listener, e.getMessage());
                        }
                    }

                    @Override
                    public void onResponse(String response, int id) {

                        BaseWorkResult<WorkList> result = JsonUtils.fromJson(response, new TypeToken<BaseWorkResult<WorkList>>() {
                        }.getType());

                        if (result == null) {
                            callFailed(listener, "全景发布异常3");
                            return;
                        }

                        if (result.isSuccess()) {

                            for (WorkRecord record : result.getData().getRecords()) {

                                if (info.getCustomerName().equals(record.getCustomer_name()) &&
                                        info.getCustomerPhone().equals(record.getCustomer_phone()) &&
                                        info.getWorkName().equals(record.getName()) &&
                                        info.getDetailAddress().equals(record.getDetail_address())) {

                                    if (record.getIs_pay_work() == 0) {
                                        record.setPano_url(BASE_URL + "/tour/" + record.getId());
                                    } else {
                                        record.setPano_url(BASE_URL + "/viptour/" + record.getUuid());
                                    }

                                    isDealing = false;

                                    if (listener != null) {
                                        listener.onSuccess(record);
                                    }
                                    return;
                                }
                            }

                            /**
                             * 增加重试，避免没有及时获取到最新的全景作品信息
                             */
                            if (retryNum < maxRetry) {
                                retryNum++;
                                getPanoWork(token, info, listener);
                            } else {
                                callFailed(listener, "全景发布异常4");
                            }

                        } else {
                            if (TextUtils.isEmpty(result.getMsg())) {
                                callFailed(listener, result.getMessage());
                            } else {
                                callFailed(listener, result.getMsg());
                            }
                        }

                    }
                });
    }

    public static void getPanoWork(String userToken, int currentPage, int pageSize, WorkCallback listener) {

        getPanoWork(userToken, "", currentPage, pageSize, listener);
    }

    public static void getPanoWork(String userToken, String keyword, int currentPage, int pageSize, WorkCallback listener) {

        if (TextUtils.isEmpty(userToken)) {
            if (listener != null) {
                listener.onFailed("userToken不能为空");
            }
            return;
        }

        if (pageSize < 1 || pageSize > 100) {
            if (listener != null) {
                listener.onFailed("pageSize为1-100之间");
            }
            return;
        }

        if (keyword == null) {
            keyword = "";
        }

        QueryWork queryWork = new QueryWork();
        queryWork.setPid(0);
        queryWork.setName(keyword);
        queryWork.setIs_public(new ArrayList<>());
        QueryWork.OrderBean orderBean = new QueryWork.OrderBean();
        orderBean.setCreate_time("desc");
        queryWork.setOrder(orderBean);
        queryWork.setCurrent_page(currentPage);
        queryWork.setPer_page(pageSize);

        OkHttpUtils
                .postString()
                .url(BASE_URL + "/api/work/dir")
                .addHeader("Token", userToken)
                .content(JsonUtils.toJson(queryWork))
                .mediaType(MediaType.parse("application/json; charset=utf-8"))
                .build()
                .execute(new StringCallback() {
                    @Override
                    public void onError(Call call, Exception e, int id) {

                        if (listener != null) {
                            listener.onFailed(e.getMessage());
                        }
                    }

                    @Override
                    public void onResponse(String response, int id) {

                        BaseWorkResult<WorkList> result = JsonUtils.fromJson(response, new TypeToken<BaseWorkResult<WorkList>>() {
                        }.getType());

                        if (result == null) {
                            if (listener != null) {
                                listener.onFailed("系统错误");
                            }
                            return;
                        }

                        if (result.isSuccess()) {

                            if (listener != null) {

                                for (WorkRecord record : result.getData().getRecords()) {
                                    if (record.getIs_pay_work() == 0) {
                                        record.setPano_url(BASE_URL + "/tour/" + record.getId());
                                    } else {
                                        record.setPano_url(BASE_URL + "/viptour/" + record.getUuid());
                                    }
                                }

                                listener.onSuccess(result.getData().getRecords());
                            }

                        } else {
                            if (TextUtils.isEmpty(result.getMsg())) {
                                if (listener != null) {
                                    listener.onFailed(result.getMessage());
                                }
                            } else {
                                if (listener != null) {
                                    listener.onFailed(result.getMsg());
                                }
                            }
                        }

                    }
                });
    }


    private static boolean needTry(Exception e) {

        if ("timeout".equals(e.getMessage())
                || (e.getMessage() != null && e.getMessage().contains("failed to connect to"))
                || (e.getMessage() != null && e.getMessage().contains("500"))) {

            return true;
        }
        return false;
    }

    private static Context getContext() {
        return ContextProvider.get().getContext();
    }

    public static LoginInfo getLoginInfo() {
        return mLoginInfo;
    }

    private static void callFailed(UploadCallback listener, String text) {
        isDealing = false;
        if (listener != null) {
            listener.onFailed(text);
        }
    }


    private static String getFilePrefix(String fileName) {
        if (fileName.contains(".")) {
            return fileName.substring(0, fileName.lastIndexOf("."));
        }
        return fileName;
    }
}
