/*
 *  Copyright (c) 2023. felord.cn
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *       https://www.apache.org/licenses/LICENSE-2.0
 *  Website:
 *       https://felord.cn
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package cn.felord.mp.retrofit;


import cn.felord.mp.MpApp;
import cn.felord.mp.WeMpTokenCacheable;

/**
 * The type Token api.
 *
 * @author dax
 * @since 2021 /10/7 15:19
 */
public abstract class AbstractTokenApi implements TokenApi {
    private final WeMpTokenCacheable wecomCacheable;
    private final MpApp mpApp;

    /**
     * Instantiates a new Token api.
     *
     * @param wecomCacheable the wecom cacheable
     * @param mpApp          the mp app
     */
    protected AbstractTokenApi(WeMpTokenCacheable wecomCacheable, MpApp mpApp) {
        this.wecomCacheable = wecomCacheable;
        this.mpApp = mpApp;
    }

    /**
     * Gets token response.
     *
     * @return the token response
     */
    @Override
    public final String getToken() {

        final String appid = mpApp.getAppid();
        String tokenCache = wecomCacheable.getAccessToken(appid);
        if (tokenCache == null) {
            synchronized (this) {
                tokenCache = wecomCacheable.getAccessToken(appid);
                if (tokenCache == null) {
                    tokenCache = wecomCacheable.putAccessToken(appid, doGetToken(mpApp));
                }
            }
        }
        return tokenCache;
    }

    @Override
    public MpApp mpApp() {
        return mpApp;
    }

    @Override
    public void clearToken() {
        wecomCacheable.clearAccessToken(mpApp().getAppid());
    }

    /**
     * Do get token string.
     *
     * @param mpApp the mp app
     * @return the string
     */
    protected abstract String doGetToken(MpApp mpApp);

}
