/*
 * Copyright (c) 2023 EOVA.CN. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.eova.tools.tool;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Jieven
 */
public class TimeTool {

    private static final String TIMES = "yyyy-MM-dd HH:mm:ss.SSS";
    private static final String TIME = "yyyy-MM-dd HH:mm:ss";
    private static final String DATE = "yyyy-MM-dd";

    /**
     * 获得当前毫秒数1
     *
     * @return
     */
    public long now() {
        return System.currentTimeMillis();
    }

    /**
     * 语法简化 x.time.format()
     * @param ms
     * @param pattern
     * @return
     */
    @Deprecated
    public String formatMS(long ms, String pattern) {
        return format(ms, pattern);
    }

    /**
     * 格式化
     *
     * @param ms      毫秒数
     * @param pattern 格式
     * @return
     */
    public String format(long ms, String pattern) {
        return format(new Date(ms), pattern);
    }
    /**
     * 格式化
     *
     * @param date    日期
     * @param pattern 格式
     * @return
     */
    public String format(Date date, String pattern) {
        return new SimpleDateFormat(pattern).format(date);
    }

    /**
     * 格式化(yyyy-MM-dd)
     *
     * @param ms 毫秒
     * @return
     */
    public String formatDate(long ms) {
        return format(ms, DATE);
    }
    /**
     * 格式化(yyyy-MM-dd)
     *
     * @param date 时间
     * @return
     */
    public String formatDate(Date date) {
        return format(date, DATE);
    }

    /**
     * 格式化(yyyy-MM-dd HH:mm:ss)
     *
     * @param ms 毫秒
     * @return
     */
    public String formatTime(long ms) {
        return format(ms, TIME);
    }

    public String formatTime(Date date) {
        return format(date, TIME);
    }

    /**
     * 格式化当前时间(自定义规则)
     *
     * @param pattern
     * @return
     */
    public String formatNow(String pattern) {
        return formatMS(now(), pattern);
    }

    /**
     * 当前时间格式化(yyyy-MM-dd HH:mm:ss)
     *
     * @return
     */
    public String formatNowTime() {
        return format(now(), TIME);
    }

    /**
     * 当前时间格式化(yyyy-MM-dd HH:mm:ss:SSS)
     *
     * @return
     */
    public String formatNowTimes() {
        return format(now(), TIMES);
    }

    /**
     * 当前时间格式化(yyyy-MM-dd)
     *
     * @return
     */
    public String formatNowDate() {
        return format(now(), DATE);
    }

    /**
     * 时间转毫秒
     *
     * @param time 时间(yyyy-MM-dd HH:mm:ss)
     * @return 毫秒
     */
    public long parse(String time) {
        LocalDateTime time1 = LocalDateTime.parse(time, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        return LocalDateTime.from(time1).atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * 时间转Date
     * @param time 时间(yyyy-MM-dd HH:mm:ss)
     * @return Date
     */
    public Date parseDate(String time) {
        return new Date(parse(time));
    }

    /**
     * 消耗毫秒数
     *
     * @param time
     */
    public void costTime(long time) {
        System.err.println("Load Cost Time:" + (System.currentTimeMillis() - time) + "ms\n");
    }

    /**
     * 是否超过当前时间xx秒
     *
     * @param oldTime    过去的时间点
     * @param timeoutSec 已过期时长
     * @return
     */
    public boolean isTimeout(Long oldTime, int timeoutSec) {
        if (oldTime == null) {
            return true;
        }
        if ((System.currentTimeMillis() - oldTime) / 1000 > timeoutSec) {
            return true;
        }
        return false;
    }

    /**
     * 计算天
     * +1 当前时间+1天 eg. calcDay(3) -> 3天后
     * -1 当前间-1天 eg. calcDay(-3) -> 3天前
     * @param n
     * @return
     */
    public Timestamp calcDay(long n) {
        return new Timestamp(now() + 60000L * 60 * 24 * n);
    }


    /**
     * 计算时
     * +1 当前时间+1小时 eg. calcHour(3) -> 3小时后
     * -1 当前间-1小时 eg. calcHour(-3) -> 3小时前
     * @param n
     * @return
     */
    public Timestamp calcHour(long n) {
        return new Timestamp(now() + 60000L * 60 * n);
    }

    /**
     * 计算分
     * +1 当前时间+1分 eg. calcMin(3) -> 3分钟后
     * -1 当前间-1分 eg. calcMin(-3) -> 3分钟前
     * @param n
     * @return
     */
    public Timestamp calcMin(long n) {
        return new Timestamp(now() + 60000L * n);
    }

    /**
     * 计算秒
     * +1 当前时间+1秒 eg. calcSec(3) -> 3秒后
     * -1 当前间-1秒 eg. calcSec(-3) -> 3秒前
     * @param n
     * @return
     */
    public Timestamp calcSec(long n) {
        return new Timestamp(now() + 1000L * n);
    }


    /**
     * 日期范围转集合(自动生成范围内的所有日期集合)
     * 用于各种日期循环场景
     * @param start 开始日期 yyyy-MM-dd
     * @param end 结束日期 yyyy-MM-dd
     * @return 开启~结束日期之间的所有日期
     */
    public List<String> parseDates(String start, String end) {
        // 用起始时间作为流的源头，按照每次加一天的方式创建一个无限流
        return Stream.iterate(LocalDate.parse(start), localDate -> localDate.plusDays(1))
                .limit(ChronoUnit.DAYS.between(LocalDate.parse(start), LocalDate.parse(end)) + 1)
                .map(LocalDate::toString)
                .collect(Collectors.toList());
    }

}
