/*
 * Copyright (c) 2023 EOVA.CN. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.eova.tools.tool;

import java.io.File;
import java.util.Collection;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import cn.eova.tools.x;

/**
 * @author Jieven
 */
public class StringTool {

    /**
     * 格式化字符串->'str'
     *
     * @param str
     * @return
     */
    public String format(Object str) {
        return "'" + str.toString() + "'";
    }

    /**
     * 格式化文件路径中的目录分隔符
     *
     * @param path 路径
     * @param args 参数
     * @return
     */
    public String formatPath(String path, Object... args) {
        return String.format(path.replace("/", File.separator), args);
    }

    /**
     * 集合转字符串
     *
     * @param s      集合
     * @param parcel 包裹符 e.g 'aa','bb'
     * @param sign   分割符 e.g 1,2
     * @return 格式化后的字符串
     */
    public String join(Collection<?> s, String parcel, String sign) {
        if (s.isEmpty()) {
            return "";
        }

        // 使用Stream API简化循环逻辑
        return s.stream()
                .map(Object::toString)
                .filter(Objects::nonNull)
                .map(item -> parcel + item + parcel)
                .collect(Collectors.joining(sign));
    }

    public String join(Collection<?> s) {
        return join(s, "", ",");
    }

    /**
     * 删除开头字符串
     *
     * @param s    待处理字符串
     * @param sign 需要删除的符号
     * @return
     */
    public String delStart(String s, String sign) {
        if (s.startsWith(sign)) {
            return s.substring(s.lastIndexOf(sign) + sign.length(), s.length());
        }
        return s;
    }

    /**
     * 删除末尾字符串
     *
     * @param s    待处理字符串
     * @param sign 需要删除的符号
     * @return
     */
    public String delEnd(String s, String sign) {
        if (x.isEmpty(s)) {
            return s;
        }
        if (s.endsWith(sign)) {
            return s.substring(0, s.lastIndexOf(sign));
        }
        return s;
    }

    /**
     * 替换空格,换行符等
     *
     * @param s
     * @return
     */
    public String replaceBlank(String s) {
        if (x.isEmpty(s)) {
            return s;
        }
        Pattern p = Pattern.compile("\\s*|\t|\r|\n");
        Matcher m = p.matcher(s);
        return m.replaceAll("");
    }

    /**
     * 格式化输出JSON
     *
     * @param json
     * @return
     */
    public String formatJson(String json) {
        int level = 0;
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < json.length(); i++) {
            char c = json.charAt(i);
            if (level > 0 && '\n' == sb.charAt(sb.length() - 1)) {
                sb.append(getLevelStr(level));
            }
            switch (c) {
                case '{':
                case '[':
                    sb.append(c + "\n");
                    level++;
                    break;
                case ',':
                    sb.append(c + "\n");
                    break;
                case '}':
                case ']':
                    sb.append("\n");
                    level--;
                    sb.append(getLevelStr(level));
                    sb.append(c);
                    break;
                default:
                    sb.append(c);
                    break;
            }
        }
        return sb.toString();
    }

    private String getLevelStr(int level) {
        StringBuffer levelStr = new StringBuffer();
        for (int levelI = 0; levelI < level; levelI++) {
            levelStr.append("  ");
        }
        return levelStr.toString();
    }

}
