/*
 * Copyright (c) 2023 EOVA.CN. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.eova.tools.tool;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.eova.tools.x;

/**
 * 签名工具
 * MD5 和 HMAC 的主要区别在于：
 * 1.应用场景不同：MD5 主要用于数据完整性验证和数字签名，HMAC 主要用于消息认证、用户认证等场景。
 * 2.实现方式不同：MD5 是单向哈希算法，不需要密钥；HMAC 是基于哈希算法的消息认证码，需要密钥作为输入。
 * 3.安全性不同：MD5 已经被证明存在安全漏洞，不建议用于安全性要求较高的场景；HMAC 的安全性取决于使用的哈希算法和密钥长度，可以提供较高的安全性保障。
 * 4.RSA 非对称加密, 更具有安全性, 需要生成和管理公私钥.
 * <p>
 * 行业参考案例:
 * 微信: https://pay.weixin.qq.com/wiki/doc/apiv3/wechatpay/wechatpay4_1.shtml
 * 支付宝: https://open.taobao.com/api.htm?docId=285&docType=2
 *
 * @author Jieven
 */
public class SignTool {

    public static String SIGN_MD5 = "md5";
    public static String SIGN_HMAC = "hmac";
    private static final Charset UTF8 = StandardCharsets.UTF_8;

    public String appKey;
    public String appSecret;
    public String getway;
    public String signMethod;

    public SignTool(String signMethod, String appKey, String appSecret, String getway) {
        this(signMethod, appKey, appSecret);
        this.getway = getway;
    }

    public SignTool(String signMethod, String appKey, String appSecret) {
        this.appKey = appKey;
        this.appSecret = appSecret;
        this.signMethod = signMethod;
    }

    /**
     * 生成签名和URL
     *
     * @param method 请求方法名
     * @param body   请求业务内容
     */
    public SignUrl generate(String method, String body) {
        return generate(method, body, x.time.now());
    }

    /**
     * 生成签名和URL
     *
     * @param method    请求方法名
     * @param body      请求业务内容
     * @param timestamp 当前时间戳
     * @return
     */
    public SignUrl generate(String method, String body, long timestamp) {
        HashMap<String, String> params = new HashMap<>();
        params.put("app_key", appKey);
        params.put("method", method);
        params.put("timestamp", timestamp + "");
        String sign = sign(params, body, appSecret, signMethod);

        String link = toGetQuery(params);
        String url = String.format("%s?" + link + "&sign=%s", getway, sign);

        return new SignUrl(sign, url);
    }

    public static String sign(Map<String, String> params, String body, String secret, String signMethod) {
        // 第一步：检查参数是否已经排序
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);

        // 第二步：把所有参数名和参数值串在一起
        StringBuilder query = new StringBuilder();
        if (SIGN_MD5.equals(signMethod)) {
            query.append(secret);
        }
        for (String key : keys) {
            String value = params.get(key);
            if (!x.isOneEmpty(key, value)) {
                query.append(key).append(value);
            }
        }
        if (body != null) {
            query.append(body);
        }
        // 第三步：使用MD5/HMAC加密
        byte[] bytes;
        if (SIGN_HMAC.equals(signMethod)) {
            bytes = encryptHMAC(query.toString(), secret);
            return byte2hex(bytes);
        } else {
            query.append(secret);
            return x.encrypt.md5(query.toString());
        }
    }


    /**
     * 参数转成 url get query 方式
     *
     * @param params
     * @return
     */
    private static String toGetQuery(Map<String, String> params) {
        List<String> keys = new ArrayList(params.keySet());
        Collections.sort(keys);
        String prestr = "";
        for (int i = 0; i < keys.size(); i++) {
            String key = keys.get(i);
            String value = params.get(key);
            if (i == keys.size() - 1) {//拼接时，不包括最后一个&字符
                prestr = prestr + key + "=" + value;
            } else {
                prestr = prestr + key + "=" + value + "&";
            }
        }
        return prestr;
    }


    private static byte[] encryptHMAC(String data, String secret) {
        byte[] bytes = null;
        try {
            SecretKey secretKey = new SecretKeySpec(secret.getBytes(UTF8), "HmacMD5");
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data.getBytes(UTF8));
        } catch (GeneralSecurityException gse) {
            x.log.error("HMAC签名算法异常:" + gse.getMessage());
        }
        return bytes;
    }

    private static String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();

        for (int i = 0; i < bytes.length; ++i) {
            String hex = Integer.toHexString(bytes[i] & 255);
            if (hex.length() == 1) {
                sign.append("0");
            }

            sign.append(hex.toUpperCase());
        }

        return sign.toString();
    }


    public static class SignUrl {

        private String sign;
        private String url;

        public SignUrl(String sign, String url) {
            this.sign = sign;
            this.url = url;
        }

        public String getSign() {
            return sign;
        }

        public void setSign(String sign) {
            this.sign = sign;
        }

        public String getUrl() {
            return url;
        }

        public void setUrl(String url) {
            this.url = url;
        }
    }

}


