/**
 * Copyright (c) 2011-2023, James Zhan 詹波 (jfinal@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.eova.tools.serializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import cn.eova.tools.x;

/**
 * JDK 默认序列化方案
 * 优点: 无第三方依赖
 * 缺点: 数据量大时 性能低下
 *
 * @author Jieven
 */
public class JdkSerializer implements ISerializer {

    public static final ISerializer me = new JdkSerializer();

    @Override
    public byte[] serialize(Object obj) {
        ObjectOutputStream objectOut = null;
        try {
            ByteArrayOutputStream bytesOut = new ByteArrayOutputStream();
            objectOut = new ObjectOutputStream(bytesOut);
            objectOut.writeObject(obj);
            objectOut.flush();
            return bytesOut.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (objectOut != null)
                try {
                    objectOut.close();
                } catch (Exception e) {
                    x.log.error(e.getMessage(), e);
                }
        }
    }

    @Override
    public Object deserialize(byte[] bytes) {
        if (bytes == null || bytes.length == 0)
            return null;

        ObjectInputStream objectInput = null;
        try {
            ByteArrayInputStream bytesInput = new ByteArrayInputStream(bytes);
            objectInput = new ObjectInputStream(bytesInput);
            return objectInput.readObject();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (objectInput != null)
                try {
                    objectInput.close();
                } catch (Exception e) {
                    x.log.error(e.getMessage(), e);
                }
        }
    }

}



