package com.enited.union.ui.dialog;

import android.content.Context;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.recyclerview.widget.RecyclerView;

import com.enited.union.R;
import com.enited.union.app.AppActivity;
import com.enited.union.app.AppAdapter;
import com.enited.union.base.BaseAdapter;
import com.enited.union.base.BaseDialog;
import com.enited.union.entity.BankListEntity;
import com.enited.union.http.api.SearchBankListApi;
import com.enited.union.http.model.HttpListData;
import com.enited.union.view.ClearEditText;
import com.hjq.http.EasyHttp;
import com.hjq.http.listener.HttpCallback;

import java.util.List;
import java.util.concurrent.TimeUnit;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;
import io.reactivex.subjects.PublishSubject;


/**
 * author : ZhengYuanle
 * time   : 2022/7/6
 * desc   : 搜索银行对话框
 */
public class SearchBankDialog {
    public static final class Builder
            extends BaseDialog.Builder<Builder>
            implements BaseAdapter.OnItemClickListener {

        private final AppCompatImageView ivClose;
        private OnSelectResult onSearchResult;
        private final BankAdapter mAdapter;
        private final AppActivity activity;
        private final PublishSubject<String> mSubject = PublishSubject.create();


        public Builder(AppActivity activity) {
            super(activity);
            setContentView(R.layout.layout_search_product_dialog);
            this.activity = activity;

            TextView tvTitle = findViewById(R.id.tv_title);
            ivClose = findViewById(R.id.iv_close);
            ClearEditText etSearchKeyword = findViewById(R.id.et_search_keyword);
            RecyclerView mRecyclerView = findViewById(R.id.recyclerView);

            tvTitle.setText("请选择开户银行");
            setOnClickListener(ivClose);

            mAdapter = new BankAdapter(activity);
            mAdapter.setOnItemClickListener(this);
            mRecyclerView.setAdapter(mAdapter);

            mSubject.debounce(300, TimeUnit.MILLISECONDS)
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread())
                    .doOnNext(this::getBankList)
                    .subscribe();

            etSearchKeyword.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {

                }

                @Override
                public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                    mSubject.onNext(charSequence.toString());
                }

                @Override
                public void afterTextChanged(Editable editable) {

                }
            });

            getBankList("");
        }

        @Override
        public void onItemClick(RecyclerView recyclerView, View itemView, int position) {
            if (onSearchResult != null) {
                onSearchResult.onResult(mAdapter.getItem(position));
            }
            dismiss();
        }

        public Builder setOnSearchResult(OnSelectResult onSearchResult) {
            this.onSearchResult = onSearchResult;
            return this;
        }

        @Override
        public void onClick(View view) {
            if (view == ivClose) {
                dismiss();
            }
        }

        private void getBankList(String bankName) {
            EasyHttp.get(activity)
                    .api(new SearchBankListApi().setBankName(bankName))
                    .request(new HttpCallback<HttpListData<BankListEntity>>(null) {
                        @Override
                        public void onSucceed(HttpListData<BankListEntity> result) {
                            List<BankListEntity> list = result.getRows();
                            mAdapter.setData(list);
                        }
                    });
        }
    }

    public interface OnSelectResult {
        void onResult(BankListEntity entity);
    }

    private static final class BankAdapter extends AppAdapter<BankListEntity> {

        public BankAdapter(@NonNull Context context) {
            super(context);
        }

        @NonNull
        @Override
        public BaseAdapter<?>.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
            return new ViewHolder();
        }


        private final class ViewHolder extends AppAdapter<BankListEntity>.ViewHolder {

            private final TextView tvName;

            public ViewHolder() {
                super(R.layout.item_search_custom_list);
                tvName = findViewById(R.id.tv_name);
            }

            @Override
            public void onBindView(int position) {
                BankListEntity entity = getItem(position);
                tvName.setText(entity.getBankName());
            }
        }
    }
}
