package com.enited.union.ui.activity;

import android.content.Intent;
import android.text.TextUtils;
import android.view.View;
import android.widget.TextView;

import androidx.appcompat.widget.AppCompatButton;
import androidx.appcompat.widget.AppCompatEditText;

import com.enited.union.R;
import com.enited.union.app.AppActivity;
import com.enited.union.base.BaseActivity;
import com.enited.union.entity.BankListEntity;
import com.enited.union.entity.UserDetailsEntity;
import com.enited.union.http.api.BindBankCardApi;
import com.enited.union.http.api.GetCodeApi;
import com.enited.union.http.api.UserDetailsApi;
import com.enited.union.http.model.HttpData;
import com.enited.union.ui.dialog.AddressDialog;
import com.enited.union.ui.dialog.SearchBankDialog;
import com.enited.union.view.CountdownView;
import com.hjq.http.EasyHttp;
import com.hjq.http.listener.HttpCallback;

import java.util.Objects;

/**
 * author : ZhengYuanle
 * time   : 2022/9/22
 * desc   :
 */
public class BindBankCardActivity extends AppActivity {

    private TextView tvName;
    private TextView tvBank;
    private AppCompatEditText etBankNumber;
    private TextView tvBankAddress;
    private AppCompatEditText etPhone;
    private AppCompatEditText etCode;
    private CountdownView btnCode;
    private AppCompatButton btnBind;

    private AddressDialog.AddressBean mProvince;
    private AddressDialog.AddressBean mCity;
    private BankListEntity mBankEntity;

    @Override
    protected int getLayoutId() {
        return R.layout.activity_bind_bank_card;
    }

    public static void start(BaseActivity activity, OnBindBankListener listener) {
        Intent intent = new Intent(activity, BindBankCardActivity.class);
        activity.startActivityForResult(intent, (resultCode, data) -> {
            if (listener == null) {
                return;
            }
            if (resultCode == RESULT_OK) {
                listener.onSucceed();
            } else {
                listener.onCancel();
            }
        });
    }

    @Override
    protected void initView() {
        tvName = findViewById(R.id.tv_name);
        tvBank = findViewById(R.id.tv_bank);
        etBankNumber = findViewById(R.id.et_bank_number);
        tvBankAddress = findViewById(R.id.tv_bank_address);
        etPhone = findViewById(R.id.et_phone);
        etCode = findViewById(R.id.et_code);
        btnCode = findViewById(R.id.btn_code);
        btnBind = findViewById(R.id.btn_bind);

        mTitleBar.setLeftClickListener(v -> onBackPressed());

        setOnClickListener(btnCode, tvBankAddress, tvBank, btnBind);
    }

    @Override
    protected void initData() {
        getUserDetails();
    }

    @Override
    public void onClick(View view) {
        if (view == btnCode) {
            String phone = Objects.requireNonNull(etPhone.getText()).toString();

            if (phone.length() != 11) {
                toast(R.string.common_phone_input_error);
                return;
            }
            //发送短信
            EasyHttp.post(this)
                    .api(new GetCodeApi().setMobile(phone).setType(4))
                    .request(new HttpCallback<HttpData<Void>>(this) {
                        @Override
                        public void onSucceed(HttpData<Void> result) {
                            toast(R.string.common_code_send_hint);
                            btnCode.start();
                        }
                    });
        } else if (view == tvBankAddress) {
            new AddressDialog.Builder(this)
                    .setIgnoreArea()
                    .setListener((dialog, province, city, area) -> {
                        mProvince = province;
                        mCity = city;
                        String address = province.getName() + city.getName();
                        tvBankAddress.setText(address);
                    })
                    .show();
        } else if (view == tvBank) {
            new SearchBankDialog.Builder(BindBankCardActivity.this)
                    .setOnSearchResult(entity -> {
                        mBankEntity = entity;
                        tvBank.setText(entity.getBankName());
                    })
                    .show();
        } else if (view == btnBind) {
            String bankNum = Objects.requireNonNull(etBankNumber.getText()).toString();
            String bankCode = Objects.requireNonNull(mBankEntity).getBankCode();
            String bankName = Objects.requireNonNull(mBankEntity).getBankName();
            String bankPhone = Objects.requireNonNull(etPhone.getText()).toString();
            String bankProv = Objects.requireNonNull(mProvince).getId();
            String bankArea = Objects.requireNonNull(mCity).getId();
            String username = tvName.getText().toString();
            String phoneCode = Objects.requireNonNull(etCode.getText()).toString();

            bindBankCard(bankNum, bankCode, bankName, bankPhone, bankProv, bankArea, username, phoneCode);
        }
    }

    private void getUserDetails() {
        EasyHttp.get(this)
                .api(new UserDetailsApi())
                .request(new HttpCallback<HttpData<UserDetailsEntity>>(this) {
                    @Override
                    public void onSucceed(HttpData<UserDetailsEntity> result) {
                        tvName.setText(result.getData().getRealName());
                    }
                });
    }

    /**
     * 绑定银行卡
     *
     * @param bankNum   银行卡号
     * @param bankCode  银行code
     * @param bankName  银行名
     * @param bankPhone 预留手机号
     * @param bankProv  省id
     * @param bankArea  市id
     * @param username  用户姓名
     * @param phoneCode 验证码
     */
    public void bindBankCard(String bankNum, String bankCode, String bankName, String bankPhone, String bankProv, String bankArea
            , String username, String phoneCode) {

        if (TextUtils.isEmpty(bankCode)) {
            toast("请选择银行");
            return;
        }
        if (TextUtils.isEmpty(bankNum)) {
            toast("请输入银行账号");
            return;
        }
        if (TextUtils.isEmpty(bankProv) || TextUtils.isEmpty(bankArea)) {
            toast("请选择开户银行省市");
            return;
        }
        if (TextUtils.isEmpty(bankPhone)) {
            toast("请输入预留手机号");
            return;
        }
        if (TextUtils.isEmpty(phoneCode)) {
            toast("请输入验证码");
            return;
        }

        EasyHttp.post(this)
                .api(new BindBankCardApi()
                        .setBankNum(bankNum)
                        .setBankCode(bankCode)
                        .setBankName(bankName)
                        .setBankPhone(bankPhone)
                        .setBankProv(bankProv)
                        .setBankArea(bankArea)
                        .setOrgOrUserId("")
                        .setOrgOrUserName(username)
                        .setPhoneCode(phoneCode)
                )
                .request(new HttpCallback<HttpData<Void>>(this) {
                    @Override
                    public void onSucceed(HttpData<Void> result) {
                        toast("绑定成功");
                        postDelayed(() -> {
                            setResult(RESULT_OK);
                            finish();
                        }, 1000);
                    }
                });
    }

    /**
     * 监听
     */
    public interface OnBindBankListener {

        /**
         * 成功
         */
        void onSucceed();

        /**
         * 取消注册
         */
        default void onCancel() {
        }
    }
}
