package com.enited.union.app;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import com.enited.union.R;
import com.enited.union.base.BaseActivity;
import com.enited.union.base.BaseDialog;
import com.enited.union.http.model.HttpData;
import com.enited.union.ui.dialog.WaitDialog;
import com.enited.union.utils.StatusBarUtil;
import com.enited.union.utils.ToastUtils;
import com.enited.union.view.MyTitleBar;
import com.hjq.http.listener.OnHttpListener;

import org.greenrobot.eventbus.EventBus;

import okhttp3.Call;

/**
 * author : ZhengYuanle
 * time   : 2022/6/28
 * desc   : Activity 业务基类
 */
public abstract class AppActivity extends BaseActivity
        implements OnHttpListener<Object> {

    /**
     * 加载对话框
     */
    private BaseDialog mDialog;
    /**
     * 对话框数量
     */
    private int mDialogCount;
    /**
     * 标题栏对象
     */
    public MyTitleBar mTitleBar;

    /**
     * 当前加载对话框是否在显示中
     */
    public boolean isShowDialog() {
        return mDialog != null && mDialog.isShowing();
    }

    /**
     * 需要接收事件 重写该方法 并返回true
     */
    protected boolean regEvent() {
        return false;
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        if (newConfig.fontScale != 1)//非默认值
            getResources();
        super.onConfigurationChanged(newConfig);
    }

    @Override
    public Resources getResources() {
        Resources res = super.getResources();
        if (res.getConfiguration().fontScale != 1) {//非默认值
            Configuration newConfig = new Configuration();
            newConfig.setToDefaults();//设置默认
            res.updateConfiguration(newConfig, res.getDisplayMetrics());
        }
        return res;
    }


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        StatusBarUtil.StatusBarLightMode(this); //设置白底黑字
        if (regEvent()) {
            EventBus.getDefault().register(this);
        }
    }

    @Override
    protected void initLayout() {
        super.initLayout();
        mTitleBar = obtainTitleBar(getContentView());
    }

    /**
     * 递归获取 ViewGroup 中的 TitleBar 对象
     */
    private MyTitleBar obtainTitleBar(ViewGroup group) {
        if (group == null) {
            return null;
        }
        for (int i = 0; i < group.getChildCount(); i++) {
            View view = group.getChildAt(i);
            if ((view instanceof MyTitleBar)) {
                return (MyTitleBar) view;
            }

            if (view instanceof ViewGroup) {
                MyTitleBar titleBar = obtainTitleBar((ViewGroup) view);
                if (titleBar != null) {
                    return titleBar;
                }
            }
        }
        return null;
    }

    /**
     * 显示加载对话框
     */
    public void showDialog() {
        if (isFinishing() || isDestroyed()) {
            return;
        }

        mDialogCount++;
        if (mDialogCount <= 0 || isFinishing() || isDestroyed()) {
            return;
        }

        if (mDialog == null) {
            mDialog = new WaitDialog.Builder(this)
                    .setCancelable(false)
                    .create();
        }
        if (!mDialog.isShowing()) {
            mDialog.show();
        }
    }

    /**
     * 隐藏加载对话框
     */
    public void hideDialog() {
        if (isFinishing() || isDestroyed()) {
            return;
        }

        if (mDialogCount > 0) {
            mDialogCount--;
        }

        if (mDialogCount != 0 || mDialog == null || !mDialog.isShowing()) {
            return;
        }

        mDialog.dismiss();
    }


    @Override
    public void startActivityForResult(Intent intent, int requestCode, @Nullable Bundle options) {
        super.startActivityForResult(intent, requestCode, options);
        overridePendingTransition(R.anim.right_in_activity, R.anim.right_out_activity);
    }

    @Override
    public void finish() {
        super.finish();
        overridePendingTransition(R.anim.left_in_activity, R.anim.left_out_activity);
    }

    /**
     * {@link OnHttpListener}
     */

    @Override
    public void onStart(Call call) {
        showDialog();
    }

    @Override
    public void onSucceed(Object result) {
        if (result instanceof HttpData) {
            toast(((HttpData<?>) result).getMessage());
        }
    }

    public void toast(String str) {
        if (!TextUtils.isEmpty(str)) {
            ToastUtils.showToast(str);
        }
    }

    public void toast(int strId) {
        ToastUtils.showToast(getResources().getString(strId));
    }

    @Override
    public void onFail(Exception e) {
        toast(e.getMessage());
    }

    @Override
    public void onEnd(Call call) {
        hideDialog();
    }

    @SuppressLint("ObsoleteSdkInt")
    public boolean hasPermissions(@NonNull String... perms) {
        // Always return true for SDK < M, let the system deal with the permissions
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            return true;
        }

        for (String perm : perms) {
            if (ContextCompat.checkSelfPermission(this, perm)
                    != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }

        return true;
    }


    public void requestPermissions(int requestCode, @NonNull String... perms) {
        ActivityCompat.requestPermissions(this, perms, requestCode);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (regEvent()) {
            EventBus.getDefault().unregister(this);
        }
        if (isShowDialog()) {
            hideDialog();
        }
        mDialog = null;
    }
}
