package com.enited.union;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;
import android.view.Gravity;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import com.enited.union.app.AppConfig;
import com.enited.union.base.BaseActivity;
import com.enited.union.http.model.RequestHandler;
import com.enited.union.http.model.RequestServer;
import com.enited.union.listener.OnBankResultListener;
import com.enited.union.listener.OnResultListener;
import com.enited.union.manager.ActivityManager;
import com.enited.union.ui.activity.CapitalAccountActivity;
import com.enited.union.ui.activity.ChangePhoneActivity;
import com.enited.union.ui.activity.ForgetActivity;
import com.enited.union.ui.activity.ImageSelectActivity;
import com.enited.union.ui.activity.MyBankCardActivity;
import com.enited.union.ui.activity.UserCertificationActivity;
import com.enited.union.utils.AccountUtils;
import com.esign.esignsdk.EsignSdk;
import com.hjq.http.EasyConfig;
import com.hjq.http.EasyLog;
import com.hjq.http.config.IRequestApi;
import com.hjq.http.config.IRequestInterceptor;
import com.hjq.http.model.HttpHeaders;
import com.hjq.http.model.HttpParams;

import java.util.List;

import okhttp3.OkHttpClient;

/**
 * author : ZhengYuanle
 * time   : 2022/9/19
 * desc   : SDK公开管理类
 */
public class EnitedUnionSDKManager {

    @SuppressLint("StaticFieldLeak")
    private static Application mApplication;

    public static Application getApp() {
        return mApplication;
    }

    /**
     * 初始化SDK
     *
     * @param systemCode 所属系统 1："亿联工坊" , 2："彼仙草" , 3："临床试验" , 4："PBM患者端" , 5："PBM医生端"
     */
    public static void init(Application application, int systemCode) {

        mApplication = application;

        // 网络请求框架初始化
        OkHttpClient okHttpClient = new OkHttpClient.Builder()
                .build();

        EasyConfig.with(okHttpClient)
                // 是否打印日志
                .setLogEnabled(AppConfig.isLogEnable())
                // 设置服务器配置
                .setServer(new RequestServer())
                // 设置请求处理策略
                .setHandler(new RequestHandler(application))
                // 设置请求重试次数
                .setRetryCount(1)
                .setInterceptor((api, params, headers) -> {
                    String token = AccountUtils.getInstance().getToken();
                    if (!TextUtils.isEmpty(token)) {
                        headers.put("Authorization", token);
                    }
                    headers.put("SystemCode", String.valueOf(systemCode));
                    headers.put("ClientType", "2");
                    headers.put("OsType", "2");
                })
                .into();

        // Activity 栈管理初始化
        ActivityManager.getInstance().init(application);

        EasyConfig.getInstance().setLogEnabled(AppConfig.isLogEnable());

    }

    /**
     * 登录
     *
     * @param token 用户token
     */
    public static void login(String token) {
        AccountUtils.getInstance().saveToken(token);
    }

    /**
     * 退出账号
     */
    public static void logout() {
        AccountUtils.getInstance().saveToken("");
    }


    /**
     * 进入人脸识别页面（进行人脸识别）
     *
     * @param listener 结果回调（只有进行人脸识别操作才返回，否者不回调）
     */
    public static void startUserCertification(Activity activity, OnResultListener listener) {
        UserCertificationActivity.start(activity, listener);
    }

    /**
     * 进入修改密码页
     *
     * @param listener 结果回调
     */
    public static void startModifyPassword(Context context, OnResultListener listener) {
        ForgetActivity.start(context, 1, listener);
    }

    /**
     * 进入忘记密码页
     *
     * @param listener 结果回调
     */
    public static void startForgetPassword(Context context, OnResultListener listener) {
        ForgetActivity.start(context, 0, listener);
    }

    /**
     * 进入忘记密码页
     *
     * @param listener 结果回调
     */
    public static void startChangePhone(Context context, OnResultListener listener) {
        ChangePhoneActivity.start(context, listener);
    }

    /**
     * 进入我的银行卡页
     *
     * @param listener 结果回调
     */
    public static void startMyBank(Context context, OnBankResultListener listener) {
        MyBankCardActivity.start(context, listener);
    }

    /**
     * 进入开通资金账户页面
     *
     * @param listener 结果回调
     */
    public static void startCapitalAccount(Context context, OnResultListener listener) {
        CapitalAccountActivity.start(context, listener);
    }

    public static void initESign() {
        String eSignKey = readMetaDataFromApplication("com.esign.esignsdk.API_KEY");
        String eSignLicense = readMetaDataFromApplication("com.esign.esignsdk.API_LICENSE");
        if (TextUtils.isEmpty(eSignKey) || TextUtils.isEmpty(eSignLicense)) {
            throw new IllegalStateException("E签宝初始化失败?");
        }
        EsignSdk.getInstance().init(eSignKey, eSignLicense);
    }

    public static void goSign(AppCompatActivity activity, String url) {
        EsignSdk.getInstance().setShowLoading(true);
        EsignSdk.getInstance().startH5Activity(activity, url);
    }

    public static void closeH5Activity() {
        EsignSdk.getInstance().finishH5Activity();
    }

    /**
     * 读取application 节点  meta-data 信息
     */
    public static String readMetaDataFromApplication(String key) {
        try {
            ApplicationInfo appInfo = getApp().getPackageManager().getApplicationInfo(getApp().getApplicationContext().getPackageName(), PackageManager.GET_META_DATA);
            return appInfo.metaData.getString(key);

        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return "";
        }
    }
}
