package cn.elwy.common.util.sysinfo;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;
import cn.elwy.common.util.encode.EncodeUtil;

/**
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class HardWareUtil {

	private static final Logger logger = LoggerFactory.getLogger(HardWareUtil.class);

	private static class LazyHolder {
		private static final HardWareUtil INSTANCE = new HardWareUtil();
	}

	private MachineCodeService sysInfo = null;

	public static HardWareUtil getInstance() {
		return LazyHolder.INSTANCE;
	}

	private HardWareUtil() {
		if (OSUtil.isWindows()) {
			sysInfo = new WindowsSystemInfo();
		} else {
			sysInfo = new LinuxSystemInfo();
		}
	}

	/**
	 * 读取CPU序列号
	 * @return
	 */
	public String getCPUSN() {
		return sysInfo.getCPUSN();
	}

	/**
	 * 读取主板序列号
	 * @return
	 */
	public String getMotherboardSN() {
		return sysInfo.getMotherboardSN();
	}

	/**
	 * 读取硬盘序列号
	 * @return
	 */
	public String getHardDiskSN() {
		return sysInfo.getHardDiskSN();
	}

	/**
	 * 获取硬件序列号:CPU-主板-硬盘
	 * @return
	 */
	public String getMachineCode() {
		return EncodeUtil.encryptAES(getHardWareInfo());
	}

	/**
	 * 获取硬件序列号:CPU-主板-硬盘
	 * @return
	 */
	public String getHardWareInfo() {
		return getHardWareInfo("-");
	}

	/**
	 * 获取硬件序列号，包含cpu,主板,硬盘序号
	 * @param separator 分隔符
	 * @return
	 * @throws IOException
	 */
	public String getHardWareInfo(String separator) {
		StringBuffer str = new StringBuffer();
		try {
			str.append(getCPUSN()).append(separator);
		} catch (Exception e) {
			logger.error(e.getMessage());
			str.deleteCharAt(str.length() - separator.length());
		}
		try {
			str.append(getMotherboardSN()).append(separator);
		} catch (Exception e) {
			logger.error(e.getMessage());
			str.deleteCharAt(str.length() - separator.length());
		}
		try {
			str.append(getHardDiskSN());
		} catch (Exception e) {
			logger.error(e.getMessage());
			str.deleteCharAt(str.length() - separator.length());
		}
		return str.toString();
	}

	/**
	 * 获取主机名称
	 * @return
	 */
	public static String getLocalHostName() {
		String hostName = "";
		try {
			InetAddress addr = InetAddress.getLocalHost();
			hostName = addr.getHostName();
		} catch (Exception e) {
			logger.error(e.getMessage());
		}
		return hostName;
	}

	/**
	 * 获取所有本地主机IP
	 * @return
	 */
	public static String[] getAllLocalHostIP() {
		String[] ret = null;
		try {
			String hostName = getLocalHostName();
			if (hostName != null && !hostName.isEmpty()) {
				InetAddress[] addrs = InetAddress.getAllByName(hostName);
				if (addrs != null && addrs.length > 0) {
					ret = new String[addrs.length];
					for (int i = 0; i < addrs.length; i++) {
						ret[i] = addrs[i].getHostAddress();
						System.out.println(ret[i]);
					}
				}
			}
		} catch (Exception e) {
			logger.error(e.getMessage());
		}
		return ret;
	}

	/**
	 * 获取本地IP的MAC地址
	 * @return
	 */
	public static String getMACAddress() {
		try {
			InetAddress address = InetAddress.getLocalHost();
			return getMACAddress(address);
		} catch (UnknownHostException e) {
			logger.error(e.getMessage());
		}
		return null;
	}

	/**
	 * 根据IP获取指定的MAC地址
	 * @param ip
	 * @return
	 */
	public static String getMACAddress(String ip) {
		if ("localhost".equalsIgnoreCase(ip) || "127.0.0.1".equals(ip)) {
			return getMACAddress();
		}
		try {
			InetAddress address = InetAddress.getByName(ip);
			return getMACAddress(address);
		} catch (UnknownHostException e) {
			logger.error(e.getMessage());
		}
		return null;
	}

	/**
	 * 根据InetAddress获取指定的MAC地址
	 * @param address
	 * @return
	 */
	public static String getMACAddress(InetAddress address) {
		try {
			/*- 获得当前主机NetworkInterface,然后读取硬件地址 */
			NetworkInterface ni = NetworkInterface.getByInetAddress(address);
			if (ni != null) {
				byte[] mac = ni.getHardwareAddress();
				if (mac != null) {
					/*- 提取每个数组的MAC地址并使用以下格式将它转换成十六进制：08-00-27-DC-4A-9E */
					StringBuffer macAddress = new StringBuffer();
					for (int i = 0; i < mac.length; i++) {
						macAddress.append(String.format("%02X%s", mac[i], (i < mac.length - 1) ? "-" : ""));
					}
					return macAddress.toString();
				} else {
					logger.warn("Address doesn't exist or is not accessible.");
				}
			} else {
				logger.warn("Network Interface for the specified address is not found.");
			}
		} catch (SocketException e) {
			logger.error(e.getMessage());
		}
		throw new RuntimeException("读取硬盘序列号失败！");
	}

	/**
	 * 获取本地系统的所有Mac地址
	 * @return
	 */
	public static List<String> getLocalMacs() {
		String os = System.getProperty("os.name");
		List<String> list = new ArrayList<String>();
		if (os != null && os.startsWith("Windows")) {
			String command = "cmd.exe /c ipconfig /all";
			return getLocalMac(command);
		} else if (os != null && os.startsWith("Linux")) {
			String command = "/bin/sh -c ifconfig -a";
			return getLocalMac(command);
		}
		return list;
	}

	/**
	 * 获取本地系统的所有Mac地址
	 * @param command 操作系统对应的操作命令
	 * @return
	 */
	public static List<String> getLocalMac(String command) {
		String address = null;
		List<String> list = new ArrayList<String>();
		try {
			Process p = Runtime.getRuntime().exec(command);
			BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));
			String line = null;
			while ((line = br.readLine()) != null) {
				if (line
						.matches(".*[:]{1}.[0-9A-Z]{2}[-][0-9A-Z]{2}[-][0-9A-Z]{2}[-][0-9A-Z]{2}[-][0-9A-Z]{2}[-][0-9A-Z]{2}$")) {
					int index = line.indexOf(":");
					index += 2;
					address = line.substring(index);
					list.add(address);
				}
			}
			br.close();
		} catch (IOException e) {
			logger.error(e.getMessage());
		}
		return list;
	}

	/**
	 * 获取Linux操作系统的Mac地址
	 * @return
	 */
	public static String getLinuxMac() {
		String address = null;
		try {
			String command = "/bin/sh -c ifconfig -a";
			Process p = Runtime.getRuntime().exec(command);
			BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));
			String line;
			while ((line = br.readLine()) != null) {
				if (line.indexOf("HWaddr") > 0) {
					int index = line.indexOf("HWaddr") + "HWaddr".length();
					address = line.substring(index);
					break;
				}
			}
			br.close();
		} catch (IOException e) {
		}
		return address;
	}

	public static void main(String[] args) throws IOException {
		System.out.println("机器码:" + HardWareUtil.getInstance().getMachineCode());
		System.out.println("机器码:" + HardWareUtil.getInstance().getHardWareInfo());
		System.out.println("CPU  SN:" + HardWareUtil.getInstance().getCPUSN());
		System.out.println("主板  SN:" + HardWareUtil.getInstance().getMotherboardSN());
		System.out.println("C盘   SN:" + HardWareUtil.getInstance().getHardDiskSN());
		System.out.println("MAC  SN:" + HardWareUtil.getMACAddress());

		System.out.println(getMACAddress("127.0.0.1"));
		try {
			InetAddress address = InetAddress.getLocalHost();
			System.out.println(getMACAddress(address));
		} catch (UnknownHostException e) {
			e.printStackTrace();
		}
		List<String> list = getLocalMacs();
		for (int i = 0; i < list.size(); i++) {
			String valuee = list.get(i);
			System.out.println("MACs ===" + valuee);
		}

		getAllLocalHostIP();
		System.out.println(getLocalHostName());

	}

}
