package cn.elwy.common.util.sort;

import java.math.BigDecimal;
import java.text.Collator;
import java.text.Format;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;
import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.ReflectUtil;

/**
 * 利用反射机制进行排序工具类，支持数组、List、Set、Map的排序
 * 可以支持多属性排序，支持子对象中的属性排序（例user对象中的role的name属性进行排序，method="role.name"）
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class SortUtil {

	private static Logger logger = LoggerFactory.getLogger(SortUtil.class);
	/** 中文排序 */
	public static final Comparator<Object> CHINA_COMPARATOR = Collator.getInstance(Locale.CHINA);

	private SortUtil() {
	}

	/**
	 * 根据指定的比较器排序一个数组中的元素
	 * @param arrays 需要排序的数组
	 * @param comparator 排序比较器
	 */
	public static void sort(Object[] arrays, Comparator<Object> comparator) {
		Arrays.sort(arrays, comparator);
	}

	/**
	 * 根据指定的比较器排序一个List中的元素
	 * @param list 需要排序的集合
	 * @param comparator 排序比较器
	 */
	public static void sort(List<?> list, Comparator<Object> comparator) {
		Collections.sort(list, comparator);
	}

	/**
	 * 根据指定的比较器排序一个Set中的元素
	 * @param set 需要排序的集合
	 * @param comparator 排序比较器
	 * @return Set TreeSet对象
	 */
	@SuppressWarnings("unchecked")
	public static <T> Set<T> sort(Set<?> set, Comparator<Object> comparator) {
		Set<Object> sortSet = new TreeSet<Object>(comparator);
		sortSet.addAll(set);
		return (Set<T>) sortSet;
	}

	/**
	 * 排序一个数组中的元素，元素必需为Java提供的数据类型
	 * @param arrays 需要排序的数组
	 * @param isAsc 是否升序
	 */
	public static void sort(Object[] arrays, boolean isAsc) {
		sort(arrays, getComparator(isAsc));
	}

	/**
	 * 排序一个List中的元素，元素必需为Java提供的数据类型
	 * @param list 需要排序的集合
	 * @param isAsc 是否升序
	 */
	public static void sort(List<?> list, boolean isAsc) {
		sort(list, getComparator(isAsc));
	}

	/**
	 * 排序一个Set中的元素，元素必需为Java提供的数据类型
	 * @param set 需要排序的集合
	 * @param isAsc 是否升序
	 */
	public static <T> Set<T> sort(Set<?> set, boolean isAsc) {
		return sort(set, getComparator(isAsc));
	}

	/**
	 * 利用java反射机制排序一个对象数组
	 * @param arrays 需要排序的数组
	 * @param isAsc 是否升序
	 * @param name 属性名称或方法名称(方法名称带括号)
	 */
	public static void sort(Object[] arrays, boolean isAsc, String name) {
		sort(arrays, getComparator(isAsc, name));
	}

	/**
	 * 利用java反射机制排序一个集合对象
	 * @param list 需要排序的集合
	 * @param isAsc 是否升序
	 * @param name 属性名称或方法名称(方法名称带括号)
	 */
	public static void sort(List<?> list, boolean isAsc, String name) {
		sort(list, getComparator(isAsc, name));
	}

	/**
	 * 利用java反射机制排序一个Set对象
	 * @param set 需要排序的集合
	 * @param isAsc 是否升序
	 * @param name 属性名称或方法名称(方法名称带括号)
	 */
	public static <T> Set<T> sort(Set<?> set, boolean isAsc, String name) {
		return sort(set, getComparator(isAsc, name));
	}

	/**
	 * 利用java反射机制排序一个对象数组
	 * @param <T>
	 * @param arrays 需要排序的数组
	 * @param isAsc 是否升序，如果数组下标不足属性参数个数，将按最后一个状态表示后续属性的排序是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表
	 */
	public static void sort(Object[] arrays, boolean[] isAsc, String... names) {
		sort(arrays, getComparator(isAsc, names));
	}

	/**
	 * 利用java反射机制排序一个集合对象
	 * @param list 需要排序的集合
	 * @param isAsc 是否升序，如果数组下标不足属性参数个数，将按最后一个状态表示后续属性的排序是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表
	 */
	public static void sort(List<?> list, boolean[] isAsc, String... names) {
		sort(list, getComparator(isAsc, names));
	}

	/**
	 * 利用java反射机制排序一个Set对象
	 * @param set 需要排序的集合
	 * @param isAsc 是否升序，如果数组下标不足属性参数个数，将按最后一个状态表示后续属性的排序是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表
	 */
	public static <T> Set<T> sort(Set<?> set, boolean[] isAsc, String... names) {
		return sort(set, getComparator(isAsc, names));
	}

	/**
	 * 根据指定的条件排序一个数组中的元素
	 * @param arrays 需要排序的数组
	 * @param condition 排序条件
	 */
	public static void sort(Object[] arrays, SortCondition condition) {
		sort(arrays, getComparator(condition));
	}

	/**
	 * 根据指定的条件排序一个List中的元素
	 * @param list 需要排序的集合
	 * @param condition 排序条件
	 */
	public static void sort(List<?> list, SortCondition condition) {
		sort(list, getComparator(condition));
	}

	/**
	 * 根据指定的条件排序一个Set中的元素
	 * @param set 需要排序的集合
	 * @param condition 排序条件
	 * @return Set TreeSet对象
	 */
	public static <T> Set<T> sort(Set<?> set, SortCondition condition) {
		return sort(set, getComparator(condition));
	}

	/**
	 * 根据指定的条件列表排序一个数组中的元素
	 * @param arrays 需要排序的数组
	 * @param conditionList 条件集，按list先后顺序进行排序
	 */
	public static void sort(Object[] arrays, List<SortCondition> conditionList) {
		sort(arrays, getComparator(conditionList));
	}

	/**
	 * 根据指定的条件列表排序一个List中的元素
	 * @param list 需要排序的集合
	 * @param conditionList 条件集，按list先后顺序进行排序
	 */
	public static void sort(List<?> list, List<SortCondition> conditionList) {
		sort(list, getComparator(conditionList));
	}

	/**
	 * 根据指定的条件列表排序一个Set中的元素
	 * @param set 需要排序的Set
	 * @param conditionList 条件集，按list先后顺序进行排序
	 * @return Set TreeSet对象
	 */
	public static <T> Set<T> sort(Set<?> set, List<SortCondition> conditionList) {
		return sort(set, getComparator(conditionList));
	}

	/**
	 * 根据指定的比较器排序一个树形结构的数组对象中的元素
	 * @param arrays 需要排序的数组
	 * @param comparator 排序比较器
	 * @param name 属性名称或方法名称(方法名称带括号)
	 */
	public static void sortTree(Object[] arrays, Comparator<Object> comparator, String name) {
		if (AssertUtil.isEmpty(arrays)) {
			return;
		}
		sort(arrays, comparator);
		for (Object object : arrays) {
			Object[] child = ReflectUtil.getValue(object, name);
			if (AssertUtil.isNotEmpty(child)) {
				sortTree(child, comparator, name);
			}
		}
	}

	/**
	 * 根据指定的比较器排序一个树形结构的List对象中的元素
	 * @param list 需要排序的集合
	 * @param comparator 排序比较器
	 * @param name 属性名称或方法名称(方法名称带括号)
	 */
	public static void sortTree(List<?> list, Comparator<Object> comparator, String name) {
		if (AssertUtil.isEmpty(list)) {
			return;
		}
		sort(list, comparator);
		for (Object object : list) {
			List<?> child = ReflectUtil.getValue(object, name);
			if (AssertUtil.isNotEmpty(child)) {
				sortTree(child, comparator, name);
			}
		}
	}

	/**
	 * 根据Map的Key排序一个Map中的元素
	 * @param <T>
	 * @param <K, V>
	 * @param msgMaps 需要排序的Map
	 * @param comparator 排序比较器
	 * @return Map TreeMap对象
	 */
	@SuppressWarnings("unchecked")
	public static <K, V, T> Map<K, V> sort(Map<K, V> map, Comparator<T> comparator) {
		Map<K, V> sortMap = new TreeMap<K, V>((Comparator<? super K>) comparator);
		sortMap.putAll(map);
		return (Map<K, V>) sortMap;
	}

	/**
	 * 根据Map的Key排序一个Map中的元素，Key必需为Java提供的数据类型
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param isAsc 是否升序
	 * @return
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, boolean isAsc) {
		return sort(map, getComparator(isAsc));
	}

	/**
	 * 利用java反射机制排序一个map对象，根据Map的Key排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param isAsc 是否升序
	 * @param name 属性名称或方法名称(方法名称带括号)
	 * @return
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, boolean isAsc, String name) {
		return sort(map, getComparator(isAsc, name));
	}

	/**
	 * 利用java反射机制排序一个map对象，根据Map的Key排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param isAsc 是否升序，如果数组下标不足属性参数个数，将按最后一个状态表示后续属性的排序是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表
	 * @return
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, boolean[] isAsc, String... names) {
		return sort(map, getComparator(isAsc, names));
	}

	/**
	 * 根据Map的Value排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的Map
	 * @param comparator 排序比较器
	 * @return Map TreeMap对象
	 */
	public static <K, V> Map<K, V> sortMapByEntry(Map<K, V> map, final Comparator<Entry<K, V>> comparator) {
		Map<K, V> sortedMap = new LinkedHashMap<K, V>();
		if (map != null && !map.isEmpty()) {
			List<Entry<K, V>> entryList = new ArrayList<Entry<K, V>>(map.entrySet());
			Collections.sort(entryList, comparator);
			Iterator<Map.Entry<K, V>> iter = entryList.iterator();
			Map.Entry<K, V> tmpEntry = null;
			while (iter.hasNext()) {
				tmpEntry = iter.next();
				sortedMap.put(tmpEntry.getKey(), tmpEntry.getValue());
			}
		}
		return sortedMap;
	}

	/**
	 * 根据Map的Value排序一个Map中的元素，Value必需为Java提供的数据类型
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param byValue 是否按值排序
	 * @param isAsc 是否升序
	 * @return
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, boolean byValue, boolean isAsc) {
		return sortMapByEntry(map, getComparator(map, byValue, isAsc));
	}

	/**
	 * 利用java反射机制排序一个map对象，根据Map的Value排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param byValue 是否按值排序
	 * @param isAsc 是否升序，如果数组下标不足方法参数个数，将按最后一个状态表示后续方法的排序是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表
	 * @return
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, boolean byValue, boolean[] isAsc, String... names) {
		return sortMapByEntry(map, getComparator(map, byValue, isAsc, names));
	}

	/**
	 * 根据指定的比较器排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的Map
	 * @param condition 排序条件
	 * @return Map TreeMap对象
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, SortCondition condition) {
		return sort(map, getComparator(condition));
	}

	/**
	 * 根据指定的条件列表排序一个Map中的元素
	 * @param <K, V>
	 * @param msgMaps 需要排序的Map
	 * @param conditionList 条件集，按list先后顺序进行排序
	 * @return Map TreeMap对象
	 */
	public static <K, V> Map<K, V> sort(Map<K, V> map, List<SortCondition> conditionList) {
		return sort(map, getComparator(conditionList));
	}

	/**
	 * 根据排序方式获取Comparator
	 * @param <T>
	 * @param isAsc 是否升序
	 * @return Comparator 排序比较器
	 */
	public static <T> Comparator<T> getComparator(final boolean isAsc) {
		return new Comparator<T>() {

			public int compare(T value1, T value2) {
				return compareTo(isAsc, value1, value2);
			}
		};
	}

	/**
	 * 根据方法和排序方式获取Comparator，如果方法名称为空，则不使用反射机制
	 * @param isAsc 是否升序
	 * @param name 属性名称或方法名称(方法名称带括号)
	 * @return Comparator 排序比较器
	 */
	public static Comparator<Object> getComparator(final boolean isAsc, final String name) {
		return new Comparator<Object>() {

			public int compare(Object o1, Object o2) {
				Object value1 = null;
				Object value2 = null;
				if (AssertUtil.isEmpty(name)) {
					value1 = o1;
					value2 = o2;
				} else {
					// 利用反射得到具体的值
					value1 = ReflectUtil.getValue(o1, name);
					value2 = ReflectUtil.getValue(o2, name);
				}
				return compareTo(isAsc, value1, value2);
			}
		};
	}

	/**
	 * 根据方法名称列表和排序方式获取Comparator，方法名称列表不允许为空
	 * @param isAsc 是否升序
	 * @param names 属性名称或方法名称(方法名称带括号)列表，按先后顺序进行排序
	 * @return Comparator 排序比较器
	 */
	public static Comparator<Object> getComparator(boolean[] isAsc, String... names) {
		List<Comparator<Object>> comparatorList = new ArrayList<Comparator<Object>>();
		int index = isAsc.length - 1;
		for (int i = 0; i < names.length; i++) {
			if (i >= index) {
				comparatorList.add(getComparator(isAsc[index], names[i]));
			} else {
				comparatorList.add(getComparator(isAsc[i], names[i]));
			}
		}
		return getComparatorByComparatorList(comparatorList);
	}

	/**
	 * 根据Map的value排序方式获取Comparator
	 * @param <V>
	 * @param <K>
	 * @param <T>
	 * @param isAsc 是否升序
	 * @return Comparator 排序比较器
	 */
	@SuppressWarnings("unlikely-arg-type")
	public static <V, K, T> Comparator<T> getComparator(final Map<K, V> map, final boolean isAsc) {
		return new Comparator<T>() {

			public int compare(T o1, T o2) {
				return compareTo(isAsc, map.get(o1), map.get(o2));
			}
		};
	}

	/**
	 * 根据Map的value的方法和排序方式获取Comparator
	 * @param msgMaps 需要排序的map
	 * @param isAsc 是否升序
	 * @param name Map值对象的属性名称或方法名称(方法名称带括号)
	 * @return Comparator 排序比较器
	 */
	public static Comparator<Object> getComparator(final Map<?, ?> map, final boolean isAsc, final String name) {
		return new Comparator<Object>() {

			public int compare(Object o1, Object o2) {
				Object value1 = null;
				Object value2 = null;
				if (AssertUtil.isEmpty(name)) {
					value1 = o1;
					value2 = o2;
				} else {
					// 利用反射得到具体的值
					value1 = ReflectUtil.getValue(o1, name);
					value2 = ReflectUtil.getValue(o2, name);
				}
				return compareTo(isAsc, map.get(value1), map.get(value2));
			}
		};
	}

	/**
	 * 根据Map的value的方法名称列表和排序方式获取Comparator，方法名称列表不允许为空
	 * @param isAsc 是否升序
	 * @param names 值对象的属性名称或方法名称(方法名称带括号)列表，按先后顺序进行排序
	 * @return Comparator 排序比较器
	 */
	public static Comparator<Object> getComparator(final Map<?, ?> map, final boolean[] isAsc, String... names) {
		final List<Comparator<Object>> comparatorList = new ArrayList<Comparator<Object>>();
		int index = isAsc.length - 1;
		for (int i = 0; i < names.length; i++) {
			if (i >= index) {
				comparatorList.add(getComparator(map, isAsc[index], names[i]));
			} else {
				comparatorList.add(getComparator(map, isAsc[i], names[i]));
			}
		}
		return getComparatorByComparatorList(comparatorList);
	}

	/**
	 * 根据Map的value排序方式获取Comparator
	 * @param <K, V>
	 * @param isAsc 是否升序
	 * @return Comparator 排序比较器
	 */
	public static <K, V> Comparator<Entry<K, V>> getComparator(final Map<K, V> map, final boolean byValue,
			final boolean isAsc) {
		return new Comparator<Entry<K, V>>() {

			public int compare(Entry<K, V> o1, Entry<K, V> o2) {
				if (byValue) {
					return compareTo(isAsc, o1.getValue(), o2.getValue());
				} else {
					return compareTo(isAsc, o1.getKey(), o2.getKey());
				}
			}
		};
	}

	/**
	 * 根据Map的value的方法和排序方式获取Comparator
	 * @param <K, V>
	 * @param msgMaps 需要排序的map
	 * @param isAsc 是否升序
	 * @param name Map值对象的属性名称或方法名称(方法名称带括号)
	 * @return Comparator 排序比较器
	 */
	public static <K, V> Comparator<Entry<K, V>> getEntryComparator(final Map<K, V> map, final boolean byValue,
			final boolean isAsc, final String name) {
		return new Comparator<Entry<K, V>>() {

			public int compare(Entry<K, V> o1, Entry<K, V> o2) {
				Object value1 = null;
				Object value2 = null;
				if (byValue) {
					value1 = o1.getValue();
					value2 = o2.getValue();
				} else {
					value1 = o1.getKey();
					value2 = o2.getKey();
				}
				if (AssertUtil.isNotEmpty(name)) {
					// 利用反射得到具体的值
					value1 = ReflectUtil.getValue(value1, name);
					value2 = ReflectUtil.getValue(value2, name);
				}
				return compareTo(isAsc, value1, value2);
			}
		};
	}

	/**
	 * 根据Map的value的方法名称列表和排序方式获取Comparator，方法名称列表不允许为空
	 * @param <K, V>
	 * @param isAsc 是否升序
	 * @param names 值对象的属性名称或方法名称(方法名称带括号)列表，按先后顺序进行排序
	 * @return Comparator 排序比较器
	 */
	public static <K, V> Comparator<Entry<K, V>> getComparator(final Map<K, V> map, final boolean byValue,
			final boolean[] isAsc, String... names) {
		final List<Comparator<Entry<K, V>>> comparatorList = new ArrayList<Comparator<Entry<K, V>>>();
		int index = isAsc.length - 1;
		for (int i = 0; i < names.length; i++) {
			if (i >= index) {
				comparatorList.add(getEntryComparator(map, byValue, isAsc[index], names[i]));
			} else {
				comparatorList.add(getEntryComparator(map, byValue, isAsc[i], names[i]));
			}
		}
		return getEntryComparatorByComparatorList(comparatorList);
	}

	/**
	 * 根据排序条件获取Comparator，如果方法名称为空，则不使用反射机制
	 * @param condition 排序条件
	 * @return Comparator 排序比较器
	 */
	public static Comparator<Object> getComparator(final SortCondition condition) {
		return new Comparator<Object>() {

			public int compare(Object o1, Object o2) {
				Object value1 = null;
				Object value2 = null;

				String name = condition.getName();
				Map<?, ?> map = condition.getMap();
				if (map != null) { // 排序一个Map中的元素
					if (condition.isSortMapByValue()) { // 根据value排序
						value1 = map.get(o1);
						value2 = map.get(o2);
					} else { // 根据Key排序
						value1 = o1;
						value2 = o2;
					}
				} else if (AssertUtil.isEmpty(name)) { // 方法名称为空，直接比较两个对象
					value1 = o1;
					value2 = o2;
				} else if (condition.isTable()) {
					value1 = ReflectUtil.getTableColumnValue(o1, name, condition.isSupportMap());
					value2 = ReflectUtil.getTableColumnValue(o2, name, condition.isSupportMap());
				} else {
					value1 = ReflectUtil.getTreeColumnValue(o1, name, condition.isSupportMap());
					value2 = ReflectUtil.getTreeColumnValue(o2, name, condition.isSupportMap());
				}
				try {
					Format format = condition.getFormat();
					if (format != null) {
						if (condition.getParsePosition() != null) {
							value1 = format.parseObject(value1.toString(), condition.getParsePosition());
							value2 = format.parseObject(value2.toString(), condition.getParsePosition());
						} else {
							value1 = format.parseObject(value1.toString());
							value2 = format.parseObject(value2.toString());
						}
					}
					if (condition.getComparator() != null) {
						int result = condition.getComparator().compare(value1, value2);
						if (!condition.isAsc()) {
							result = -result;
						}
						return result;
					} else {
						return compareTo(condition.isAsc(), value1, value2);
					}
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
					return 0;
				}
			}

		};
	}

	/**
	 * 获取支持多个条件进行排序的比较器
	 * @param conditions 条件集，按list先后顺序进行排序
	 * @return Comparator 支持多个条件排序的比较器
	 */
	public static Comparator<Object> getComparator(List<SortCondition> conditions) {
		final List<Comparator<Object>> comparatorList = new ArrayList<Comparator<Object>>();
		for (SortCondition sortCondition : conditions) {
			comparatorList.add(getComparator(sortCondition));
		}
		return getComparatorByComparatorList(comparatorList);
	}

	/**
	 * 获取支持多个条件进行排序的比较器
	 * @param comparatorList 排序比较器列表，按list先后顺序进行排序
	 * @return Comparator 支持多个条件排序的比较器
	 */
	public static Comparator<Object> getComparatorByComparatorList(final List<Comparator<Object>> comparatorList) {
		return new Comparator<Object>() {

			public int compare(Object o1, Object o2) {
				for (Comparator<Object> c : comparatorList) {
					int result = c.compare(o1, o2);
					if (result != 0) {
						return result;
					}
				}
				return 0;
			}
		};
	}

	/**
	 * 获取支持多个条件进行排序的比较器
	 * @param <K, V>
	 * @param comparatorList 排序比较器列表，按list先后顺序进行排序
	 * @return Comparator 支持多个条件排序的比较器
	 */
	public static <K, V> Comparator<Entry<K, V>> getEntryComparatorByComparatorList(
			final List<Comparator<Entry<K, V>>> comparatorList) {
		return new Comparator<Entry<K, V>>() {

			public int compare(Entry<K, V> o1, Entry<K, V> o2) {
				for (Comparator<Entry<K, V>> c : comparatorList) {
					int result = c.compare(o1, o2);
					if (result != 0) {
						return result;
					}
				}
				return 0;
			}
		};
	}

	/**
	 * 比较两个对象的排列次序
	 * @param isAsc
	 * @param object1
	 * @param object2
	 * @return
	 */
	public static int compareTo(boolean isAsc, Object object1, Object object2) {
		int result = 0;
		try {
			result = compareTo(object1, object2);
			if (!isAsc) {
				result = -result;
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * 比较两个对象的排列次序
	 * @param object1
	 * @param object2
	 * @return
	 */
	public static int compareTo(Object object1, Object object2) {
		boolean isEmptyV1 = AssertUtil.isEmpty(object1);
		boolean isEmptyV2 = AssertUtil.isEmpty(object2);
		if (!isEmptyV1 && !isEmptyV2) {
			String value1 = object1.toString();
			String value2 = object2.toString();
			if (object1 instanceof Date) {
				return ((Date) object1).compareTo((Date) object2);
			} else if (object1 instanceof Number || (isNumeric(value1) && isNumeric(value2))) {
				try {
					return new BigDecimal(value1).compareTo(new BigDecimal(value2));
				} catch (Exception e) {
					return value1.compareTo(value2);
				}
			} else {
				return value1.compareTo(value2);
			}
		} else if (!isEmptyV1 && isEmptyV2) {
			return 1;
		} else if (isEmptyV1 && !isEmptyV2) {
			return -1;
		} else {
			return 0;
		}
	}

	/**
	 * 判断字符串的内容是不是全是数字
	 * @param str
	 * @return
	 */
	public static boolean isNumeric(String str) {
		if (str == null) {
			return false;
		}
		int sz = str.length();
		for (int i = 0; i < sz; i++) {
			if (!Character.isDigit(str.charAt(i))) {
				return false;
			}
		}
		return true;
	}

}
