package cn.elwy.common.util.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import cn.elwy.common.util.ConvertTypeUtil;

/**
 * 属性文件工具类，读取属性、保存属性
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class PropertyUtil {

	private Property property = new Property();
	private File file;

	public PropertyUtil() {
	}

	/**
	 * 加载配置文件
	 * @param filePath
	 * @throws IOException
	 */
	public void load(String filePath) throws IOException {
		load(new File(filePath));
	}

	/**
	 * 加载配置文件
	 * @param filePath
	 * @throws IOException
	 */
	public void load(String filePath, String encoding) throws IOException {
		load(new File(filePath), encoding);
	}

	/**
	 * 加载配置文件
	 * @param file
	 * @throws IOException
	 */
	public void load(File file) throws IOException {
		property.load(file);
	}

	/**
	 * 加载配置文件
	 * @param file
	 * @throws IOException
	 */
	public void load(File file, String encoding) throws IOException {
		property.load(file, encoding);
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream) throws IOException {
		property.load(inputStream);
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @param encoding
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream, String encoding) throws IOException {
		property.load(inputStream, encoding);
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @param encoding
	 * @param fileType
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream, String encoding, String fileType) throws IOException {
		property.load(inputStream, encoding, fileType);
	}

	/**
	 * 获取属性的值，如果值不存在将返回""
	 * @param key 关键字
	 * @return
	 */
	public String get(String key) {
		return get(key, "");
	}

	/**
	 * 获取属性的值，如果值不存在将返回""
	 * @param key 关键字
	 * @param defaultValue
	 * @return
	 */
	public String get(String key, String defaultValue) {
		String value = property.getProperty(key);
		return value == null ? defaultValue : value;
	}

	/**
	 * 根据分隔符获取属性的值，以数组的形式返回
	 * @param key 关键字
	 * @return
	 */
	public String[] getArray(String key) {
		return ConvertTypeUtil.toArray(get(key));
	}

	/**
	 * 根据分隔符获取属性的值，以数组的形式返回
	 * @param key 关键字
	 * @param separate 分隔符
	 * @return
	 */
	public String[] getArray(String key, String separate) {
		return ConvertTypeUtil.toArray(get(key), separate);
	}

	/**
	 * 获取属性的值，如果值不存在将返回0
	 * @param key 关键字
	 * @return
	 */
	public int getInt(String key) {
		return getInt(key, 0);
	}

	/**
	 * 获取属性的值，如果值不存在将返回defaultValue
	 * @param key 关键字
	 * @param defaultValue
	 * @return
	 */
	public int getInt(String key, int defaultValue) {
		return ConvertTypeUtil.toInteger(get(key), defaultValue);
	}

	/**
	 * 获取属性的值，如果值不存在将返回false
	 * @param key 关键字
	 * @return
	 */
	public boolean getBoolean(String key) {
		return getBoolean(key, false);
	}

	/**
	 * 获取属性的值，如果值不存在将返回defaultValue
	 * @param key 关键字
	 * @param defaultValue
	 * @return
	 */
	public boolean getBoolean(String key, boolean defaultValue) {
		return ConvertTypeUtil.toBoolean(get(key), defaultValue);
	}

	/**
	 * 设置属性值，如果属性值为null，则不设置相应的属性
	 * @param key
	 * @param object
	 */
	public void setProperty(String key, Object object) {
		setProperty(key, object, null);
	}

	/**
	 * 设置属性值，如果属性值为null，则不设置相应的属性
	 * @param key
	 * @param object
	 * @param defaultValue
	 */
	public void setProperty(String key, Object object, String defaultValue) {
		String value = ConvertTypeUtil.toString(object, defaultValue);
		if (value != null) {
			property.put(key, value);
		}
	}

	public Property getProperty() {
		return property;
	}

	public Properties getProps() {
		return property.getProps();
	}

	/**
	 * 保存属性文件
	 * @throws IOException
	 */
	public boolean save() throws IOException {
		return save(file, null, null);
	}

	/**
	 * 保存属性文件
	 * @param filePath
	 * @throws IOException
	 */
	public boolean save(String filePath) throws IOException {
		return save(new File(filePath), null, null);
	}

	/**
	 * 保存属性文件
	 * @param filePath
	 * @param charset
	 * @param comments
	 * @throws IOException
	 */
	public boolean save(String filePath, String charset, String comments) throws IOException {
		return save(new File(filePath), charset, comments);
	}

	/**
	 * 保存属性文件
	 * @param file
	 * @throws IOException
	 */
	public boolean save(File file) throws IOException {
		return property.save(file, null, null);
	}

	/**
	 * 保存属性文件
	 * @param file
	 * @param charset
	 * @param comments
	 * @throws IOException
	 */
	public boolean save(File file, String charset, String comments) throws IOException {
		return property.save(file, charset, comments);
	}

	public static String getExtName(String path) {
		return Property.getExtName(path);
	}

}
