package cn.elwy.common.util.io;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Serializable;
import java.io.StringReader;
import java.io.Writer;
import java.util.Enumeration;
import java.util.InvalidPropertiesFormatException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import cn.elwy.common.util.Assert;
import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.CloseUtil;

/**
 * 扩展Properties类，添加支持注释的保存和保留文件原顺序的功能
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class Property implements Serializable {
	private static final long serialVersionUID = 1;

	private static final String BLANK = "";

	/**
	 * 保存key与comment的映射， 同时利用这个映射来保证key的顺序。
	 */
	private final LinkedHashMap<String, String> keyCommentMap = new LinkedHashMap<String, String>();
	private final Properties property;
	private boolean escUnicode = true;
	private boolean commentEscUnicode;
	private String defaultEncoding;

	public Property() {
		this(null);
	}

	public Property(Properties defaults) {
		super();
		property = new Properties(defaults);
	}

	/**
	 * 设置一个属性，如果key已经存在，那么将其对应value值覆盖。
	 * @param key
	 * @param value
	 * @return
	 */
	public String setProperty(String key, String value) {
		return setProperty(key, value, BLANK);
	}

	/**
	 * 设置一个属性，如果key已经存在，那么将其对应value值覆盖。
	 * @param key 键
	 * @param value 与键对应的值
	 * @param comment 对键值对的说明
	 * @return
	 */
	public synchronized String setProperty(String key, String value, String comment) {
		return (String) put(key, value, comment);
	}

	/**
	 * 设置一个属性，如果key已经存在，那么将其对应value值覆盖。
	 * @param key
	 * @param value
	 * @return
	 */
	public Object put(String key, Object value) {
		return put(key, value, BLANK);
	}

	/**
	 * 设置一个属性，如果key已经存在，那么将其对应value值覆盖。
	 * @param key 键
	 * @param value 与键对应的值
	 * @param comment 对键值对的说明
	 * @return
	 */
	public synchronized Object put(String key, Object value, String comment) {
		if (value == null) {
			System.err.println("key=" + key + " value is null");
			return null;
		}
		Object oldValue = property.put(key, value.toString());
		if (BLANK.equals(comment)) {
			if (!keyCommentMap.containsKey(key)) {
				keyCommentMap.put(key, comment);
			}
		} else {
			keyCommentMap.put(key, comment);
		}
		return (String) oldValue;
	}

	/**
	 * 根据key获取属性表中相应的value。
	 * @param key
	 * @return
	 */
	public String getProperty(String key) {
		return property.getProperty(key);
	}

	/**
	 * 根据key获取属性表中相应的value。 如果没找到相应的value，返回defaultValue。
	 * @param key
	 * @param defaultValue
	 * @return
	 */
	public String getProperty(String key, String defaultValue) {
		return property.getProperty(key, defaultValue);
	}

	/**
	 * 从一个文件中读取属性到属性表中 默认字符集为utf-8
	 * @param file 属性文件
	 * @throws IOException
	 */
	public synchronized void load(File file) throws IOException {
		defaultEncoding = FileUtil.getFileEncoding(file);
		load(file, defaultEncoding);
	}

	/**
	 * 从一个文件中读取属性到属性表中
	 * @param file 属性文件
	 * @param encoding 字符集
	 * @throws IOException
	 */
	public synchronized void load(File file, String encoding) throws IOException {
		// FileInputStream inputStream = new FileInputStream(file);
		// InputStreamReader reader = new InputStreamReader(inputStream, encoding);
		// load0(new LineReader(reader));
		FileInputStream inputStream = null;
		try {
			String fileType = getExtName(file.getName());
			inputStream = new FileInputStream(file);
			load(inputStream, encoding, fileType);
		} catch (Exception e) {
			throw new IOException("读取属性文件[" + file.getPath() + "]出错！", e);
		} finally {
			CloseUtil.close(inputStream);
		}
	}

	/**
	 * 从一个字符流中读取属性到属性表中
	 * @param reader
	 * @throws IOException
	 */
	public synchronized void load(Reader reader) throws IOException {
		load0(new LineReader(reader));
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream) throws IOException {
		load(inputStream, "utf-8");
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @param encoding
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream, String encoding) throws IOException {
		defaultEncoding = encoding;
		UnicodeReader reader = new UnicodeReader(inputStream, encoding);
		load0(new LineReader(reader));
	}

	/**
	 * 从一个字节流中读取属性到属性表中
	 * @param inputStream
	 * @param encoding
	 * @param fileType
	 * @throws IOException
	 */
	public synchronized void load(InputStream inputStream, String encoding, String fileType) throws IOException {
		defaultEncoding = encoding;
		if (fileType != null && "xml".equalsIgnoreCase(fileType)) {
			loadFromXML(inputStream);
		} else {
			load(inputStream, encoding);
		}
	}

	private void load0(LineReader lr) throws IOException {
		char[] convtBuf = new char[1024];
		int limit;
		int keyLen;
		int valueStart;
		char c;
		boolean hasSep;
		boolean precedingBackslash;
		StringBuffer buffer = new StringBuffer();

		while ((limit = lr.readLine()) >= 0) {
			c = 0;
			keyLen = 0;
			valueStart = limit;
			hasSep = false;
			// 获取注释
			c = lr.lineBuf[keyLen];
			if (c == '#' || c == '!') {
				String comment = loadConvert(lr.lineBuf, 1, limit - 1, convtBuf);
				if (buffer.length() > 0) {
					buffer.append("\r\n");
				}
				comment = comment.replaceAll("\n", "\\\\n");
				comment = comment.replaceAll("\r", "\\\\r");
				comment = comment.replaceAll("\t", "\\\\t");
				comment = comment.replaceAll("\f", "\\\\f");
				buffer.append(comment);
				continue;
			}
			precedingBackslash = false;
			while (keyLen < limit) {
				c = lr.lineBuf[keyLen];
				// need check if escaped.
				if ((c == '=' || c == ':') && !precedingBackslash) {
					valueStart = keyLen + 1;
					hasSep = true;
					break;
				} else if ((c == ' ' || c == '\t' || c == '\f') && !precedingBackslash) {
					valueStart = keyLen + 1;
					break;
				}
				if (c == '\\') {
					precedingBackslash = !precedingBackslash;
				} else {
					precedingBackslash = false;
				}
				keyLen++;
			}
			while (valueStart < limit) {
				c = lr.lineBuf[valueStart];
				if (c != ' ' && c != '\t' && c != '\f') {
					if (!hasSep && (c == '=' || c == ':')) {
						hasSep = true;
					} else {
						break;
					}
				}
				valueStart++;
			}
			String key = loadConvert(lr.lineBuf, 0, keyLen, convtBuf);
			String value = loadConvert(lr.lineBuf, valueStart, limit - valueStart, convtBuf);
			setProperty(key, value, buffer.toString());
			// reset buffer
			buffer = new StringBuffer();
		}
	}

	class LineReader {
		public LineReader(InputStream inStream) {
			this.inStream = inStream;
			inByteBuf = new byte[8192];
		}

		public LineReader(Reader reader) {
			this.reader = reader;
			inCharBuf = new char[8192];
		}

		byte[] inByteBuf;
		char[] inCharBuf;
		char[] lineBuf = new char[1024];
		int inLimit = 0;
		int inOff = 0;
		InputStream inStream;
		Reader reader;

		int readLine() throws IOException {
			int len = 0;
			char c = 0;

			boolean skipWhiteSpace = true;
			boolean isNewLine = true;
			boolean appendedLineBegin = false;
			boolean precedingBackslash = false;
			boolean skipLF = false;

			while (true) {
				if (inOff >= inLimit) {
					inLimit = (inStream == null) ? reader.read(inCharBuf) : inStream.read(inByteBuf);
					inOff = 0;
					if (inLimit <= 0) {
						if (len == 0) {
							return -1;
						}
						return len;
					}
				}
				if (inStream != null) {
					// The line below is equivalent to calling a
					// ISO8859-1 decoder.
					c = (char) (0xff & inByteBuf[inOff++]);
				} else {
					c = inCharBuf[inOff++];
				}
				if (skipLF) {
					skipLF = false;
					if (c == '\n') {
						continue;
					}
				}
				if (skipWhiteSpace) {
					if (c == ' ' || c == '\t' || c == '\f') {
						continue;
					}
					if (!appendedLineBegin && (c == '\r' || c == '\n')) {
						continue;
					}
					skipWhiteSpace = false;
					appendedLineBegin = false;
				}
				if (isNewLine) {
					isNewLine = false;
				}

				if (c != '\n' && c != '\r') {
					lineBuf[len++] = c;
					if (len == lineBuf.length) {
						int newLength = lineBuf.length * 2;
						if (newLength < 0) {
							newLength = Integer.MAX_VALUE;
						}
						char[] buf = new char[newLength];
						System.arraycopy(lineBuf, 0, buf, 0, lineBuf.length);
						lineBuf = buf;
					}
					// flip the preceding backslash flag
					if (c == '\\') {
						precedingBackslash = !precedingBackslash;
					} else {
						precedingBackslash = false;
					}
				} else {
					// reached EOL
					if (len == 0) {
						isNewLine = true;
						skipWhiteSpace = true;
						len = 0;
						continue;
					}
					if (inOff >= inLimit) {
						inLimit = (inStream == null) ? reader.read(inCharBuf) : inStream.read(inByteBuf);
						inOff = 0;
						if (inLimit <= 0) {
							return len;
						}
					}
					if (precedingBackslash) {
						len -= 1;
						// skip the leading whitespace characters in following line
						skipWhiteSpace = true;
						appendedLineBegin = true;
						precedingBackslash = false;
						if (c == '\r') {
							skipLF = true;
						}
					} else {
						return len;
					}
				}
			}
		}
	}

	/*
	 * Converts encoded &#92;uxxxx to unicode chars and changes special saved chars
	 * to their original forms
	 */
	private String loadConvert(char[] in, int off, int len, char[] convtBuf) {
		if (convtBuf.length < len) {
			int newLen = len * 2;
			if (newLen < 0) {
				newLen = Integer.MAX_VALUE;
			}
			convtBuf = new char[newLen];
		}
		char aChar;
		char[] out = convtBuf;
		int outLen = 0;
		int end = off + len;

		while (off < end) {
			aChar = in[off++];
			if (aChar == '\\') {
				aChar = in[off++];
				if (aChar == 'u') {
					// Read the xxxx
					int value = 0;
					for (int i = 0; i < 4; i++) {
						aChar = in[off++];
						switch (aChar) {
						case '0':
						case '1':
						case '2':
						case '3':
						case '4':
						case '5':
						case '6':
						case '7':
						case '8':
						case '9':
							value = (value << 4) + aChar - '0';
							break;
						case 'a':
						case 'b':
						case 'c':
						case 'd':
						case 'e':
						case 'f':
							value = (value << 4) + 10 + aChar - 'a';
							break;
						case 'A':
						case 'B':
						case 'C':
						case 'D':
						case 'E':
						case 'F':
							value = (value << 4) + 10 + aChar - 'A';
							break;
						default:
							throw new IllegalArgumentException("Malformed \\uxxxx encoding.");
						}
					}
					out[outLen++] = (char) value;
				} else {
					if (aChar == 't')
						aChar = '\t';
					else if (aChar == 'r')
						aChar = '\r';
					else if (aChar == 'n')
						aChar = '\n';
					else if (aChar == 'f')
						aChar = '\f';
					out[outLen++] = aChar;
				}
			} else {
				out[outLen++] = (char) aChar;
			}
		}
		return new String(out, 0, outLen);
	}

	private String saveConvert(String theString, boolean escapeSpace, boolean escapeUnicode) {
		if (theString == null || theString.isEmpty()) {
			return BLANK;
		}
		int len = theString.length();
		int bufLen = len * 2;
		if (bufLen < 0) {
			bufLen = Integer.MAX_VALUE;
		}
		StringBuffer outBuffer = new StringBuffer(bufLen);

		for (int x = 0; x < len; x++) {
			char aChar = theString.charAt(x);
			// Handle common case first, selecting largest block that
			// avoids the specials below
			if ((aChar > 61) && (aChar < 127)) {
				if (aChar == '\\') {
					outBuffer.append('\\');
					outBuffer.append('\\');
					continue;
				}
				outBuffer.append(aChar);
				continue;
			}
			switch (aChar) {
			case ' ':
				if (x == 0 || escapeSpace)
					outBuffer.append('\\');
				outBuffer.append(' ');
				break;
			case '\t':
				outBuffer.append('\\');
				outBuffer.append('t');
				break;
			case '\n':
				outBuffer.append('\\');
				outBuffer.append('n');
				break;
			case '\r':
				outBuffer.append('\\');
				outBuffer.append('r');
				break;
			case '\f':
				outBuffer.append('\\');
				outBuffer.append('f');
				break;
			case '=': // Fall through
			case ':': // Fall through
			case '#': // Fall through
			case '!':
				outBuffer.append('\\');
				outBuffer.append(aChar);
				break;
			default:
				if (((aChar < 0x0020) || (aChar > 0x007e)) & escapeUnicode) {
					outBuffer.append('\\');
					outBuffer.append('u');
					outBuffer.append(toHex((aChar >> 12) & 0xF));
					outBuffer.append(toHex((aChar >> 8) & 0xF));
					outBuffer.append(toHex((aChar >> 4) & 0xF));
					outBuffer.append(toHex(aChar & 0xF));
				} else {
					outBuffer.append(aChar);
				}
			}
		}
		return outBuffer.toString();
	}

	private void writeComments(BufferedWriter bw, String comments) throws IOException {
		bw.write("#");
		int len = comments.length();
		int current = 0;
		int last = 0;
		char[] uu = new char[6];
		uu[0] = '\\';
		uu[1] = 'u';
		while (current < len) {
			char c = comments.charAt(current);
			if (c > '\u00ff' || c == '\n' || c == '\r') {
				if (last != current)
					bw.write(comments.substring(last, current));
				if (c > '\u00ff') {
					if (commentEscUnicode) {
						uu[2] = toHex((c >> 12) & 0xf);
						uu[3] = toHex((c >> 8) & 0xf);
						uu[4] = toHex((c >> 4) & 0xf);
						uu[5] = toHex(c & 0xf);
						bw.write(new String(uu));
					} else {
						bw.write(c);
					}
				} else {
					bw.newLine();
					if (c == '\r' && current != len - 1 && comments.charAt(current + 1) == '\n') {
						current++;
					}
					if (current == len - 1 || (comments.charAt(current + 1) != '#' && comments.charAt(current + 1) != '!'))
						bw.write("#");
				}
				last = current + 1;
			}
			current++;
		}
		if (last != current)
			bw.write(comments.substring(last, current));
		bw.newLine();
	}

	/**
	 * 保存属性文件
	 * @param filePath
	 * @throws IOException
	 */
	public boolean save(String filePath) throws IOException {
		return save(new File(filePath), defaultEncoding, null);
	}

	/**
	 * 保存属性文件
	 * @param filePath
	 * @param encoding
	 * @throws IOException
	 */
	public boolean save(String filePath, String encoding) throws IOException {
		return save(new File(filePath), defaultEncoding, encoding);
	}

	/**
	 * 保存属性文件
	 * @param filePath
	 * @param encoding
	 * @param comments
	 * @throws IOException
	 */
	public boolean save(String filePath, String encoding, String comments) throws IOException {
		return save(new File(filePath), encoding, comments);
	}

	/**
	 * 保存属性文件
	 * @param file
	 * @throws IOException
	 */
	public boolean save(File file) throws IOException {
		return save(file, defaultEncoding, null);
	}

	/**
	 * 保存属性文件
	 * @param file
	 * @param encoding
	 * @throws IOException
	 */
	public boolean save(File file, String encoding) throws IOException {
		return save(file, encoding, null);
	}

	/**
	 * 保存属性文件
	 * @param file
	 * @param encoding
	 * @param comments
	 * @throws IOException
	 */
	public boolean save(File file, String encoding, String comments) throws IOException {
		FileOutputStream outStream = null;
		Assert.notNull(file);
		try {
			file.getParentFile().mkdirs();

			outStream = new FileOutputStream(file);
			String fileType = getExtName(file.getName());
			if (fileType != null && "xml".equalsIgnoreCase(fileType)) {
				storeToXML(outStream, comments, encoding);
			} else {
				store(outStream, encoding, comments);// 保存属性文件
			}
			return true;
		} finally {
			CloseUtil.close(outStream);
		}
	}

	/**
	 * 将属性表中的属性写到字符流里面。
	 * @param writer
	 * @param commons
	 * @throws IOException
	 */
	public void store(Writer writer, String comments) throws IOException {
		store0((writer instanceof BufferedWriter) ? (BufferedWriter) writer : new BufferedWriter(writer), comments,
				escUnicode);
	}

	public void store(OutputStream out, String encoding, String comments) throws IOException {
		if (encoding == null || "".equals(encoding)) {
			encoding = "utf-8";
		}
		store0(new BufferedWriter(new OutputStreamWriter(out, encoding)), comments, escUnicode);
	}

	private void store0(BufferedWriter bw, String comments, boolean escUnicode) throws IOException {
		if (comments != null) {
			writeComments(bw, comments);
			bw.newLine();
		}
		synchronized (this) {
			// for (Enumeration e = keys(); e.hasMoreElements();) {
			// String key = (String) e.nextElement();
			// String val = (String) get(key);
			Iterator<Map.Entry<String, String>> kvIter = keyCommentMap.entrySet().iterator();
			while (kvIter.hasNext()) {
				Map.Entry<String, String> entry = kvIter.next();
				String key = entry.getKey();
				String val = getProperty(key);
				String comment = entry.getValue();
				key = saveConvert(key, true, escUnicode);
				/*
				 * No need to escape embedded and trailing spaces for value, hence pass false to
				 * flag.
				 */
				val = saveConvert(val, false, escUnicode);
				if (!comment.equals(BLANK)) {
					// comment = saveConvert(comment, false, commentEscUnicode);
					writeComments(bw, comment);
				}
				bw.write(key + "=" + val);
				bw.newLine();
			}
		}
		bw.flush();
	}

	public synchronized void loadFromXML(InputStream in) throws IOException, InvalidPropertiesFormatException {
		if (in == null)
			throw new NullPointerException();
		try {
			XMLUtils.load(property, in);
		} finally {
			CloseUtil.close(in);
		}
	}

	public synchronized void storeToXML(OutputStream os, String comment) throws IOException {
		if (os == null)
			throw new NullPointerException();
		storeToXML(os, comment, "UTF-8");
	}

	public synchronized void storeToXML(OutputStream os, String comment, String encoding) throws IOException {
		if (os == null)
			throw new NullPointerException();
		XMLUtils.save(property, os, comment, encoding);
	}

	/**
	 * 如果属性表中某个key对应的value值和参数value相同 那么返回true，否则返回false。
	 * @param value
	 * @return
	 */
	public boolean containsValue(String value) {
		return property.containsValue(value);
	}

	/**
	 * 如果属性表中存在参数key，返回true，否则返回false。
	 * @param key
	 * @return
	 */
	public boolean containsKey(String key) {
		return property.containsKey(key);
	}

	public Enumeration<?> propertyNames() {
		return property.propertyNames();
	}

	public Set<String> stringPropertyNames() {
		return property.stringPropertyNames();
	}

	public Set<Object> keySet() {
		return property.keySet();
	}

	/**
	 * 获取属性表中键值对数量
	 * @return
	 */
	public int size() {
		return property.size();
	}

	/**
	 * 检查属性表是否为空
	 * @return
	 */
	public boolean isEmpty() {
		return property.isEmpty();
	}

	/**
	 * 清空属性表
	 */
	public synchronized void clear() {
		property.clear();
		keyCommentMap.clear();
	}

	public void list(PrintStream out) {
		property.list(out);
	}

	public void list(PrintWriter out) {
		property.list(out);
	}

	public synchronized String toString() {
		StringBuffer buffer = new StringBuffer();
		Iterator<Map.Entry<String, String>> kvIter = keyCommentMap.entrySet().iterator();
		buffer.append("[");
		while (kvIter.hasNext()) {
			buffer.append("{");
			Map.Entry<String, String> entry = kvIter.next();
			String key = entry.getKey();
			String val = getProperty(key);
			String comment = entry.getValue();
			buffer.append("key=" + key + ",value=" + val + ",comment=" + comment);
			buffer.append("}");
		}
		buffer.append("]");
		return buffer.toString();
	}

	public boolean equals(Object o) {
		// 不考虑注释说明是否相同
		return property.equals(o);
	}

	public int hashCode() {
		return property.hashCode();
	}

	public boolean isEscUnicode() {
		return escUnicode;
	}

	public void setEscUnicode(boolean escUnicode) {
		this.escUnicode = escUnicode;
	}

	public boolean isCommentEscUnicode() {
		return commentEscUnicode;
	}

	public void setCommentEscUnicode(boolean commentEscUnicode) {
		this.commentEscUnicode = commentEscUnicode;
	}

	public Properties getProps() {
		return property;
	}

	/**
	 * 获取文件扩展名
	 * @param path 文件名称或路径
	 * @return
	 */
	public static String getExtName(String path) {
		if (AssertUtil.isNotEmpty(path)) {
			int index = path.lastIndexOf(".");
			String extName = path.substring(index + 1);
			return extName;
		} else {
			return "";
		}
	}

	/**
	 * Convert a nibble to a hex character
	 * @param nibble the nibble to convert.
	 */
	private static char toHex(int nibble) {
		return hexDigit[(nibble & 0xF)];
	}

	/** A table of hex digits */
	private static final char[] hexDigit = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E',
			'F' };

}

/**
 * A class used to aid in Properties load and save in XML. Keeping this code
 * outside of Properties helps reduce the number of classes loaded when
 * Properties is loaded.
 * @author Michael McCloskey
 */
class XMLUtils {

	// XML loading and saving methods for Properties

	// The required DTD URI for exported properties
	private static final String PROPS_DTD_URI = "http://java.sun.com/dtd/properties.dtd";

	private static final String PROPS_DTD = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + "<!-- DTD for properties -->"
			+ "<!ELEMENT properties ( comment?, entry* ) >" + "<!ATTLIST properties" + " version CDATA #FIXED \"1.0\">"
			+ "<!ELEMENT comment (#PCDATA) >" + "<!ELEMENT entry (#PCDATA) >" + "<!ATTLIST entry " + " key CDATA #REQUIRED>";

	/**
	 * Version number for the format of exported properties files.
	 */
	private static final String EXTERNAL_XML_VERSION = "1.0";

	static void load(Properties props, InputStream in) throws IOException, InvalidPropertiesFormatException {
		Document doc = null;
		try {
			doc = getLoadingDoc(in);
		} catch (SAXException saxe) {
			throw new InvalidPropertiesFormatException(saxe);
		}
		Element propertiesElement = doc.getDocumentElement();
		String xmlVersion = propertiesElement.getAttribute("version");
		if (xmlVersion.compareTo(EXTERNAL_XML_VERSION) > 0)
			throw new InvalidPropertiesFormatException("Exported Properties file format version " + xmlVersion
					+ " is not supported. This java installation can read" + " versions " + EXTERNAL_XML_VERSION
					+ " or older. You" + " may need to install a newer version of JDK.");
		importProperties(props, propertiesElement);
	}

	static Document getLoadingDoc(InputStream in) throws SAXException, IOException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setValidating(true);
		dbf.setCoalescing(true);
		dbf.setIgnoringComments(true);
		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			db.setEntityResolver(new Resolver());
			db.setErrorHandler(new EH());
			InputSource is = new InputSource(in);
			return db.parse(is);
		} catch (ParserConfigurationException x) {
			throw new Error(x);
		}
	}

	static void importProperties(Properties props, Element propertiesElement) {
		NodeList entries = propertiesElement.getChildNodes();
		int numEntries = entries.getLength();
		int start = numEntries > 0 && entries.item(0).getNodeName().equals("comment") ? 1 : 0;
		for (int i = start; i < numEntries; i++) {
			Element entry = (Element) entries.item(i);
			if (entry.hasAttribute("key")) {
				Node n = entry.getFirstChild();
				String val = (n == null) ? "" : n.getNodeValue();
				props.setProperty(entry.getAttribute("key"), val);
			}
		}
	}

	static void save(Properties props, OutputStream os, String comment, String encoding) throws IOException {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		DocumentBuilder db = null;
		try {
			db = dbf.newDocumentBuilder();
		} catch (ParserConfigurationException pce) {
			assert (false);
		}
		Document doc = db.newDocument();
		Element properties = (Element) doc.appendChild(doc.createElement("properties"));

		if (comment != null) {
			Element comments = (Element) properties.appendChild(doc.createElement("comment"));
			comments.appendChild(doc.createTextNode(comment));
		}

		Set<?> keys = props.keySet();
		Iterator<?> i = keys.iterator();
		while (i.hasNext()) {
			String key = (String) i.next();
			Element entry = (Element) properties.appendChild(doc.createElement("entry"));
			entry.setAttribute("key", key);
			entry.appendChild(doc.createTextNode(props.getProperty(key)));
		}
		emitDocument(doc, os, encoding);
	}

	static void emitDocument(Document doc, OutputStream os, String encoding) throws IOException {
		TransformerFactory tf = TransformerFactory.newInstance();
		Transformer t = null;
		try {
			t = tf.newTransformer();
			t.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, PROPS_DTD_URI);
			t.setOutputProperty(OutputKeys.INDENT, "yes");
			t.setOutputProperty(OutputKeys.METHOD, "xml");
			t.setOutputProperty(OutputKeys.ENCODING, encoding);
		} catch (TransformerConfigurationException tce) {
			assert (false);
		}
		DOMSource doms = new DOMSource(doc);
		StreamResult sr = new StreamResult(os);
		try {
			t.transform(doms, sr);
		} catch (TransformerException te) {
			IOException ioe = new IOException();
			ioe.initCause(te);
			throw ioe;
		}
	}

	private static class Resolver implements EntityResolver {
		public InputSource resolveEntity(String pid, String sid) throws SAXException {
			if (sid.equals(PROPS_DTD_URI)) {
				InputSource is;
				is = new InputSource(new StringReader(PROPS_DTD));
				is.setSystemId(PROPS_DTD_URI);
				return is;
			}
			throw new SAXException("Invalid system identifier: " + sid);
		}
	}

	private static class EH implements ErrorHandler {
		public void error(SAXParseException x) throws SAXException {
			throw x;
		}

		public void fatalError(SAXParseException x) throws SAXException {
			throw x;
		}

		public void warning(SAXParseException x) throws SAXException {
			throw x;
		}
	}

}
