package cn.elwy.common.util.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import cn.elwy.common.util.CloseUtil;

/**
 * Java读取Jar包中的文件的工具类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class JarUtil {

	private JarUtil() {
	}

	/**
	 * 读取Jar包中的文本文件，将文件内容作为字符串返回
	 * @param jarPath jar包的路径
	 * @param fileName 相对于包根目录的路径
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(String jarPath, String fileName, String charsetName) throws IOException {
		JarFile jarFile = null;
		try {
			jarFile = new JarFile(jarPath);
			JarEntry entry = jarFile.getJarEntry(fileName);
			InputStream inputStream = jarFile.getInputStream(entry);// 读入需要的文件
			return FileUtil.readInputStreamToString(inputStream, charsetName);
		} finally {
			CloseUtil.close(jarFile);
		}
	}

	/**
	 * 读取Jar包中的文本文件，返回按行读取的列表
	 * @param jarPath jar包的路径
	 * @param fileName 相对于包根目录的路径
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readFileToList(String jarPath, String fileName, String charsetName) throws IOException {
		JarFile jarFile = null;
		try {
			jarFile = new JarFile(jarPath);
			JarEntry entry = jarFile.getJarEntry(fileName);
			InputStream inputStream = jarFile.getInputStream(entry);// 读入需要的文件
			return FileUtil.readInputStreamToList(inputStream, charsetName);
		} finally {
			CloseUtil.close(jarFile);
		}
	}

	/**
	 * 读取Jar包中的MANIFEST.MF文件，返回Manifest对象
	 * @param jarPath jar包的路径
	 * @return Manifest对象
	 * @throws IOException
	 */
	public static Manifest readFileToManifest(String jarPath) throws IOException {
		JarFile jarFile = null;
		try {
			jarFile = new JarFile(jarPath);
			return jarFile.getManifest();
		} finally {
			CloseUtil.close(jarFile);
		}
	}

	/**
	 * 读取Jar包中的序列化文件，返回序列化对象
	 * @param jarPath jar包的路径
	 * @param fileName 相对于包根目录的路径
	 * @return 序列化对象
	 * @throws ClassNotFoundException
	 * @throws IOException
	 */
	@SuppressWarnings("unchecked")
	public static <T> T readFileToObject(String jarPath, String fileName) throws ClassNotFoundException, IOException {
		JarFile jarFile = null;
		ObjectInputStream ois = null;
		InputStream inputStream = null;
		try {
			jarFile = new JarFile(jarPath);
			JarEntry entry = jarFile.getJarEntry(fileName);
			inputStream = jarFile.getInputStream(entry);
			ois = new ObjectInputStream(inputStream);
			return (T) ois.readObject();
		} finally {
			CloseUtil.close(inputStream);
			CloseUtil.close(ois);
			CloseUtil.close(jarFile);
		}
	}

	/**
	 * 读取Jar包中的文件，以byte方式返回
	 * @param jarPath jar包的路径
	 * @param fileName 相对于包根目录的路径
	 * @return byte 数组
	 * @throws IOException
	 */
	public static byte[] readFileToByte(String jarPath, String fileName) throws IOException {
		JarFile jarFile = null;
		InputStream inputStream = null;
		try {
			jarFile = new JarFile(jarPath);
			JarEntry entry = jarFile.getJarEntry(fileName);
			inputStream = jarFile.getInputStream(entry);
			byte[] byteContent = new byte[inputStream.available()];
			inputStream.read(byteContent);
			return byteContent;
		} finally {
			CloseUtil.close(inputStream);
			CloseUtil.close(jarFile);
		}
	}

}