package cn.elwy.common.util.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Writer;
import java.net.URL;
import java.net.URLConnection;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Map;

import cn.elwy.common.exception.RunException;
import cn.elwy.common.i18n.Msg;
import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * 读写ini配置文件
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class IniUtil {

	Logger log = LoggerFactory.getLogger(IniUtil.class);

	private final String[] SYMBOL_COMMENT = new String[] { "#", ";" };
	private final String SYMBOL_EQUAL = " = ";
	private final String SYMBOL_SQUARE_BRACE_L = "[";
	private final String SYMBOL_SQUARE_BRACE_R = "]";

	private String fileName; // 配置文件全称

	private Map<String, Section> sectionMap; // 配置信息

	public IniUtil(String fileName) {
		this.fileName = fileName;
		this.sectionMap = new LinkedHashMap<String, Section>();
		if (!checkFile(fileName)) {
			log.error("操作失败！");
			throw new RunException("操作失败！");
		}
		readFile();
	}

	public IniUtil(String fileName, int connectTimeout, int readTimeout) {
		this.fileName = fileName;
		this.sectionMap = new LinkedHashMap<String, Section>();
		if (!checkFile(fileName)) {
			log.error("操作失败！");
			throw new RunException("操作失败！");
		}
		readFile(fileName, readTimeout, readTimeout);
	}

	public Map<String, Section> getSectionMap() {
		return sectionMap;
	}

	public int getTotalSection() {
		return sectionMap.size();
	}

	/**
	 * 读取本地ini配置信息
	 * @return 配置信息
	 */
	public Map<?, ?> readFile() {
		try {
			File file = new File(fileName);
			if (file.isFile() && file.exists()) {
				InputStreamReader inputStreamReader = new InputStreamReader(new FileInputStream(file), "UTF-8");
				BufferedReader bufferedReader = new BufferedReader(inputStreamReader);
				return readFile(bufferedReader);
			} else {
				throw new RunException("指定的文件不存在！");
			}
		} catch (RunException e) {
			throw e;
		} catch (Exception e) {
			log.error("操作失败！", e);
			throw new RunException("操作失败！", e);
		}
	}

	/**
	 * 读取网络中的ini配置信息
	 * @param fileName 配置文件全称
	 * @return 配置信息
	 */
	public Map<?, ?> readFile(String fileName, int connectTimeout, int readTimeout) {
		try {
			URL serverURL = new URL(fileName);
			URLConnection urlConn = serverURL.openConnection();
			urlConn.setConnectTimeout(connectTimeout);
			urlConn.setReadTimeout(readTimeout);
			urlConn.connect();
			InputStream inStream = urlConn.getInputStream();
			BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inStream));
			return readFile(bufferedReader);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * 读取ini配置信息
	 * @param bufferedReader 配置文件
	 * @return 配置信息
	 */
	private Map<String, Section> readFile(BufferedReader bufferedReader) {
		int iPosition = -1;
		String strLine = null;
		String strSection = null;
		String strRemark = null;
		Section iniSection = null;

		try {
			while (bufferedReader.ready()) {
				strLine = bufferedReader.readLine().trim();
				iPosition = -1;
				if (strLine == null || strLine.length() == 0) {
				} else if (strLine.startsWith(SYMBOL_SQUARE_BRACE_L) && strLine.endsWith(SYMBOL_SQUARE_BRACE_R)) {
					strSection = strLine.substring(1, strLine.length() - 1);
					iniSection = new Section(strSection.trim(), strRemark);
					this.sectionMap.put(strSection.trim(), iniSection);
				} else if (commentIndex(strLine) < 0 && (iPosition = strLine.indexOf(SYMBOL_EQUAL.trim())) > 0
						&& iniSection != null) {
					iniSection.setProperty(strLine.substring(0, iPosition).trim(), strLine.substring(iPosition + 1).trim(),
							strRemark);
				} else if (iniSection != null) {
					iniSection.setProperty(strLine.trim(), "", strRemark);
				} else {
					strSection = strLine.trim();
					iniSection = new Section(strLine.trim(), strRemark);
					this.sectionMap.put(strSection.trim(), iniSection);
				}
			}
		} catch (IOException e) {
			this.sectionMap.clear();
			String message = Msg.E_LOAD_CONFIG_ERROR;
			log.error(message, e);
			throw new RunException(message, e);
		} finally {
			if (bufferedReader != null) {
				close(bufferedReader);
				bufferedReader = null;
			}
			if (iniSection != null)
				iniSection = null;
		}
		return sectionMap;
	}

	/**
	 * 获得属性值
	 * @param sectionName
	 * @param propertyName
	 * @return
	 */
	public String getPropertyValue(String sectionName, String propertyName) {
		String propertyValue = null;
		Property property = null;
		Section section = getSection(sectionName);
		if (section != null) {
			property = section.getProperty(propertyName);
			if (property != null) {
				propertyValue = property.getPropertyValue();
				property = null;
			}
			section = null;
		}
		return propertyValue;
	}

	/**
	 * 获得属性值
	 * @param sectionName
	 * @param propertyName
	 * @return boolean
	 */
	public int getPropertyIntValue(String sectionName, String propertyName) {
		try {
			return Integer.parseInt(getPropertyValue(sectionName, propertyName));
		} catch (Exception e) {
			String errorMsg = "读取属性值错误，数据类型不匹配！";
			log.warn(errorMsg, e);
			throw new RunException(errorMsg);
		}
	}

	/**
	 * 获得属性值
	 * @param sectionName
	 * @param propertyName
	 * @return boolean
	 */
	public boolean getPropertyBooleanValue(String sectionName, String propertyName) {
		String propertyValue = getPropertyValue(sectionName, propertyName);
		try {
			propertyValue = propertyValue.toUpperCase();
			if ("YES".equals(propertyValue) || "TRUE".equals(propertyValue) || "1".equals(propertyValue)) {
				return true;
			}
			return false;
		} catch (Exception e) {
			String errorMsg = "读取属性值错误，数据类型不匹配！";
			log.warn(errorMsg, e);
			throw new RunException(errorMsg);
		}
	}

	public Property getProperty(String sectionName, String propertyName) {
		Section section = getSection(sectionName);
		if (section != null) {
			return section.getProperty(propertyName);
		}
		return null;
	}

	/**
	 * 获得属性的名称
	 * @param sectionName
	 * @return
	 */
	public String[] getPropertyNames(String sectionName) {
		Section section = getSection(sectionName);
		if (section != null) {
			return section.getPropertyNames();
		}
		return null;
	}

	public Map<?, ?> getProperty(String sectionName) {
		Section section = getSection(sectionName);
		if (section != null) {
			return section.getProperty();
		}
		return null;
	}

	public boolean existSection(String sectionName) {
		return getSection(sectionName) != null;
	}

	/**
	 * 获得Section
	 * @param sectionName
	 * @return Section
	 */
	public Section getSection(Object sectionName) {
		return this.sectionMap.get(sectionName);
	}

	/**
	 * 获得所有Section的名称
	 * @return
	 */
	public String[] getSectionNames() {
		int iCntr = 0;
		String[] sectionNames = null;
		if (sectionMap.size() > 0) {
			sectionNames = new String[sectionMap.size()];
			for (Object key : sectionMap.keySet()) {
				Section section = sectionMap.get(key);
				sectionNames[iCntr] = section.getSectionName();
				iCntr++;
			}
		}
		return sectionNames;
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param propertyValue
	 */
	public void setPropertyValue(String sectionName, String propertyName, String propertyValue) {
		Section section = setSection(sectionName);
		section.setProperty(propertyName, propertyValue);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param propertyValue
	 * @param propertyComment
	 */
	public void setPropertyValue(String sectionName, String propertyName, String propertyValue, String propertyComment) {
		Section section = setSection(sectionName);
		section.setProperty(propertyName, propertyValue, propertyComment);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param intValue
	 */
	public void setPropertyValue(String sectionName, String propertyName, int intValue) {
		String propertyValue = intValue + "";
		setPropertyValue(sectionName, propertyName, propertyValue);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param intValue
	 * @param propertyComment
	 */
	public void setPropertyValue(String sectionName, String propertyName, int intValue, String propertyComment) {
		String propertyValue = intValue + "";
		setPropertyValue(sectionName, propertyName, propertyValue, propertyComment);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param booleanValue
	 */
	public void setPropertyValue(String sectionName, String propertyName, boolean booleanValue) {
		String propertyValue = (booleanValue == true) ? "TRUE" : "FALSE";
		setPropertyValue(sectionName, propertyName, propertyValue);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param booleanValue
	 * @param propertyComment
	 */
	public void setPropertyValue(String sectionName, String propertyName, boolean booleanValue, String propertyComment) {
		String propertyValue = (booleanValue == true) ? "TRUE" : "FALSE";
		setPropertyValue(sectionName, propertyName, propertyValue, propertyComment);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param dateValue
	 */
	public void setPropertyValue(String sectionName, String propertyName, Date dateValue) {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		String propertyValue = sdf.format(dateValue);
		setPropertyValue(sectionName, propertyName, propertyValue);
	}

	/**
	 * 设置属性值
	 * @param sectionName
	 * @param propertyName
	 * @param dateValue
	 * @param propertyComment
	 */
	public void setPropertyValue(String sectionName, String propertyName, Date dateValue, String propertyComment) {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		String propertyValue = sdf.format(dateValue);
		setPropertyValue(sectionName, propertyName, propertyValue, propertyComment);
	}

	/**
	 * 设置Section对象，不存在则创建，并返回Section对象
	 * @param sectionName Section名称
	 * @return Section
	 */
	public Section setSection(String sectionName) {
		Section section = getSection(sectionName);
		if (section == null) {
			section = new Section(sectionName);
			sectionMap.put(sectionName, section);
		}
		return section;
	}

	/**
	 * 设置Section对象，不存在则创建，并返回Section对象
	 * @param sectionName Section名称
	 * @param sectionComment Section注释
	 * @return Section
	 */
	public Section setSection(String sectionName, String sectionComment) {
		Section section = getSection(sectionName);
		if (section == null) {
			section = new Section(sectionName, sectionComment);
			sectionMap.put(sectionName, section);
		} else {
			section.setSectionComment(sectionComment);
		}
		return section;
	}

	/**
	 * 删除Proerty
	 * @param sectionName
	 * @param propertyName
	 */
	public boolean removeProperty(String sectionName, String propertyName) {
		Section section = getSection(sectionName);
		if (section != null) {
			section.removeProperty(propertyName);
			section = null;
			return true;
		}
		return false;
	}

	/**
	 * 删除Section
	 * @param sectionName
	 * @param propertyName
	 */
	public boolean removeSection(String sectionName) {
		if (sectionMap.containsKey(sectionName)) {
			sectionMap.remove(sectionName);
			return true;
		}
		return false;
	}

	/**
	 * 修改ini配置信息
	 * @return
	 */
	public boolean writeFile() {
		boolean flag = false;
		File file = null;
		Section iniSection = null;
		FileWriter fileWriter = null;
		try {
			if (sectionMap.isEmpty()) {
				return false;
			}
			file = new File(fileName);
			if (file.exists()) {
				file.delete();
			}
			fileWriter = new FileWriter(file);
			for (Object key : sectionMap.keySet()) {
				iniSection = getSection(key);
				fileWriter.write(iniSection.toString());
				fileWriter.write("\r\n");
				iniSection = null;
			}
			flag = true;
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		} finally {
			if (fileWriter != null) {
				close(fileWriter);
				fileWriter = null;
			}
			if (file != null)
				file = null;
		}
		return flag;
	}

	/**
	 * 注释符所在的位置
	 * @param string
	 * @return
	 */
	public int commentIndex(String string) {
		int index = -1;
		for (int i = 0; i < SYMBOL_COMMENT.length; i++) {
			if ((index = string.indexOf(SYMBOL_COMMENT[i].trim())) == 0) {
				return index;
			}
		}
		return -1;
	}

	/**
	 * 清除注释符号
	 * @param content
	 * @return
	 */
	private String clearRemSymbol(String content) {
		if (content == null || "".equals(content)) {
			return null;
		}
		int intPos = 0;
		if ((intPos = commentIndex(content)) >= 0) {
			if (intPos == 0) {
				content = content.substring(intPos + 1);
			} else if (intPos > 0) {
				content = content.substring(0, intPos) + content.substring(intPos + 1);
			}
		}
		return content;
	}

	/**
	 * 添加注释符号
	 * @param content
	 * @return
	 */
	private String addRemSymbol(String content) {
		int intLen = 2;
		int intPos = 0;
		int intPrev = 0;

		String strLeft = null;
		String strRight = null;

		if (content == null || "".equals(content))
			return null;
		while (intPos >= 0) {
			intLen = 2;
			intPos = content.indexOf("\r\n", intPrev);
			if (intPos < 0) {
				intLen = 1;
				intPos = content.indexOf("\n", intPrev);
				if (intPos < 0)
					intPos = content.indexOf("\r", intPrev);
			}
			if (intPos == 0) {
				content = SYMBOL_COMMENT[0] + "\r\n" + content.substring(intPos + intLen);
				intPrev = intPos + intLen + 1;
			} else if (intPos > 0) {
				strLeft = content.substring(0, intPos);
				strRight = content.substring(intPos + intLen);
				if (strRight == null)
					content = strLeft;
				else if (strRight.length() == 0)
					content = strLeft;
				else
					content = strLeft + "\r\n" + SYMBOL_COMMENT[0] + strRight;
				intPrev = intPos + intLen + 1;
			}
		}
		if (!content.substring(0, 1).equals(SYMBOL_COMMENT[0]))
			content = SYMBOL_COMMENT[0] + content;
		content += "\r\n";
		return content;
	}

	private void close(Reader reader) {
		try {
			if (reader != null) {
				reader.close();
			}
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		}
	}

	private void close(Writer writer) {
		try {
			if (writer != null) {
				writer.close();
			}
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		}
	}

	/**
	 * 判断文件是否存在且可读写
	 * @param fileName 文件名
	 * @return 文件读写状态，true为可读写
	 */
	private boolean checkFile(String fileName) {
		boolean flag = false;
		File file = null;
		try {
			file = new File(fileName);
			// flag = file.exists() && file.isFile();
			flag = file.exists() && file.isFile() && file.canWrite() && file.canRead();
		} catch (Exception e) {
			flag = false;
			log.error("加载配置文件失败！", e);
			throw new RunException("加载配置文件失败！");
		} finally {
			if (file != null) {
				file = null;
			}
		}
		return flag;
	}

	/**
	 * Section Class
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	class Section {

		private String sectionName; // 模块名称
		private String sectionComment; // 模块备注
		private Map<String, Property> propertyMap;// 属性Map

		public Section(String sectionName) {
			this(sectionName, null);
		}

		public Section(String sectionName, String sectionComment) {
			this.sectionName = sectionName;
			this.sectionComment = clearRemSymbol(sectionComment);
			this.propertyMap = new LinkedHashMap<String, Property>();
		}

		public String getSectionComment() {
			return sectionComment;
		}

		public String getSectionName() {
			return sectionName;
		}

		public void setSectionComment(String sectionComment) {
			this.sectionComment = clearRemSymbol(sectionComment);
		}

		public void setSectionName(String sectionName) {
			this.sectionName = sectionName;
		}

		/**
		 * 设置属性
		 * @param propertyName
		 * @param propertyValue
		 */
		public void setProperty(String propertyName, String propertyValue) {
			this.propertyMap.put(propertyName, new Property(propertyName, propertyValue));
		}

		/**
		 * 设置属性
		 * @param propertyName
		 * @param propertyValue
		 * @param propertyComment
		 */
		public void setProperty(String propertyName, String propertyValue, String propertyComment) {
			this.propertyMap.put(propertyName, new Property(propertyName, propertyValue, propertyComment));
		}

		/**
		 * 获得当前Section的所有属性
		 * @return
		 */
		public Map<String, Property> getProperty() {
			return Collections.unmodifiableMap(propertyMap);
		}

		/**
		 * 获得指定的属性
		 * @param propertyName
		 * @return
		 */
		public Property getProperty(String propertyName) {
			if (this.propertyMap.containsKey(propertyName)) {
				return this.propertyMap.get(propertyName);
			}
			return null;
		}

		/**
		 * 获得当前Section的所有属性名称
		 * @return
		 */
		public String[] getPropertyNames() {
			int iCntr = 0;
			String[] propertyNames = null;
			if (propertyMap.size() > 0) {
				propertyNames = new String[propertyMap.size()];
				for (Object key : propertyMap.keySet()) {
					// Property property = propertyMap.get(key);
					propertyNames[iCntr] = key.toString();
					// Property property = propertyMap.get(key);
					// propertyNames[iCntr] = property.getPropertyName();
					iCntr++;
				}
			}
			return propertyNames;
		}

		/**
		 * 删除属性
		 * @param propertyName
		 */
		public void removeProperty(String propertyName) {
			if (propertyMap.containsKey(propertyName)) {
				propertyMap.remove(propertyName);
			}
		}

		@Override
		public String toString() {
			/*
			 * Set colKeys = null; String strRet = ""; Iterator iter = null; Property
			 * objProp = null; StringBuffer objBuf = new StringBuffer(); if
			 * (this.sectionComment != null)
			 * objBuf.append(addRemChars(this.sectionComment)); objBuf.append("[" +
			 * this.sectionName + "]\r\n"); colKeys = this.propertyMap.keySet(); if (colKeys
			 * != null) { iter = colKeys.iterator(); if (iter != null) { while
			 * (iter.hasNext()) { objProp = (Property) this.propertyMap.get(iter.next());
			 * objBuf.append(objProp.toString()); objBuf.append("\r\n"); objProp = null; } }
			 * } strRet = objBuf.toString(); objBuf = null; iter = null; colKeys = null;
			 * return strRet;
			 */
			StringBuffer stringBuffer = new StringBuffer();
			if (sectionComment != null) {
				stringBuffer.append(addRemSymbol(sectionComment));
			}
			stringBuffer.append(SYMBOL_SQUARE_BRACE_L + sectionName + SYMBOL_SQUARE_BRACE_R + "\r\n");
			for (Object sectionName : propertyMap.keySet()) {
				Property property = propertyMap.get(sectionName);
				stringBuffer.append(property.toString());
				stringBuffer.append("\r\n");
				property = null;
			}
			// System.out.println("______________________________________"); //
			// System.out.println(stringBuffer.toString()); //
			// System.out.println("______________________________________end");
			return stringBuffer.toString();

		}
	}

	/**
	 * Property Class
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	class Property {

		private String propertyName; // 属性名称
		private String propertyValue; // 属性值
		private String propertyComment; // 备注

		public Property(String propertyName, String propertyValue) {
			this(propertyName, propertyValue, null);
		}

		public Property(String propertyName, String propertyValue, String propertyComment) {
			this.propertyName = propertyName;
			this.propertyValue = propertyValue;
			this.propertyComment = clearRemSymbol(propertyComment);
		}

		public String getPropertyName() {
			return propertyName;
		}

		public String getPropertyValue() {
			return propertyValue;
		}

		public String getPropertyComment() {
			return propertyComment;
		}

		public void setPropertyName(String propertyName) {
			this.propertyName = propertyName;
		}

		public void setPropertyValue(String propertyValue) {
			this.propertyValue = propertyValue;
		}

		public void setPropertyComment(String propertyComment) {
			this.propertyComment = clearRemSymbol(propertyComment);
		}

		@Override
		public String toString() {
			String result = "";
			if (propertyComment != null) {
				result += addRemSymbol(propertyComment);
			}
			if (propertyValue == null || "".equals(propertyValue)) {
				result += propertyName;
			} else {
				result += propertyName + SYMBOL_EQUAL + propertyValue;
			}
			return result;
		}
	}

}