package cn.elwy.common.util.io;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

import cn.elwy.common.Constant;
import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;
import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.CloseUtil;

/**
 * 文件处理工具类,实现文件的复制、移动、查找、追加内容等。
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class FileUtil {

	private static Logger logger = LoggerFactory.getLogger(FileUtil.class);

	private static int BYTE_BUFFER_SIZE = 819200; // 800K
	private static int CHAR_BUFFER_SIZE = 8192; // 8K
	public static final String LINE_SEPARATOR = Constant.LINE_SEPARATOR;
	public static final Separator LINE_SEPARATOR_ALL = new Separator(LINE_SEPARATOR, LINE_SEPARATOR);
	public static final Separator LINE_SEPARATOR_START = new Separator(LINE_SEPARATOR, "");
	public static final Separator LINE_SEPARATOR_END = new Separator("", LINE_SEPARATOR);
	public static final Separator LINE_SEPARATOR_BR = new Separator("", "<br />");
	public static final Separator LINE_SEPARATOR_P = new Separator("<p>", "</p>");

	private FileUtil() {
	}

	/**
	 * 判断指定的文件是否可以创建
	 * @param file
	 * @return
	 */
	public static boolean canCreateFile(File file) {
		if (file == null || file.isDirectory()) {
			return false;
		} else if (file.isFile()) {
			return file.canWrite();
		}
		return canCreateDirectory(file.getParentFile());
	}

	/**
	 * 判断指定的目录是否可以创建
	 * @param file
	 * @return
	 */
	public static boolean canCreateDirectory(File file) {
		if (file == null || file.isFile()) {
			return false;
		}
		while (!file.exists()) {
			file = file.getParentFile();
			if (file == null) {
				return false;
			}
		}
		return file.isDirectory() && file.canWrite();
	}

	/**
	 * 判断文件是否有读权限
	 * @param file
	 * @return
	 */
	public static Boolean canRead(File file) {
		if (file.isDirectory()) {
			try {
				File[] listFiles = file.listFiles();
				if (listFiles == null) { // 返回null表示无法读取或访问，如果为空目录返回的是一个空数组
					return false;
				} else {
					return true;
				}
				/*- 如果是目录，不为空则表示有读取权限，不需要再做以下判断。Linux下未测试
				for (File child : listFiles) {
				    if (canRead(child)) { // 如果有一个文件有读权限，表示这个目录有读权限
				return true;
				    }
				}
				file = new File(file, "canReadTestDeleteOnExit.temp");
				if (file.exists()) {
				    return checkRead(file);
				} else if (!file.createNewFile()) {
				    return false;
				} else {
				    boolean checkRead = checkRead(file);
				    if (!deleteFile(file)) {
				file.deleteOnExit();
				    }
				    return checkRead;
				}
				 */
			} catch (Exception e) {
				logger.debug(e.getMessage());
				return false;
			}
		} else if (!file.exists()) { // 文件不存在
			return false;
		}
		return checkRead(file);
	}

	/**
	 * 检测文件是否有读权限
	 * @param file
	 * @return
	 */
	private static boolean checkRead(File file) {
		FileReader fd = null;
		try {
			fd = new FileReader(file);
			while ((fd.read()) != -1) {
				break;
			}
			return true;
		} catch (IOException e) {
			logger.debug(e.getMessage());
			return false;
		} finally {
			CloseUtil.close(fd);
		}
	}

	/**
	 * 判断文件是否有写权限
	 * @param file
	 * @return
	 */
	public static Boolean canWrite(File file) {
		if (file.isDirectory()) {
			try {
				file = new File(file, "canWriteTestDeleteOnExit.temp");
				if (file.exists()) {
					boolean checkWrite = checkWrite(file);
					if (!deleteFile(file)) {
						file.deleteOnExit();
					}
					return checkWrite;
				} else if (file.createNewFile()) {
					if (!deleteFile(file)) {
						file.deleteOnExit();
					}
					return true;
				} else {
					return false;
				}
			} catch (Exception e) {
				logger.debug(e.getMessage());
				return false;
			}
		}
		return checkWrite(file);
	}

	/**
	 * 检测文件是否有写权限
	 * @param file
	 * @return
	 */
	private static boolean checkWrite(File file) {
		FileWriter fw = null;
		boolean delete = !file.exists();
		boolean result = false;
		try {
			fw = new FileWriter(file, true);
			fw.write("");
			fw.flush();
			result = true;
			return result;
		} catch (IOException e) {
			logger.debug(e.getMessage());
			return false;
		} finally {
			CloseUtil.close(fw);
			if (delete && result) {
				deleteFile(file);
			}
		}
	}

	/**
	 * 复制文件(以超快的速度复制文件)
	 * @param srcFile 源文件File
	 * @param destDir 目标目录File
	 * @param newFileName 新文件名
	 * @param isReplace 是否替换同名文件
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFile(File srcFile, File destDir, String newFileName, boolean isReplace) throws IOException {
		return copyFile(srcFile, new File(destDir, newFileName), isReplace);
	}

	/**
	 * 复制文件(以超快的速度复制文件)
	 * @param srcFile 源文件File
	 * @param destFile 目标目录File
	 * @param isReplace 是否替换同名文件
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFile(File srcFile, File destFile, boolean isReplace) throws IOException {
		FileInputStream fis = null;
		FileOutputStream fos = null;
		FileChannel fisChannel = null;
		FileChannel fosChannel = null;
		try {
			if (destFile.exists() && !isReplace) {
				logger.warn(destFile.getAbsolutePath() + " is exists, not replace");
				return true;
			}
			destFile.getParentFile().mkdirs();
			fis = new FileInputStream(srcFile);
			fos = new FileOutputStream(destFile);
			fisChannel = fis.getChannel();
			fosChannel = fos.getChannel();
			ByteBuffer byteBuffer = ByteBuffer.allocate(BYTE_BUFFER_SIZE);
			while (fisChannel.read(byteBuffer) != -1) {
				byteBuffer.flip();
				fosChannel.write(byteBuffer);
				byteBuffer.clear();
			}
			return true;
		} finally {
			CloseUtil.close(fosChannel);
			CloseUtil.close(fisChannel);
			CloseUtil.close(fos);
			CloseUtil.close(fis);
		}
	}

	/**
	 * 复制文件或目录,如果源文件是目录将目录下的文件和子目录全部复制到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标文件
	 * @param isReplace 是否替换已经存在的文件
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(String srcFile, String destFile, boolean isReplace) throws IOException {
		File dest = new File(destFile);
		return copyFiles(new File(srcFile), dest.getParentFile(), dest.getName(), isReplace, null);
	}

	/**
	 * 复制文件或目录,如果源文件是目录将目录下的文件和子目录全部复制到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标文件
	 * @param isReplace 是否替换已经存在的文件
	 * @param fileter 文件过滤器
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(String srcFile, String destFile, boolean isReplace, FileFilter fileter)
			throws IOException {
		File dest = new File(destFile);
		return copyFiles(new File(srcFile), dest.getParentFile(), dest.getName(), isReplace, fileter);
	}

	/**
	 * 复制文件或目录,如果源文件是目录将目录下的文件和子目录全部复制到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标文件
	 * @param isReplace 是否替换已经存在的文件
	 * @param fileter 文件过滤器
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(File srcFile, File destFile, boolean isReplace) throws IOException {
		return copyFiles(srcFile, destFile.getParentFile(), destFile.getName(), isReplace, null);
	}

	/**
	 * 复制文件或目录,如果源文件是目录将目录下的文件和子目录全部复制到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标文件
	 * @param isReplace 是否替换已经存在的文件
	 * @param fileter 文件过滤器
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(File srcFile, File destFile, boolean isReplace, FileFilter fileter)
			throws IOException {
		return copyFiles(srcFile, destFile.getParentFile(), destFile.getName(), isReplace, fileter);
	}

	/**
	 * 复制一组文件到指定目录
	 * @param srcFileList 原文件
	 * @param destDir 目标目录
	 * @param isReplace 是否替换已经存在的文件
	 * @param fileter 文件过滤器
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(List<String> srcFileList, String destDir, boolean isReplace, FileFilter fileter)
			throws IOException {
		boolean result = true;
		File out = new File(destDir);
		for (String fileName : srcFileList) {
			File source = new File(fileName);
			result = copyFiles(source, out, source.getName(), isReplace, fileter);
			if (!result) {
				return result;
			}
		}
		return result;
	}

	/**
	 * 复制文件或目录到指定目录,如果是目录将目录以及目录下的文件和子目录全部复制到目标目录
	 * @param srcFile 源文件
	 * @param destDir 目标位置
	 * @param fileName 文件名称
	 * @param isReplace 是否替换已经存在的文件
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(File srcFile, File destDir, String fileName, boolean isReplace) throws IOException {
		return copyFiles(srcFile, destDir, fileName, isReplace, null);
	}

	/**
	 * 复制文件或目录到指定目录,如果是目录将目录以及目录下的文件和子目录全部复制到目标目录
	 * @param srcFile 源文件
	 * @param destDir 目标位置
	 * @param fileName 文件名称
	 * @param isReplace 是否替换已经存在的文件
	 * @param fileter 文件过滤器
	 * @return
	 * @throws IOException
	 */
	public static boolean copyFiles(File srcFile, File destDir, String fileName, boolean isReplace, FileFilter fileter)
			throws IOException {
		if (!srcFile.exists()) { // 文件不存在
			return false;
		}
		if (!destDir.exists()) {
			destDir.mkdirs();
		}
		if (!destDir.isDirectory()) {
			return false;
		}
		boolean result = true;
		if (srcFile.isFile()) {
			return copyFile(srcFile, destDir, fileName, isReplace);
		} else {
			destDir = new File(destDir, fileName);
			destDir.mkdirs();
			File[] children = srcFile.listFiles(fileter);
			if (children == null) {
				return true;
			}
			for (File file : children) {
				if (file.isFile()) {
					result = copyFile(file, destDir, file.getName(), isReplace);
				} else {
					result = copyFiles(file, destDir, file.getName(), isReplace, fileter);
				}
				if (!result) {
					return false;
				}
			}
		}
		return result;
	}

	/**
	 * 移动文件或目录到指定目位置,如果是目录将目录以及目录下的文件和子目录全部移动到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标位置
	 * @return
	 * @throws IOException
	 */
	public static boolean moveFiles(String srcFile, String destFile) throws IOException {
		return moveFiles(new File(srcFile), new File(destFile), true);
	}

	/**
	 * 移动文件或目录到指定目位置,如果是目录将目录以及目录下的文件和子目录全部移动到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标位置
	 * @return
	 * @throws IOException
	 */
	public static boolean moveFiles(File srcFile, File destFile) throws IOException {
		return moveFiles(srcFile, destFile, true);
	}

	/**
	 * 移动文件或目录到指定目位置,如果是目录将目录以及目录下的文件和子目录全部移动到目标位置
	 * @param srcFile 源文件
	 * @param destFile 目标位置
	 * @param isReplace 是否替换已经存在的文件
	 * @return
	 * @throws IOException
	 */
	public static boolean moveFiles(File srcFile, File destFile, boolean isReplace) throws IOException {
		if (!srcFile.exists()) { // 文件不存在
			return false;
		}
		boolean result = true;
		if (!destFile.exists()) {
			destFile.getParentFile().mkdirs();
			result = srcFile.renameTo(destFile);
			if (result) {
				return true;
			}
		}

		if (srcFile.isFile()) {
			result = copyFile(srcFile, destFile, isReplace);
			if (result) {
				srcFile.delete();
			}
			return result;
		} else {
			File[] children = srcFile.listFiles();
			if (children == null) {
				return true;
			}
			for (File file : children) {
				destFile = new File(destFile, file.getName());
				result = moveFiles(file, destFile, isReplace);
				if (result && isReplace) {
					deleteFile(file, true);
				}
			}
		}
		if (result && isReplace) {
			deleteFile(srcFile, true);
		}
		return result;
	}

	/**
	 * 创建文件夹
	 * @param filePath 目录路径
	 * @throws IOException
	 */
	public static boolean createDir(String filePath) {
		File file = new File(filePath);
		if (!file.exists()) {
			return file.mkdirs();
		}
		return false;
	}

	/**
	 * 创建文件
	 * @param filePath 文件路径
	 * @throws IOException
	 */
	public static boolean createFile(String filePath) throws IOException {
		File file = new File(filePath);
		if (!file.exists()) {// 如果文件不存在
			if (!file.getParentFile().exists()) {// 如果文件父目录不存在
				file.getParentFile().mkdirs();
			}
			return file.createNewFile();// 创建新文件
		}
		return false;
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，则将其所有子文件(夹)删除
	 * @param filePath 要删除的文件对象
	 * @return 删除是否成功
	 */
	public static boolean cleanFile(String filePath) {
		return cleanFile(new File(filePath), null);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，则将其所有子文件(夹)删除
	 * @param file 要删除的文件对象
	 * @return 删除是否成功
	 */
	public static boolean cleanFile(File file) {
		return cleanFile(file, null);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，则将其所有子文件(夹)删除
	 * @param file 要删除的文件对象
	 * @return 删除是否成功
	 */
	public static boolean cleanFile(File file, FileFilter fileter) {
		if (!file.exists()) { // 文件不存在
			return true;
		}
		boolean result = true;
		if (file.isFile()) {
			result = file.delete();
		} else {
			File[] children = file.listFiles(fileter);
			for (int i = 0; i < children.length; i++) { // 删除所有子文件和子文件夹
				result = deleteFile(children[i], true, fileter);// 递归删除文件
				if (!result) {
					return false;
				}
			}
		}
		return result;
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，先删除所有子文件(夹)，再删除该文件
	 * @param filePath 要删除的文件对象
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(String filePath) {
		return deleteFile(new File(filePath));
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，先删除所有子文件(夹)，再删除该文件
	 * @param file 要删除的文件对象
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(File file) {
		return deleteFile(file, true, null);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，先删除所有子文件(夹)，再删除该文件
	 * @param file 要删除的文件对象
	 * @param delDir 是否删除目录
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(String filePath, boolean delDir) {
		return deleteFile(new File(filePath), delDir, null);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，先删除所有子文件(夹)，再删除该文件
	 * @param file 要删除的文件对象
	 * @param delDir 是否删除目录
	 * @param fileter 文件过滤器
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(String filePath, boolean delDir, FileFilter fileter) {
		return deleteFile(new File(filePath), delDir, fileter);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，则根据delDir判断是否同时删除文件夹
	 * @param file 要删除的文件对象
	 * @param delDir 是否删除目录
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(File file, boolean delDir) {
		return deleteFile(file, delDir, null);
	}

	/**
	 * 删除文件，如果要删除的对象是文件夹，则根据delDir判断是否同时删除文件夹
	 * @param file 要删除的文件对象
	 * @param delDir 是否删除目录
	 * @param fileter 文件过滤器
	 * @return 删除是否成功
	 */
	public static boolean deleteFile(File file, boolean delDir, FileFilter fileter) {
		if (!file.exists()) { // 文件不存在
			return true;
		}
		if (file.isFile()) {
			return forceDelete(file);
		} else {
			boolean result = true;
			File[] children = file.listFiles(fileter);
			for (int i = 0; i < children.length; i++) { // 删除所有子文件和子文件夹
				result = deleteFile(children[i], delDir, fileter);// 递归删除文件
				if (!result) {
					return false;
				}
			}
			if (delDir) {
				result = file.delete(); // 删除当前文件夹
			}
			return result;
		}
	}

	/**
	 * 强制删除文件，删除失败调用System.gc()释放资源在删除，最多尝试10次
	 * @param file
	 * @return
	 */
	public static boolean forceDelete(File file) {
		boolean result = file.delete();
		int tryCount = 0;
		while (!result && tryCount++ < 10) {
			logger.debug("try to delete file: {0} count:{1}", file.getName(), tryCount);
			System.gc();
			result = file.delete();
			if (!result) {
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
		return result;
	}

	/**
	 * 获取文件扩展名(不包含".")，如果文件没有扩展名返回""
	 * @param fileName 文件名称
	 * @return 文件的后缀名
	 */
	public static String getExtName(String fileName) {
		int index = fileName.lastIndexOf(".");
		return (index == -1) ? "" : fileName.substring(index + 1);
	}

	/**
	 * 获取不带扩展名的文件名，如file的路径为C:/test.txt，则返回"test"; file的路径为C:/test1，则返回"test1";
	 * file的路径为C:/.project，则返回""
	 * @param file 文件
	 * @return 不含后缀名的文件名
	 */
	public static String getShortFileName(File file) {
		String name = file.getName();
		int index = name.lastIndexOf(".");
		return (index < 0) ? name : (index == 0) ? "" : name.substring(0, index);
	}

	/**
	 * 获取不带扩展名的文件名，如file的路径为C:/test.txt，则返回"test"; file的路径为C:/test1，则返回"test1";
	 * file的路径为C:/.project，则返回""
	 * @param filePath 完整的文件名
	 * @return 不含后缀名的文件名
	 */
	public static String getShortFileName(String filePath) {
		return getShortFileName(new File(filePath));
	}

	/**
	 * 获取新的文件名，调用System.currentTimeMillis()产生文件名称
	 * @param fileName 源文件名
	 * @return 新的文件名称，不包含路径
	 */
	public static String getFileNameByDataAndRandom() {
		return getFileNameByDataAndRandom(5);
	}

	/**
	 * 获取新的文件名，调用System.currentTimeMillis()产生文件名称
	 * @param fileName 源文件名
	 * @param length 随机数长度
	 * @return 新的文件名称，不包含路径
	 */
	public static String getFileNameByDataAndRandom(int length) {
		String charStr = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		StringBuffer buffer = new StringBuffer();
		Random random = new Random();
		int range = charStr.length();
		for (int i = 0; i < length; i++) {
			buffer.append(charStr.charAt(random.nextInt(range)));
		}
		return getCurrentTime() + "_" + buffer.toString();
	}

	/**
	 * 获取当前时间，以yyyyMMddHHmmssSSS格式返回
	 * @return
	 */
	private static String getCurrentTime() {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");
		return sdf.format(new Date());
	}

	/**
	 * 获取新的文件名，调用当前时间产生文件名称
	 * @param fileName 源文件名或扩展名
	 * @return 新的文件名称，不包含路径
	 */
	public static String getNewFileName(String fileName) {
		String ext = getExtName(fileName); // 扩展名
		return getFileNameByDataAndRandom() + "." + ext;
	}

	/**
	 * 下载远程文件
	 * @param fromPath 远程文件路径
	 * @param toPath 保存路径
	 * @return
	 * @throws IOException
	 */
	public static File downLoadFile(String fromPath, String toPath) throws IOException {
		return downLoadFile(new URL(fromPath), toPath);
	}

	/**
	 * 下载远程文件
	 * @param fileURL 远程文件
	 * @param toPath 保存路径
	 * @return
	 * @throws IOException
	 */
	public static File downLoadFile(URL fileURL, String toPath) throws IOException {
		File file = new File(toPath);
		writeFileByInputStream(file, fileURL.openStream());
		return file;
	}

	/**
	 * 使用系统默认方式打开文件
	 * @param filePath
	 * @throws IOException
	 */
	public static void open(String filePath) throws IOException {
		String[] cmds = new String[] { "cmd", "/c", "START", " ", filePath };
		Runtime.getRuntime().exec(cmds);
	}

	/**
	 * 使用系统默认方式打开文件
	 * @param file
	 * @throws IOException
	 */
	public static void open(File file) throws IOException {
		if (file != null && file.exists()) {
			open(file.getAbsolutePath());
		} else {
			throw new IOException("文件不存在！");
		}
	}

	/**
	 * 按行读取文件的内容，以List形式返回所有行
	 * @param filePath 要读取的文件
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readFileToList(String filePath) throws IOException {
		return readFileToList(new File(filePath), null);
	}

	/**
	 * 按行读取文件的内容，以List形式返回所有行
	 * @param file 要读取的文件
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readFileToList(File file) throws IOException {
		return readFileToList(file, null);
	}

	/**
	 * 按行读取文件的内容，以List形式返回所有行
	 * @param filePath 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readFileToList(String filePath, String charsetName) throws IOException {
		return readFileToList(new File(filePath), charsetName);
	}

	/**
	 * 按行读取文件的内容，以List形式返回所有行
	 * @param file 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readFileToList(File file, String charsetName) throws IOException {
		BufferedReader read = null;
		try {
			read = getBufferedReader(file, charsetName);

			List<String> content = new ArrayList<String>();
			String line = null;
			while ((line = read.readLine()) != null) {
				content.add(line);
			}
			return content;
		} finally {
			CloseUtil.close(read);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param filePath 要读取的文件
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(String filePath) throws IOException {
		return readFileToString(new File(filePath), null);
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param file 要读取的文件
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(File file) throws IOException {
		return readFileToString(file, null);
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param filePath 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(String filePath, String charsetName) throws IOException {
		return readFileToString(new File(filePath), null);
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param file 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(File file, String charsetName) throws IOException {
		BufferedReader bufferedReader = null;
		try {
			bufferedReader = getBufferedReader(file, charsetName);

			char[] buffer = new char[CHAR_BUFFER_SIZE];
			int length;
			StringBuffer content = new StringBuffer();
			while ((length = bufferedReader.read(buffer)) != -1) {
				content.append(buffer, 0, length);
			}
			return content.toString();
		} finally {
			CloseUtil.close(bufferedReader);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param filePath 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @param separator 换行分隔符
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(String filePath, String charsetName, Separator separator) throws IOException {
		return readFileToString(new File(filePath), charsetName, separator);
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param file 要读取的文件
	 * @param charsetName 文件的字符编码,可以为null
	 * @param separator 换行分隔符
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readFileToString(File file, String charsetName, Separator separator) throws IOException {
		BufferedReader read = null;
		try {
			read = getBufferedReader(file, charsetName);
			StringBuffer content = new StringBuffer();
			String line = null;
			while ((line = read.readLine()) != null) {
				content.append(separator.start).append(line).append(separator.end);
			}
			return content.toString();
		} finally {
			CloseUtil.close(read);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param inputStream 要读取的流
	 * @param charsetName 文件的字符编码,可以为null
	 * @param separator 换行分隔符
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readInputStreamToString(InputStream inputStream, String charsetName, Separator separator)
			throws IOException {
		BufferedReader read = null;
		try {
			read = getBufferedReader(inputStream, charsetName);
			StringBuffer content = new StringBuffer();
			String line = null;
			while ((line = read.readLine()) != null) {
				content.append(separator.start).append(line).append(separator.end);
			}
			return content.toString();
		} finally {
			CloseUtil.close(read);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param inputStream 要读取的流
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static String readInputStreamToString(InputStream inputStream, String charsetName) throws IOException {
		BufferedReader bufferedReader = null;
		try {
			bufferedReader = getBufferedReader(inputStream, charsetName);

			char[] buffer = new char[CHAR_BUFFER_SIZE];
			int length;
			StringBuffer content = new StringBuffer();
			while ((length = bufferedReader.read(buffer)) != -1) {
				content.append(buffer, 0, length);
			}
			return content.toString();
		} finally {
			CloseUtil.close(bufferedReader);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字符串返回
	 * @param inputStream 要读取的流
	 * @param charsetName 文件的字符编码,可以为null
	 * @return 文件内容
	 * @throws IOException
	 */
	public static List<String> readInputStreamToList(InputStream inputStream, String charsetName) throws IOException {
		BufferedReader bufferedReader = null;
		try {
			bufferedReader = getBufferedReader(inputStream, charsetName);

			List<String> content = new ArrayList<String>();
			String line = null;
			while ((line = bufferedReader.readLine()) != null) {
				content.add(line);
			}
			return content;
		} finally {
			CloseUtil.close(bufferedReader);
		}
	}

	/**
	 * 读取文件内容后，将文件内容作为字节数组返回
	 * @param filePath
	 * @return 文件内容字节数组
	 * @throws IOException
	 */
	public static byte[] readFileToBytes(String filePath) throws IOException {
		return readFileToBytes(new File(filePath));
	}

	/**
	 * 读取文件内容后，将文件内容作为字节数组返回
	 * @param file
	 * @return 文件内容字节数组
	 * @throws IOException
	 */
	public static byte[] readFileToBytes(File file) throws IOException {
		byte[] byteContent = null;
		InputStream inputStream = null;
		try {
			inputStream = new FileInputStream(file);
			byteContent = new byte[inputStream.available()];
			inputStream.read(byteContent);
		} finally {
			CloseUtil.close(inputStream);
		}
		return byteContent;
	}

	/**
	 * 使用FileOutputStream方法追加文件
	 * @param filePath 文件名
	 * @param content 追加的内容
	 * @throws IOException
	 */
	public static void appendFileByBufferedWriter(String filePath, String content) throws IOException {
		appendFileByBufferedWriter(new File(filePath), content);
	}

	/**
	 * 使用FileOutputStream方法追加文件
	 * @param file 文件名
	 * @param content 追加的内容
	 * @throws IOException
	 */
	public static void appendFileByBufferedWriter(File file, String content) throws IOException {
		String fileEncoding = getFileEncoding(file);
		appendFileByBufferedWriter(file, content, fileEncoding);
	}

	/**
	 * 使用FileOutputStream方法追加文件
	 * @param filePath 文件名
	 * @param content 追加的内容
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static void appendFileByBufferedWriter(String filePath, String content, String charsetName)
			throws IOException {
		appendFileByBufferedWriter(new File(filePath), content, charsetName);
	}

	/**
	 * 使用FileOutputStream方法追加文件
	 * @param file 文件名
	 * @param content 追加的内容
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static void appendFileByBufferedWriter(File file, String content, String charsetName) throws IOException {
		if (content == null) {
			return;
		}
		BufferedWriter bufferedWriter = null;
		try {
			bufferedWriter = getBufferedWriter(file, true, charsetName);
			bufferedWriter.write(content);
			bufferedWriter.flush();
		} finally {
			CloseUtil.close(bufferedWriter);
		}
	}

	/**
	 * 使用RandomAccessFile方法追加文件
	 * @param filePath 文件名
	 * @param content 追加的内容
	 * @throws IOException
	 */
	public static void appendFileByRandomAccessFile(String filePath, String content) throws IOException {
		appendFileByRandomAccessFile(new File(filePath), content);
	}

	/**
	 * 使用RandomAccessFile方法追加文件
	 * @param file 文件名
	 * @param content 追加的内容
	 * @throws IOException
	 */
	public static void appendFileByRandomAccessFile(File file, String content) throws IOException {
		String fileEncoding = getFileEncoding(file);
		appendFileByRandomAccessFile(file, content, fileEncoding);
	}

	/**
	 * 使用RandomAccessFile方法追加文件
	 * @param filePath 文件名
	 * @param content 追加的内容
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static void appendFileByRandomAccessFile(String filePath, String content, String charsetName)
			throws IOException {
		appendFileByRandomAccessFile(new File(filePath), content, charsetName);
	}

	/**
	 * 使用RandomAccessFile方法追加文件
	 * @param file 文件名
	 * @param content 追加的内容
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static void appendFileByRandomAccessFile(File file, String content, String charsetName) throws IOException {
		RandomAccessFile randomFile = null;
		try {
			// 打开一个随机访问文件流，按读写方式
			randomFile = new RandomAccessFile(file, "rw");
			// 文件长度，字节数
			long fileLength = randomFile.length();
			// 将写文件指针移到文件尾。
			randomFile.seek(fileLength);
			if (AssertUtil.isEmpty(charsetName)) {
				randomFile.write(content.getBytes());
			} else {
				randomFile.write(content.getBytes(charsetName));
			}
		} finally {
			CloseUtil.close(randomFile);
		}
	}

	/**
	 * 根据文件路径获取BufferedReader，如果文件有BOM信息，可以自动识别文件编码
	 * @param filePath 文件路径
	 * @return
	 * @throws IOException
	 */
	public static BufferedReader getBufferedReader(String filePath) throws IOException {
		return getBufferedReader(new File(filePath), null);
	}

	/**
	 * 根据文件路径获取BufferedReader，如果文件有BOM信息，可以自动识别文件编码
	 * @param file 文件
	 * @return
	 * @throws IOException
	 */
	public static BufferedReader getBufferedReader(File file) throws IOException {
		return getBufferedReader(file, null);
	}

	/**
	 * 根据文件路径获取BufferedReader，如果文件有BOM信息，可以自动识别文件编码
	 * @param filePath 文件路径
	 * @param charsetName 文件编码
	 * @return
	 * @throws IOException
	 */
	public static BufferedReader getBufferedReader(String filePath, String charsetName) throws IOException {
		return getBufferedReader(new File(filePath), charsetName);
	}

	/**
	 * 根据文件路径获取BufferedReader，如果文件有BOM信息，可以自动识别文件编码
	 * @param file 文件
	 * @param charsetName 文件编码
	 * @return
	 * @throws IOException
	 */
	public static BufferedReader getBufferedReader(File file, String charsetName) throws IOException {
		if (charsetName == null) {
			charsetName = getFileEncoding(file, charsetName);
		}
		return getBufferedReader(new FileInputStream(file), charsetName);
	}

	/**
	 * 根据文件路径获取BufferedReader，如果文件有BOM信息，可以自动识别文件编码
	 * @param inputStream 输入流
	 * @param charsetName 文件编码
	 * @return
	 * @throws IOException
	 */
	public static BufferedReader getBufferedReader(InputStream inputStream, String charsetName) throws IOException {
		UnicodeReader unicodeReader = new UnicodeReader(inputStream, charsetName);
		return new BufferedReader(unicodeReader);
	}

	/**
	 * 根据文件路径获取BufferedWriter，默认覆盖方式写文件
	 * @param filePath 文件路径
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(String filePath) throws IOException {
		return getBufferedWriter(new File(filePath), false, null);
	}

	/**
	 * 根据文件路径获取BufferedWriter，默认覆盖方式写文件
	 * @param filePath 文件路径
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(String filePath, String charsetName) throws IOException {
		return getBufferedWriter(new File(filePath), false, charsetName);
	}

	/**
	 * 根据文件路径获取BufferedWriter，默认覆盖方式写文件
	 * @param file 文件路径
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(File file) throws IOException {
		return getBufferedWriter(file, false, null);
	}

	/**
	 * 根据文件路径获取BufferedWriter，默认覆盖方式写文件
	 * @param file 文件路径
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(File file, String charsetName) throws IOException {
		return getBufferedWriter(file, false, charsetName);
	}

	/**
	 * 根据文件路径获取BufferedWriter，append值为真表示追加方式否则为覆盖方式
	 * @param filePath 文件路径
	 * @param append 是否追加方式
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(String filePath, boolean append) throws IOException {
		return getBufferedWriter(new File(filePath), append, null);
	}

	/**
	 * 根据文件路径获取BufferedWriter，append值为真表示追加方式否则为覆盖方式
	 * @param filePath 文件路径
	 * @param append 是否追加方式
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(String filePath, boolean append, String charsetName)
			throws IOException {
		return getBufferedWriter(new File(filePath), append, charsetName);
	}

	/**
	 * 根据文件路径获取BufferedWriter，append值为真表示追加方式否则为覆盖方式
	 * @param file 文件
	 * @param append 是否追加方式
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(File file, boolean append) throws IOException {
		return getBufferedWriter(file, append, null);
	}

	/**
	 * 根据文件路径获取BufferedWriter，append值为真表示追加方式否则为覆盖方式
	 * @param file 文件
	 * @param append 是否追加方式
	 * @param charsetName 文件编码
	 * @throws IOException
	 */
	public static BufferedWriter getBufferedWriter(File file, boolean append, String charsetName) throws IOException {
		if (!file.getParentFile().exists()) { // 如果目录不存在则创建目录
			file.getParentFile().mkdirs();
		}
		OutputStreamWriter outputStreamWriter = null;
		FileOutputStream fileOutputStream = new FileOutputStream(file, append);
		if (AssertUtil.isEmpty(charsetName)) {
			charsetName = getFileEncoding(file, charsetName);
		}
		if (AssertUtil.isEmpty(charsetName)) {
			outputStreamWriter = new OutputStreamWriter(fileOutputStream);
		} else {
			outputStreamWriter = new OutputStreamWriter(fileOutputStream, charsetName);
		}
		return new BufferedWriter(outputStreamWriter);
	}

	/**
	 * 往BufferedWriter写入字符串
	 * @param bufferedWriter
	 * @param content 追加的内容
	 * @throws IOException
	 */
	public static void writeBufferedWriter(BufferedWriter bufferedWriter, String content) throws IOException {
		if (content != null) {
			bufferedWriter.write(content);
			bufferedWriter.flush();
		}
	}

	/**
	 * 将byte写入文件
	 * @param filePath 文件路径
	 * @param fileByte 字节
	 * @throws IOException
	 */
	public static void writeFileByByte(String filePath, byte[] fileByte) throws IOException {
		writeFileByByte(new File(filePath), fileByte);
	}

	/**
	 * 将byte写入文件
	 * @param file 文件
	 * @param fileByte 字节
	 * @throws IOException
	 */
	public static void writeFileByByte(File file, byte[] fileByte) throws IOException {
		OutputStream outputStream = null;
		try {
			if (!file.getParentFile().exists()) { // 如果目录不存在则创建目录
				file.getParentFile().mkdirs();
			}
			outputStream = new FileOutputStream(file);
			outputStream.write(fileByte);
			outputStream.flush();
		} finally {
			CloseUtil.close(outputStream);
		}
	}

	/**
	 * 保存对象到指定的文件
	 * @param filePath
	 * @param object
	 * @throws Exception
	 */
	public static void writeFileByObject(String filePath, Object object) throws IOException {
		writeFileByObject(new File(filePath), object);
	}

	/**
	 * 保存对象到指定的文件
	 * @param file
	 * @param object
	 * @throws Exception
	 */
	public static void writeFileByObject(File file, Object object) throws IOException {
		if (!file.getParentFile().exists()) {
			file.getParentFile().mkdirs();
		}
		FileOutputStream fos = null;
		ObjectOutputStream oos = null;
		try {
			fos = new FileOutputStream(file);
			oos = new ObjectOutputStream(fos);
			oos.writeObject(object);
		} finally {
			CloseUtil.close(oos);
			CloseUtil.close(fos);
		}
	}

	/**
	 * 保存数据流到指定的文件
	 * @param filePath 输出路径
	 * @param inputStream 输入流
	 * @throws IOException
	 */
	public static void writeFileByInputStream(String filePath, InputStream inputStream) throws IOException {
		writeFileByInputStream(new File(filePath), inputStream);
	}

	/**
	 * 保存数据流到指定的文件
	 * @param file 输出文件
	 * @param inputStream 输入流
	 * @throws IOException
	 */
	public static void writeFileByInputStream(File file, InputStream inputStream) throws IOException {
		if (!file.getParentFile().exists()) { // 如果目录不存在则创建目录
			file.getParentFile().mkdirs();
		}
		OutputStream out = null;
		try {
			out = new FileOutputStream(file);
			byte[] buffer = new byte[BYTE_BUFFER_SIZE];
			int length;
			while ((length = inputStream.read(buffer)) != -1) {
				out.write(buffer, 0, length);
				out.flush();
			}
		} finally {
			CloseUtil.close(inputStream);
			CloseUtil.close(out);
		}
	}

	/**
	 * 将字符串写入文件
	 * @param filePath 文件路径
	 * @param content 字符串内容
	 * @param charsetName 字符集名称,可以为null
	 * @return 写入成功返回true,否则返回false
	 * @throws IOException
	 */
	public static boolean writeFileByString(String filePath, String content, String charsetName) throws IOException {
		return writeFileByString(new File(filePath), content, charsetName);
	}

	/**
	 * 将字符串写入文件
	 * @param file 文件
	 * @param content 字符串内容
	 * @param charsetName 字符集名称,可以为null
	 * @return 写入成功返回true,否则返回false
	 * @throws IOException
	 */
	public static boolean writeFileByString(File file, String content, String charsetName) throws IOException {
		if (content == null) {
			return false;
		}
		BufferedWriter bufferedWriter = null;
		try {
			bufferedWriter = getBufferedWriter(file, false, charsetName);
			bufferedWriter.write(content);
			bufferedWriter.flush();
			return true;
		} finally {
			CloseUtil.close(bufferedWriter);
		}
	}

	/**
	 * 根据文件BOM获取文件的编码，如果无BOM返回默认编码
	 * @param filePath 文件路径
	 * @return 文件编码格式
	 */
	public static String getFileEncoding(String filePath) {
		return getFileEncoding(new File(filePath), null);
	}

	/**
	 * 根据文件BOM获取文件的编码，如果无BOM返回默认编码
	 * @param filePath 文件路径
	 * @param defaultEncoding 默认编码
	 * @return 文件编码格式
	 */
	public static String getFileEncoding(String filePath, String defaultEncoding) {
		return getFileEncoding(new File(filePath), defaultEncoding);
	}

	/**
	 * 根据文件BOM获取文件的编码，如果无BOM返回默认编码
	 * @param file 文件
	 * @return 文件编码格式
	 */
	public static String getFileEncoding(File file) {
		return getFileEncoding(file, null);
	}

	/**
	 * 根据文件BOM获取文件的编码，如果无BOM返回默认编码
	 * @param file 文件
	 * @param defaultEncoding 默认编码
	 * @return 文件编码格式
	 */
	public static String getFileEncoding(File file, String defaultEncoding) {
		try {
			return CharsetDetectorUtil.getCharsetName(file);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
		return defaultEncoding;
	}

}
