package cn.elwy.common.util.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import cn.elwy.common.util.CloseUtil;

/**
 * 文件解析器,根据分隔符将多行内容解析成一个String对象，并以List的形式返回
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class FileParser {

	public static final List<String> IGNORE_SQL;

	/** 命令定界符 */
	private String delimiter = ";";
	/** 忽略大小写 */
	private boolean ignoreCase = true;
	/** 保留文件格式 */
	private boolean keepFormat = false;
	/** 忽略多行注释 */
	private boolean ignoreMultiLine = true;
	/** 忽略内容的正则表达式 */
	private List<String> ignoreLineList;
	/** 匹配内容的正则表达式 */
	private List<String> includeLineList;
	/** 多行注释标识 */
	private MultiLineComment multiLineComment = new MultiLineComment();

	static {
		IGNORE_SQL = new ArrayList<String>();
		IGNORE_SQL.add("^--|^//|^\\#|^prompt |^spool |^set |^pl/sql |^commit");
	}

	/**
	 * 文件解析器类
	 * @param file 文件路径
	 * @param encoding 文件编码
	 */
	public FileParser() {
		this(";", false, true, null, null);
	}

	/**
	 * 文件解析器构造函数
	 * @param delimiter 解析文件内容的分割符
	 * @param ignoreCase 忽略大小写
	 * @param ignoreLineList 文件中忽略的内容
	 */
	public FileParser(boolean ignoreCase, List<String> ignoreLineList) {
		this(";", false, true, ignoreLineList, null);
	}

	/**
	 * 文件解析器构造函数
	 * @param delimiter 解析文件内容的分割符
	 * @param ignoreCase 忽略大小写
	 * @param ignoreLineList 文件中忽略的内容
	 */
	public FileParser(String delimiter, boolean ignoreCase, List<String> ignoreLineList) {
		this(delimiter, false, true, ignoreLineList, null);
	}

	/**
	 * 文件解析器构造函数
	 * @param delimiter 解析文件内容的分割符
	 * @param keepFormat 保留格式
	 * @param ignoreCase 忽略大小写
	 * @param ignoreLineList 文件中忽略的内容
	 * @param includeLineList 文件包含的内容
	 */
	public FileParser(String delimiter, boolean keepFormat, boolean ignoreCase, List<String> ignoreLineList,
			List<String> includeLineList) {
		this.delimiter = delimiter;
		this.keepFormat = keepFormat;
		this.ignoreCase = ignoreCase;
		this.ignoreLineList = ignoreLineList;
		this.includeLineList = includeLineList;
	}

	/**
	 * 解析文件内容为一个命令列表
	 * @return
	 * @throws IOException
	 */
	public List<String> parseToList(File file, String encoding) throws IOException {
		BufferedReader read = null;
		try {
			read = FileUtil.getBufferedReader(file, encoding);
			return readLineToList(read);
		} finally {
			CloseUtil.close(read);
		}
	}

	/**
	 * 解析文件内容为一个命令列表
	 * @return
	 * @throws IOException
	 */
	public List<String> parseToList(InputStream inputStream, String encoding) throws IOException {
		BufferedReader read = null;
		try {
			read = FileUtil.getBufferedReader(inputStream, encoding);
			return readLineToList(read);
		} finally {
			CloseUtil.close(read);
		}
	}

	/**
	 * 读取文件内容后，将文件内容解析为一个命令列表
	 * @param read
	 * @param charsetName
	 * @return
	 * @throws IOException
	 */
	protected List<String> readLineToList(BufferedReader read) throws IOException {
		String lineSseparator = System.getProperty("line.separator");
		multiLineComment.setComment(false);
		StringBuffer textBuffer = new StringBuffer();
		String line = null;
		List<String> commandList = new ArrayList<String>();
		while ((line = read.readLine()) != null) {
			if (isIgnoreLine(line)) {
				continue;
			}
			if (keepFormat) {
				textBuffer.append(lineSseparator).append(line);
				line = line.trim();
			} else {
				line = line.trim();
				line = line.replaceAll("\\s+", " ");
				if (textBuffer.length() > 0) {
					textBuffer.append(" ").append(line);
				} else {
					textBuffer.append(line);
				}
			}
			if (line.endsWith(delimiter)) {
				String text = textBuffer.toString();
				int index = text.lastIndexOf(delimiter);
				text = text.substring(0, index);
				commandList.add(text);
				textBuffer.setLength(0);
			}
		}
		return commandList;
	}

	/**
	 * 忽略行
	 * @param line
	 * @return
	 */
	protected boolean isIgnoreLine(String line) {
		line = line.trim();
		if (ignoreLineList != null) {
			for (String regex : ignoreLineList) {
				if (isMatcher(regex, line)) {
					return true;
				}
			}
		}
		if (includeLineList != null) {
			for (String regex : includeLineList) {
				if (isMatcher(regex, line)) {
					return false;
				}
			}
		}
		if (ignoreMultiLine) {
			if (line.endsWith(multiLineComment.end)) {
				multiLineComment.isComment = false;
				return true;
			}
			if (multiLineComment.isComment) {
				return true;
			}
			if (line.startsWith(multiLineComment.start)) {
				multiLineComment.isComment = true;
				return true;
			}
		}
		return false;
	}

	/**
	 * 判断内容是否匹配正则表达式
	 * @param content
	 * @return
	 */
	protected boolean isMatcher(String regex, String content) {
		if (ignoreCase) {
			return Pattern.compile(regex, Pattern.CASE_INSENSITIVE).matcher(content).find();
		} else {
			return Pattern.compile(regex).matcher(content).find();
		}
	}

	/**
	 * @return the keepFormat
	 */
	public boolean isKeepFormat() {
		return keepFormat;
	}

	/**
	 * @param keepFormat the keepFormat to set
	 */
	public void setKeepFormat(boolean keepFormat) {
		this.keepFormat = keepFormat;
	}

	/**
	 * @return the delimiter
	 */
	public String getDelimiter() {
		return delimiter;
	}

	/**
	 * @param delimiter the delimiter to set
	 */
	public void setDelimiter(String delimiter) {
		this.delimiter = delimiter;
	}

	/**
	 * @return the ignoreLineList
	 */
	public List<String> getIgnoreLineList() {
		return ignoreLineList;
	}

	/**
	 * @param ignoreLineList the ignoreLineList to set
	 */
	public void setIgnoreLineList(List<String> ignoreLineList) {
		this.ignoreLineList = ignoreLineList;
	}

	/**
	 * @return the ignoreMultiLine
	 */
	public boolean isIgnoreMultiLine() {
		return ignoreMultiLine;
	}

	/**
	 * @param ignoreMultiLine the ignoreMultiLine to set
	 */
	public void setIgnoreMultiLine(boolean ignoreMultiLine) {
		this.ignoreMultiLine = ignoreMultiLine;
	}

	/**
	 * @return the multiLineComment
	 */
	public MultiLineComment getMultiLineComment() {
		return multiLineComment;
	}

	/**
	 * @param multiLineComment the multiLineComment to set
	 */
	public void setMultiLineComment(MultiLineComment multiLineComment) {
		this.multiLineComment = multiLineComment;
	}

	/**
	 * 多行注释标识类
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	public class MultiLineComment {

		private String start;
		private String end;
		private boolean isComment;

		public MultiLineComment() {
			this("/*", "*/");
		}

		public MultiLineComment(String start, String end) {
			this.start = start;
			this.end = end;
		}

		public String getStart() {
			return start;
		}

		public void setStart(String start) {
			this.start = start;
		}

		public String getEnd() {
			return end;
		}

		public void setEnd(String end) {
			this.end = end;
		}

		public boolean isComment() {
			return isComment;
		}

		public void setComment(boolean isComment) {
			this.isComment = isComment;
		}

	}
}
