package cn.elwy.common.util.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import cn.elwy.common.util.AssertUtil;
import cn.elwy.common.util.CloseUtil;

/**
 * 文件内容查找，查找满足条件的行，并根据正则表达式，替换掉多余的内容
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public class FileContentSearch {

	/** 文件路径 */
	private File file;
	/** 文件编码 */
	private String encoding;
	/** 是否只查找一行满足条件的内容 */
	private boolean singleLine;
	/** 忽略大小写 */
	private boolean ignoreCase = true;
	/** 替换掉行内容的表达式 */
	private String regex;
	/** 替换成目标字符串 */
	private String replacement;
	/** 匹配内容的正则表达式 */
	private String includeRegex;

	public FileContentSearch(File file, String includeRegex, String regex) {
		this(file, null, includeRegex, regex, "");
	}

	public FileContentSearch(File file, String encoding, String includeRegex, String regex, String replacement) {
		this.file = file;
		this.encoding = encoding;
		this.includeRegex = includeRegex;
		this.regex = regex;
		this.replacement = replacement;
	}

	/**
	 * 查找符合条件的内容，并根据正则替换掉多余的内容
	 * @return 查找到的内容
	 * @throws IOException
	 */
	public String findSingleLine() throws IOException {
		singleLine = true;
		List<String> find = find();
		if (AssertUtil.isEmpty(find)) {
			return null;
		} else {
			return find.get(0);
		}
	}

	/**
	 * 查找符合条件的内容，并根据正则替换掉多余的内容
	 * @return 查找到的内容
	 * @throws IOException
	 */
	public List<String> find() throws IOException {
		BufferedReader read = null;
		try {
			if (encoding == null) {
				encoding = CharsetDetectorUtil.getCharsetName(file.getAbsolutePath());
			}
			read = FileUtil.getBufferedReader(file, encoding);

			List<String> commandList = new ArrayList<String>();
			String line = null;
			while ((line = read.readLine()) != null) {
				line = line.trim();
				if (!isIncludeLine(line)) {
					continue;
				}
				line = replace(line);
				if (AssertUtil.isNotEmpty(line)) {
					commandList.add(line);
					if (singleLine) {
						break;
					}
				}
			}
			return commandList;
		} finally {
			CloseUtil.close(read);
		}
	}

	protected String replace(String line) {
		if (ignoreCase) {
			return Pattern.compile(regex, Pattern.CASE_INSENSITIVE).matcher(line).replaceAll(replacement).trim();
		} else {
			return Pattern.compile(regex).matcher(line).replaceAll(replacement).trim();
		}
	}

	/**
	 * 判断内容是否在包含的列表中
	 * @param line
	 * @return
	 */
	protected boolean isIncludeLine(String line) {
		if (ignoreCase) {
			return Pattern.compile(includeRegex, Pattern.CASE_INSENSITIVE).matcher(line).find();
		} else {
			return Pattern.compile(includeRegex).matcher(line).find();
		}
	}

	/**
	 * @return the file
	 */
	public File getFile() {
		return file;
	}

	/**
	 * @param file the file to set
	 */
	public void setFile(File file) {
		this.file = file;
	}

	/**
	 * @return the encoding
	 */
	public String getEncoding() {
		return encoding;
	}

	/**
	 * @param encoding the encoding to set
	 */
	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	/**
	 * @return the singleLine
	 */
	public boolean isSingleLine() {
		return singleLine;
	}

	/**
	 * @param singleLine the singleLine to set
	 */
	public void setSingleLine(boolean singleLine) {
		this.singleLine = singleLine;
	}

	/**
	 * @return the ignoreCase
	 */
	public boolean isIgnoreCase() {
		return ignoreCase;
	}

	/**
	 * @param ignoreCase the ignoreCase to set
	 */
	public void setIgnoreCase(boolean ignoreCase) {
		this.ignoreCase = ignoreCase;
	}

	/**
	 * @return the regex
	 */
	public String getRegex() {
		return regex;
	}

	/**
	 * @param regex the regex to set
	 */
	public void setRegex(String regex) {
		this.regex = regex;
	}

	/**
	 * @return the replacement
	 */
	public String getReplacement() {
		return replacement;
	}

	/**
	 * @param replacement the replacement to set
	 */
	public void setReplacement(String replacement) {
		this.replacement = replacement;
	}

	/**
	 * @return the includeRegex
	 */
	public String getIncludeRegex() {
		return includeRegex;
	}

	/**
	 * @param includeRegex the includeRegex to set
	 */
	public void setIncludeLineList(String includeRegex) {
		this.includeRegex = includeRegex;
	}

}
