package cn.elwy.common.util.encode;

import cn.elwy.common.util.AssertUtil;

/**
 * 常用加密算法工具类，包括MD5、SHA、DES、HMAC、Base64等
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class SecurityUtil {

	public static final String ENCRYPT_FLAG = "__CK_";

	private SecurityUtil() {
	}

	/**
	 * 判断是否已经加密
	 * @param data
	 * @return
	 */
	public static boolean isEncrypt(String data) {
		if (AssertUtil.isNotEmpty(data) && data.endsWith(ENCRYPT_FLAG)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 判断是否已经加密
	 * @param data
	 * @return
	 */
	public static boolean isEncryptAES(String data, String key) {
		try {
			if (key != null) {
				EncodeUtil.decryptAES(data, key);
			} else {
				EncodeUtil.decryptAES(data);
			}
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 判断是否已经加密
	 * @param data
	 * @return
	 */
	public static boolean isEncryptDES(String data, String key) {
		try {
			if (key != null) {
				EncodeUtil.decryptDES(data, key);
			} else {
				EncodeUtil.decryptDES(data);
			}
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * 用MD5算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return MD5加密后的结果
	 */
	public static String encryptMD5(String plaintext) {
		if (isEncrypt(plaintext)) {
			return plaintext;
		}
		return addEncryptFlag(EncodeUtil.encryptMD5(plaintext));
	}

	/**
	 * 用SHA算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return SHA加密后的结果
	 */
	public static String encryptSHA(String plaintext) {
		if (isEncrypt(plaintext)) {
			return plaintext;
		}
		return addEncryptFlag(EncodeUtil.encryptSHA256(plaintext));
	}

	/**
	 * HMAC加密
	 * @param plaintext
	 * @param key 密钥
	 * @return
	 */
	public static String encryptHMAC(String plaintext, String key) {
		if (isEncrypt(plaintext)) {
			return plaintext;
		}
		return addEncryptFlag(EncodeUtil.encryptHmacSHA256(plaintext, key));
	}

	/**
	 * HMAC加密
	 * @param plaintext
	 * @param key 密钥
	 * @param algorithm 支持算法：HmacMD5,HmacSHA1,HmacSHA256,HmacSHA384,HmacSHA512
	 * @return
	 */
	public static String encryptHMAC(String plaintext, String key, String algorithm) {
		if (isEncrypt(plaintext)) {
			return plaintext;
		}
		return addEncryptFlag(EncodeUtil.encryptHmac(plaintext, key, algorithm));
	}

	/**
	 * 用base64算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return base64加密后的结果
	 */
	public static String encryptBASE64(String plaintext) {
		if (isEncrypt(plaintext)) {
			return plaintext;
		}
		return addEncryptFlag(EncodeUtil.encryptBASE64(plaintext));
	}

	/**
	 * 用base64算法进行解密
	 * @param ciphertext 需要解密的字符串
	 * @return base64解密后的结果
	 */
	public static String decryptBASE64(String ciphertext) {
		if (!isEncrypt(ciphertext)) {
			return ciphertext;
		}
		ciphertext = getEncode(ciphertext);
		return EncodeUtil.decryptBASE64(ciphertext);
	}

	/**
	 * 对给定的字符串进行AES加密
	 * @param plaintext
	 * @return
	 */
	public static String encryptAES(String plaintext) {
		if (isEncryptAES(plaintext, null)) {
			return plaintext;
		}
		return EncodeUtil.encryptAES(plaintext);
	}

	/**
	 * 对给定的字符串进行AES加密
	 * @param plaintext
	 * @param key
	 * @return
	 */
	public static String encryptAES(String plaintext, String key) {
		if (isEncryptAES(plaintext, key)) {
			return plaintext;
		}
		return EncodeUtil.encryptAES(plaintext, key);
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @return
	 */
	public static String decryptAES(String ciphertext) {
		return EncodeUtil.decryptAES(ciphertext);
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @param key
	 * @return
	 */
	public static String decryptAES(String ciphertext, String key) {
		return EncodeUtil.decryptAES(ciphertext, key);
	}

	/**
	 * 对给定的字符串进行DES加密
	 * @param plaintext
	 * @return
	 */
	public static String encryptDES(String plaintext) {
		if (isEncryptDES(plaintext, null)) {
			return plaintext;
		}
		return EncodeUtil.encryptDES(plaintext);
	}

	/**
	 * 对给定的字符串进行DES加密
	 * @param plaintext
	 * @param key
	 * @return
	 */
	public static String encryptDES(String plaintext, String key) {
		if (isEncryptDES(plaintext, key)) {
			return plaintext;
		}
		return EncodeUtil.encryptDES(plaintext, key);
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @return
	 */
	public static String decryptDES(String ciphertext) {
		return EncodeUtil.decryptDES(ciphertext);
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @param key
	 * @return
	 */
	public static String decryptDES(String ciphertext, String key) {
		return EncodeUtil.decryptDES(ciphertext, key);
	}

	/**
	 * 获取密文编码
	 * @param ciphertext
	 * @return
	 */
	private static String getEncode(String ciphertext) {
		if (isEncrypt(ciphertext)) {
			ciphertext = ciphertext.substring(0, ciphertext.length() - ENCRYPT_FLAG.length());
		}
		return ciphertext;
	}

	/**
	 * 添加加密标识
	 * @param ciphertext
	 * @return
	 */
	private static String addEncryptFlag(String ciphertext) {
		if (AssertUtil.isNotEmpty(ciphertext)) {
			ciphertext = ciphertext + ENCRYPT_FLAG;
		}
		return ciphertext;
	}
}
