package cn.elwy.common.util.encode;

import java.io.File;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;

import javax.crypto.Cipher;

import cn.elwy.common.exception.EncodeException;
import cn.elwy.common.util.SerializeUtil;

/**
 * RSA加解密,RSA签名、签名验证类
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class RSAUtil {

	public static final String PUBLIC_KEY = "publicKey";
	public static final String PRIVATE_KEY = "privateKey";

	private static final String RSA = "RSA";
	private static int KEY_SIZE = 1024;

	private RSAUtil() {
	}

	/**
	 * 根据keyInfo产生公钥和私钥，并且保存到Map中。通过publicKey、privateKey获取相应的密钥
	 * @param keyInfo 密钥信息
	 * @return 公钥和私钥的Code
	 * @throws Exception
	 */
	public static Map<String, String> generateKeys(String keyInfo) {
		try {
			Map<String, String> keyPairMap = new HashMap<String, String>();
			KeyPair keyPair = getKeyPair(keyInfo);
			String privateKey = EncodeUtil.encryptBASE64(keyPair.getPrivate().getEncoded());
			String publicKey = EncodeUtil.encryptBASE64(keyPair.getPublic().getEncoded());

			keyPairMap.put(PRIVATE_KEY, privateKey);
			keyPairMap.put(PUBLIC_KEY, publicKey);
			return keyPairMap;
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 根据keyInfo产生公钥和私钥，并且保存到文件中
	 * @param keyInfo 密钥信息
	 * @param publicKeyFile 公钥文件
	 * @param privateKeyFile 私钥文件
	 * @throws Exception
	 */
	public static void generateKeys(String keyInfo, File publicKeyFile, File privateKeyFile) {
		try {
			KeyPair keyPair = getKeyPair(keyInfo);
			SerializeUtil.serialize(keyPair.getPublic(), publicKeyFile);
			SerializeUtil.serialize(keyPair.getPrivate(), privateKeyFile);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 使用公钥对给定的字符串进行RSA加密，解密需要使用私钥进行解密
	 * @param plaintext 明文
	 * @param publicKey 公钥
	 * @return
	 * @throws Exception
	 */
	public static String encryptByPublicKey(String plaintext, String publicKey) {
		Key key = getPublicKey(publicKey);
		byte[] ciphertext = EncodeUtil.encode(plaintext.getBytes(), key, Cipher.ENCRYPT_MODE);
		return EncodeUtil.encryptBASE64(ciphertext);
	}

	/**
	 * 使用私钥对给定的字符串进行RSA加密，解密需要使用公钥进行解密
	 * @param plaintext 明文
	 * @param privateKey 私钥
	 * @return
	 * @throws Exception
	 */
	public static String encryptByPrivateKey(String plaintext, String privateKey) {
		Key key = getPrivateKey(privateKey);
		byte[] ciphertext = EncodeUtil.encode(plaintext.getBytes(), key, Cipher.ENCRYPT_MODE);
		return EncodeUtil.encryptBASE64(ciphertext);
	}

	/**
	 * 使用公钥对给定的字符串进行RSA解密，解密需要使用私钥进行加密
	 * @param ciphertext 密文
	 * @param publicKey 公钥
	 * @return
	 * @throws Exception
	 */
	public static String decryptByPublicKey(String ciphertext, String publicKey) {
		Key key = getPublicKey(publicKey);
		byte[] ciphertextByte = EncodeUtil.decryptBASE64ToByte(ciphertext);
		byte[] plaintext = EncodeUtil.encode(ciphertextByte, key, Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 使用私钥对给定的字符串进行RSA解密，解密需要使用公钥进行加密
	 * @param ciphertext 密文
	 * @param privateKey 私钥
	 * @return
	 * @throws Exception
	 */
	public static String decryptByPrivateKey(String ciphertext, String privateKey) {
		Key key = getPrivateKey(privateKey);
		byte[] ciphertextByte = EncodeUtil.decryptBASE64ToByte(ciphertext);
		byte[] plaintext = EncodeUtil.encode(ciphertextByte, key, Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 用私钥签名
	 * @param plaintext
	 * @param privateKey
	 * @return
	 * @throws Exception
	 */
	public static String sign(String plaintext, String privateKey) {
		try {
			PrivateKey key = getPrivateKey(privateKey);
			Signature signature = getSignature();
			signature.initSign(key);
			signature.update(plaintext.getBytes());
			return EncodeUtil.encryptBASE64(signature.sign());
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 用公钥验证签名的正确性
	 * @param plaintext
	 * @param singInfo
	 * @return
	 * @throws Exception
	 */
	public static boolean verifySign(String plaintext, String publicKey, String singInfo) {
		if (plaintext == null || singInfo == null || publicKey == null) {
			return false;
		}
		try {
			PublicKey key = getPublicKey(publicKey);
			Signature signature = getSignature();
			signature.initVerify(key);
			signature.update(plaintext.getBytes());
			return signature.verify(EncodeUtil.decryptBASE64ToByte(singInfo));
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 获取密钥对
	 * @param keyInfo
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private static KeyPair getKeyPair(String keyInfo) throws NoSuchAlgorithmException {
		KeyPairGenerator keygen = KeyPairGenerator.getInstance(RSA);
		SecureRandom random = new SecureRandom();
		random.setSeed(keyInfo.getBytes());
		// 初始加密，长度为512，必须是大于512才可以的
		keygen.initialize(KEY_SIZE, random);
		// 取得密钥对
		return keygen.generateKeyPair();
	}

	/**
	 * 得到公钥
	 * @param keyCode 密钥字符串（经过base64编码）
	 * @throws Exception
	 */
	private static PublicKey getPublicKey(String keyCode) {
		byte[] keyBytes = EncodeUtil.decryptBASE64ToByte(keyCode);
		try {
			X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance(RSA);
			PublicKey publicKey = keyFactory.generatePublic(keySpec);
			return publicKey;
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 得到私钥
	 * @param keyCode 密钥字符串（经过base64编码）
	 * @throws Exception
	 */
	private static PrivateKey getPrivateKey(String keyCode) {
		byte[] keyBytes = EncodeUtil.decryptBASE64ToByte(keyCode);
		try {
			PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			PrivateKey privateKey = keyFactory.generatePrivate(keySpec);
			return privateKey;
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 获取RSA签名实例
	 * @return
	 * @throws NoSuchAlgorithmException
	 */
	private static Signature getSignature() throws NoSuchAlgorithmException {
		return Signature.getInstance("MD5withRSA");
	}

}
