package cn.elwy.common.util.encode;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.security.Key;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.Random;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.bind.DatatypeConverter;

import cn.elwy.common.exception.EncodeException;
import cn.elwy.common.util.CloseUtil;
import cn.elwy.common.util.SerializeUtil;
import cn.elwy.common.util.io.FileUtil;

/**
 * 常用加密算法工具类，包括MD5、SHA、DES、HMAC、Base62、Base64等
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class EncodeUtil {

	private static final String ENCRYPT_SEPARATOR = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

	public static final String MD5 = "MD5";
	public static final String SHA256 = "SHA-256";
	/** MAC算法可选以下多种算法 */
	public static final String HMAC_MD5 = "HmacMD5";
	public static final String HMAC_SHA1 = "HmacSHA1";
	public static final String HMAC_SHA256 = "HmacSHA256";
	public static final String HMAC_SHA384 = "HmacSHA384";
	public static final String HMAC_SHA512 = "HmacSHA512";

	public static final String AES = "AES";
	public static final String DES = "DES";

	private EncodeUtil() {
	}

	/**
	 * 用MD5算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return MD5加密后的结果
	 */
	public static String encryptMD5(String plaintext) {
		byte[] result = encode(plaintext.getBytes(), MD5);
		if (result != null) {
			return encryptBASE62(result);
		} else {
			return null;
		}
	}

	/**
	 * 用MD5算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return MD5加密后的结果
	 */
	public static byte[] encryptMD5(byte[] plaintext) {
		return encode(plaintext, MD5);
	}

	/**
	 * 用SHA算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return SHA加密后的结果
	 */
	public static String encryptSHA256(String plaintext) {
		byte[] result = encode(plaintext.getBytes(), SHA256);
		if (result != null) {
			return encryptBASE62(result);
		} else {
			return null;
		}
	}

	/**
	 * 用SHA算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return SHA加密后的结果
	 */
	public static byte[] encryptSHA256(byte[] plaintext) {
		return encode(plaintext, SHA256);
	}

	/**
	 * HMAC加密，默认使用HMAC_SHA256加密算法
	 * @param plaintext
	 * @param key 密钥
	 * @return
	 */
	public static String encryptHmacSHA256(String plaintext, String key) {
		byte[] result = encryptHmac(plaintext.getBytes(), key.getBytes(), HMAC_SHA256);
		return encryptBASE62(result);
	}

	/**
	 * HMAC加密
	 * @param plaintext
	 * @param key 密钥
	 * @param algorithm 支持算法：HmacMD5,HmacSHA1,HmacSHA256,HmacSHA384,HmacSHA512
	 * @return
	 */
	public static String encryptHmac(String plaintext, String key, String algorithm) {
		byte[] result = encryptHmac(plaintext.getBytes(), key.getBytes(), algorithm);
		return encryptBASE62(result);
	}

	/**
	 * HMAC加密
	 * @param plaintext
	 * @param key 密钥
	 * @param algorithm 支持算法：HmacMD5,HmacSHA1,HmacSHA256,HmacSHA384,HmacSHA512
	 * @return
	 */
	public static byte[] encryptHmac(byte[] plaintext, byte[] key, String algorithm) {
		try {
			SecretKey secretKey = new SecretKeySpec(key, algorithm);
			Mac mac = Mac.getInstance(secretKey.getAlgorithm());
			mac.init(secretKey);
			return mac.doFinal(plaintext);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 用base62算法进行加密
	 * @param plaintext 需要加密的字节
	 * @return base62加密后的结果
	 */
	public static String encryptBASE62(byte[] plaintext) {
		return Base62.encodeBase62(plaintext);
	}

	/**
	 * 用base62算法进行解密
	 * @param ciphertext 需要解密的字符串
	 * @return base62解密后的结果
	 */
	public static byte[] decryptBASE62ToByte(String ciphertext) {
		return Base62.decodeBase62(ciphertext);
	}

	/**
	 * 用base64算法进行加密
	 * @param plaintext 需要加密的字符串
	 * @return base64加密后的结果
	 */
	public static String encryptBASE64(String plaintext) {
		return encryptBASE64(plaintext.getBytes());
	}

	/**
	 * 用base64算法进行加密
	 * @param plaintext 需要加密的字节
	 * @return base64加密后的结果
	 */
	public static String encryptBASE64(byte[] plaintext) {
		return DatatypeConverter.printBase64Binary(plaintext);
	}

	/**
	 * 用base64算法进行解密
	 * @param ciphertext 需要解密的字符串
	 * @return base64解密后的结果
	 */
	public static byte[] decryptBASE64ToByte(String ciphertext) {
		return DatatypeConverter.parseBase64Binary(ciphertext);
	}

	/**
	 * 用base64算法进行解密
	 * @param ciphertext 需要解密的字符串
	 * @return base64解密后的结果
	 */
	public static String decryptBASE64(String ciphertext) {
		return new String(decryptBASE64ToByte(ciphertext));
	}

	/**
	 * 根据算法获取密钥
	 * @param algorithm 算法
	 * @return key 密钥
	 */
	public static Key getKeyByRandom(String algorithm) {
		try {
			// 一个可信任的随机数源
			SecureRandom sr = new SecureRandom();
			// 为我们选择的算法生成一个KeyGenerator对象
			KeyGenerator kg = KeyGenerator.getInstance(algorithm);
			kg.init(sr);
			Key key = kg.generateKey();
			return key;
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 从文件获得DES加密的密钥
	 * @param file 密钥文件
	 * @return Key 返回对称密钥
	 */
	public static Key getKeyByFile(File file) {
		return SerializeUtil.unSerialize(file, Key.class);
	}

	/**
	 * 保存密钥到指定文件
	 * @param algorithm
	 * @param filePath
	 */
	public static void saveKey(Key key, File file) {
		FileOutputStream fos = null;
		ObjectOutputStream oos = null;
		try {
			fos = new FileOutputStream(file);
			oos = new ObjectOutputStream(fos);
			oos.writeObject(key);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(oos);
			CloseUtil.close(fos);
		}
	}

	/**
	 * 对给定的字符串进行AES加密
	 * @param plaintext
	 * @return
	 */
	public static String encryptAES(String plaintext) {
		Key key = getKeyByRandom(AES);
		byte[] keyData = key.getEncoded();
		byte[] ciphertext = encodeAES(plaintext.getBytes(), keyData, Cipher.ENCRYPT_MODE);
		return toString(keyData, ciphertext);
	}

	/**
	 * 对给定的字符串进行AES加密
	 * @param plaintext
	 * @param key 密钥
	 * @return
	 */
	public static String encryptAES(String plaintext, String key) {
		byte[] ciphertext = encodeAES(plaintext.getBytes(), key.getBytes(), Cipher.ENCRYPT_MODE);
		return encryptBASE62(ciphertext);
	}

	/**
	 * 将内容加解密后保存到文件中
	 * @param plaintext 操作数据
	 * @param key 密钥
	 * @param destFile 输出路径
	 */
	public static void encryptAES(String plaintext, String key, File destFile) {
		encodeAES(plaintext.getBytes(), key.getBytes(), Cipher.ENCRYPT_MODE, destFile);
	}

	/**
	 * 读取文件srcFile内容进行加密，以字符串返回
	 * @param srcFile 要加密的文件
	 * @param key 密钥
	 * @return 加密后的字符串
	 */
	public static byte[] encryptAES(File srcFile, String key) {
		try {
			byte[] plaintext = FileUtil.readFileToBytes(srcFile.getAbsolutePath());
			return encodeAES(plaintext, key.getBytes(), Cipher.ENCRYPT_MODE);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 将文件srcFile进行加密并保存目标文件destFile中
	 * @param srcFile 要加密的文件
	 * @param destFile 加密后存放的文件
	 * @param key 密钥
	 */
	public static void encryptAES(File srcFile, File destFile, String key) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(AES, key.getBytes(), Cipher.ENCRYPT_MODE);
		InputStream is = null;
		OutputStream out = null;
		CipherInputStream cis = null;
		try {
			is = new FileInputStream(srcFile);
			out = new FileOutputStream(destFile);
			cis = new CipherInputStream(is, cipher);
			byte[] buffer = new byte[5120];
			int read;
			while ((read = cis.read(buffer)) > 0) {
				out.write(buffer, 0, read);
			}
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cis);
			CloseUtil.close(is);
			CloseUtil.close(out);
		}
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @return
	 */
	public static String decryptAES(String ciphertext) {
		if (ciphertext == null) {
			return null;
		}
		byte[][] bytes = getByteDatas(ciphertext);

		byte[] plaintext = encodeAES(bytes[0], bytes[1], Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @return
	 */
	public static String decryptAES(String ciphertext, String key) {
		if (ciphertext == null) {
			return null;
		}
		byte[] ciphertextByte = decryptBASE62ToByte(ciphertext);
		return decryptAES(ciphertextByte, key);
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @return
	 */
	public static String decryptAES(byte[] ciphertext, String key) {
		byte[] plaintext = encodeAES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 对给定的密文使用AES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @param file 输出路径
	 */
	public static void decryptAES(byte[] ciphertext, String key, File file) {
		encodeAES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE, file);
	}

	/**
	 * 读取文件srcFile内容进行解密，以字符串返回
	 * @param srcFile 要加密的文件
	 * @param key 密钥
	 * @return 解密后的字符串
	 */
	public static String decryptAES(File srcFile, String key) {
		try {
			byte[] ciphertext = FileUtil.readFileToBytes(srcFile.getAbsolutePath());
			byte[] plaintext = encodeAES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE);
			return new String(plaintext);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 将文件srcFile进行解密并保存目标文件destFile中
	 * @param srcFile 已加密的文件
	 * @param destFile 解密后存放的文件
	 * @param key 密钥
	 */
	public static void decryptAES(File srcFile, File destFile, String key) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(AES, key.getBytes(), Cipher.DECRYPT_MODE);
		InputStream is = null;
		OutputStream out = null;
		CipherOutputStream cos = null;
		try {
			is = new FileInputStream(srcFile);
			out = new FileOutputStream(destFile);
			cos = new CipherOutputStream(out, cipher);
			byte[] buffer = new byte[5120];
			int read;
			while ((read = is.read(buffer)) > 0) {
				cos.write(buffer, 0, read);
			}
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cos);
			CloseUtil.close(is);
			CloseUtil.close(out);
		}
	}

	/**
	 * 将内容加解密后保存到文件中
	 * @param data 操作数据
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @param destFile 输出路径
	 */
	public static void encodeAES(byte[] data, byte[] key, int opmode, File destFile) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(AES, key, opmode);
		OutputStream out = null;
		CipherOutputStream cos = null;
		try {
			out = new FileOutputStream(destFile);
			cos = new CipherOutputStream(out, cipher);
			cos.write(data);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cos);
			CloseUtil.close(out);
		}
	}

	/**
	 * 执行加密或解密操作，并返回加解密结果
	 * @param data 操作数据
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @return
	 */
	public static byte[] encodeAES(byte[] data, byte[] key, int opmode) {
		try {
			Cipher cipher = getCipher(AES, key, opmode);
			return cipher.doFinal(data);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 对给定的字符串进行DES加密
	 * @param plaintext
	 * @return
	 */
	public static String encryptDES(String plaintext) {
		Key key = getKeyByRandom(DES);
		byte[] keyData = key.getEncoded();
		byte[] ciphertext = encodeDES(plaintext.getBytes(), keyData, Cipher.ENCRYPT_MODE);
		return toString(keyData, ciphertext);
	}

	/**
	 * 对给定的字符串进行DES加密
	 * @param plaintext
	 * @param key 密钥
	 * @return
	 */
	public static String encryptDES(String plaintext, String key) {
		byte[] ciphertext = encodeDES(plaintext.getBytes(), key.getBytes(), Cipher.ENCRYPT_MODE);
		return encryptBASE62(ciphertext);
	}

	/**
	 * 将内容加解密后保存到文件中
	 * @param plaintext 操作数据
	 * @param key 密钥
	 * @param destFile 输出路径
	 */
	public static void encryptDES(String plaintext, String key, File destFile) {
		encodeDES(plaintext.getBytes(), key.getBytes(), Cipher.ENCRYPT_MODE, destFile);
	}

	/**
	 * 读取文件srcFile内容进行加密，以字符串返回
	 * @param srcFile 要加密的文件
	 * @param key 密钥
	 * @return 加密后的字符串
	 */
	public static byte[] encryptDES(File srcFile, String key) {
		try {
			byte[] plaintext = FileUtil.readFileToBytes(srcFile.getAbsolutePath());
			return encodeDES(plaintext, key.getBytes(), Cipher.ENCRYPT_MODE);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 将文件srcFile进行加密并保存目标文件destFile中
	 * @param srcFile 要加密的文件
	 * @param destFile 加密后存放的文件
	 * @param key 密钥
	 */
	public static void encryptDES(File srcFile, File destFile, String key) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(DES, key.getBytes(), Cipher.ENCRYPT_MODE);
		InputStream is = null;
		OutputStream out = null;
		CipherInputStream cis = null;
		try {
			is = new FileInputStream(srcFile);
			out = new FileOutputStream(destFile);
			cis = new CipherInputStream(is, cipher);
			byte[] buffer = new byte[5120];
			int read;
			while ((read = cis.read(buffer)) > 0) {
				out.write(buffer, 0, read);
			}
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cis);
			CloseUtil.close(is);
			CloseUtil.close(out);
		}
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @return
	 */
	public static String decryptDES(String ciphertext) {
		if (ciphertext == null) {
			return null;
		}
		byte[][] bytes = getByteDatas(ciphertext);
		byte[] plaintext = encodeDES(bytes[0], bytes[1], Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @return
	 */
	public static String decryptDES(String ciphertext, String key) {
		if (ciphertext == null) {
			return null;
		}
		byte[] ciphertextByte = decryptBASE62ToByte(ciphertext);
		return decryptDES(ciphertextByte, key);
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @return
	 */
	public static String decryptDES(byte[] ciphertext, String key) {
		byte[] plaintext = encodeDES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE);
		return new String(plaintext);
	}

	/**
	 * 对给定的密文使用DES进行解密
	 * @param ciphertext
	 * @param key 密钥
	 * @param file 输出路径
	 */
	public static void decryptDES(byte[] ciphertext, String key, File file) {
		encodeDES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE, file);
	}

	/**
	 * 读取文件srcFile内容进行解密，以字符串返回
	 * @param srcFile 要加密的文件
	 * @param key 密钥
	 * @return 解密后的字符串
	 */
	public static String decryptDES(File srcFile, String key) {
		try {
			byte[] ciphertext = FileUtil.readFileToBytes(srcFile.getAbsolutePath());
			byte[] plaintext = encodeDES(ciphertext, key.getBytes(), Cipher.DECRYPT_MODE);
			return new String(plaintext);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 将文件srcFile进行解密并保存目标文件destFile中
	 * @param srcFile 已加密的文件
	 * @param destFile 解密后存放的文件
	 * @param key 密钥
	 */
	public static void decryptDES(File srcFile, File destFile, String key) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(DES, key.getBytes(), Cipher.DECRYPT_MODE);
		InputStream is = null;
		OutputStream out = null;
		CipherOutputStream cos = null;
		try {
			is = new FileInputStream(srcFile);
			out = new FileOutputStream(destFile);
			cos = new CipherOutputStream(out, cipher);
			byte[] buffer = new byte[5120];
			int read;
			while ((read = is.read(buffer)) > 0) {
				cos.write(buffer, 0, read);
			}
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cos);
			CloseUtil.close(is);
			CloseUtil.close(out);
		}
	}

	/**
	 * 将内容加解密后保存到文件中
	 * @param data 操作数据
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @param destFile 输出路径
	 */
	public static void encodeDES(byte[] data, byte[] key, int opmode, File destFile) {
		if (!destFile.getParentFile().exists()) {
			destFile.getParentFile().mkdirs();
		}

		Cipher cipher = getCipher(DES, key, opmode);
		OutputStream out = null;
		CipherOutputStream cos = null;
		try {
			out = new FileOutputStream(destFile);
			cos = new CipherOutputStream(out, cipher);
			cos.write(data);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		} finally {
			CloseUtil.close(cos);
			CloseUtil.close(out);
		}
	}

	/**
	 * 执行加密或解密操作，并返回加解密结果
	 * @param data 操作数据
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @return
	 */
	public static byte[] encodeDES(byte[] data, byte[] key, int opmode) {
		try {
			Cipher cipher = getCipher(DES, key, opmode);
			return cipher.doFinal(data);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 执行加密或解密操作，并返回加解密结果
	 * @param data 操作数据
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @return
	 */
	protected static byte[] encode(byte[] data, Key key, int opmode) {
		try {
			Cipher cipher = Cipher.getInstance(key.getAlgorithm());
			cipher.init(opmode, key);
			return cipher.doFinal(data);
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 获取Cipher对象
	 * @param algorithm 算法
	 * @param key 密钥
	 * @param opmode 可选值Cipher.ENCRYPT_MODE | Cipher.DECRYPT_MODE
	 * @return
	 */
	private static Cipher getCipher(String algorithm, byte[] key, int opmode) {
		try {
			KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithm);
			// 生成一个可信任的随机数源
			SecureRandom secureRandom = null;
			if (key == null) {
				secureRandom = new SecureRandom();
			} else {
				secureRandom = new SecureRandom(key);
			}
			keyGenerator.init(secureRandom);
			SecretKey securekey = keyGenerator.generateKey();

			// Cipher对象实际完成解密操作
			Cipher cipher = Cipher.getInstance(algorithm);
			// 用密钥初始化Cipher对象
			cipher.init(opmode, securekey);
			return cipher;
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	/**
	 * 加密字符串，支持MD5，SHA算法
	 * @param plaintext 需要加密的字符串
	 * @param algorithm 加密方法（MD5或SHA）
	 * @return 加密后的结果
	 */
	private static byte[] encode(byte[] plaintext, String algorithm) {
		MessageDigest md = null;
		try {
			md = MessageDigest.getInstance(algorithm);
			md.update(plaintext);
			return md.digest();
		} catch (Exception e) {
			throw new EncodeException(e.getMessage(), e);
		}
	}

	private static String toString(byte[] keyData, byte[] ciphertext) {
		StringBuffer text = new StringBuffer();
		text.append(encryptBASE62(ciphertext)).append(",").append(separator()).append(encryptBASE62(keyData));
		return text.toString();
	}

	/**
	 * 解析密钥和密文
	 * @param ciphertext
	 * @return
	 */
	private static byte[][] getByteDatas(String ciphertext) {
		String[] str = ciphertext.split(",.{9}");
		byte[][] bytes = new byte[2][];
		bytes[0] = getByteData(str[0]);
		bytes[1] = getByteData(str[1]);
		return bytes;
	}

	/**
	 * 解析密钥和密文
	 * @param ciphertext
	 * @return
	 */
	private static byte[] getByteData(String ciphertext) {
		if (ciphertext == null) {
			return null;
		}
		return decryptBASE62ToByte(ciphertext);
	}

	public static String separator() {
		return randomChar(9);
	}

	public static String randomChar(int length) {
		StringBuffer sb = new StringBuffer();
		int range = ENCRYPT_SEPARATOR.length();
		Random random = new Random();
		for (int i = 0; i < length; i++) {
			sb.append(ENCRYPT_SEPARATOR.charAt(random.nextInt(range)));
		}
		return sb.toString();
	}

	/**
	 * Base62工具类
	 * @author huangsq
	 * @version 1.0, 2018-02-19
	 */
	static class Base62 {

		private static char[] encodes = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz+/".toCharArray();
		private static byte[] decodes = new byte[256];

		static {
			for (int i = 0; i < encodes.length; i++) {
				decodes[encodes[i]] = (byte) i;
			}
		}

		public static String encodeBase62(byte[] data) {
			StringBuffer sb = new StringBuffer(data.length * 2);
			int pos = 0, val = 0;
			for (int i = 0; i < data.length; i++) {
				val = (val << 8) | (data[i] & 0xFF);
				pos += 8;
				while (pos > 5) {
					char c = encodes[val >> (pos -= 6)];
					sb.append(c == 'i' ? "ia" : c == '+' ? "ib" : c == '/' ? "ic" : c);
					val &= ((1 << pos) - 1);
				}
			}
			if (pos > 0) {
				char c = encodes[val << (6 - pos)];
				sb.append(c == 'i' ? "ia" : c == '+' ? "ib" : c == '/' ? "ic" : c);
			}
			return sb.toString();
		}

		public static byte[] decodeBase62(String text) {
			char[] data = text.toCharArray();
			ByteArrayOutputStream baos = new ByteArrayOutputStream(data.length);
			int pos = 0, val = 0;
			for (int i = 0; i < data.length; i++) {
				char c = data[i];
				if (c == 'i') {
					c = data[++i];
					c = c == 'a' ? 'i' : c == 'b' ? '+' : c == 'c' ? '/' : data[--i];
				}
				val = (val << 6) | decodes[c];
				pos += 6;
				while (pos > 7) {
					baos.write(val >> (pos -= 8));
					val &= ((1 << pos) - 1);
				}
			}
			return baos.toByteArray();
		}

	}
}
