package cn.elwy.common.util;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;

import cn.elwy.common.log.Logger;
import cn.elwy.common.log.LoggerFactory;

/**
 * Zip压缩文件工具类.
 * @author huangsq
 * @version 1.0, 2018-02-19
 */
public final class ZipUtil {

	private static Logger logger = LoggerFactory.getLogger(ZipUtil.class);

	private static final String DEFAULT_ENCODE = System.getProperty("sun.jnu.encoding");
	public static final String UTF8_ENCODE = "UTF-8";
	public static final String GBK_ENCODE = "GBK";
	private static final String ZIP = ".zip";
	private static final int BUFFER_SIZE = 81920; // 80KB

	private ZipUtil() {
	}

	/**
	 * 将一个文件或目录压缩成一个zip文件
	 * @param srcPath 文件或目录路径
	 * @param destPath 目标路径
	 * @throws IOException
	 */
	public static void compress(String srcPath, String destPath) throws IOException {
		compress(new File(srcPath), new File(destPath), null);
	}

	/**
	 * 将一个文件或目录压缩成一个zip文件
	 * @param srcFile 文件或目录
	 * @param destFile 目标文件
	 * @throws IOException
	 */
	public static void compress(File srcFile, File destFile) throws IOException {
		compress(srcFile, destFile, null);
	}

	/**
	 * 将一个文件或目录压缩成一个zip文件
	 * @param srcFile 文件或目录
	 * @param destFile 目标文件
	 * @param filter 过滤文件(正则表达式)
	 * @throws IOException
	 */
	public static void compress(File srcFile, File destFile, List<String> filter) throws IOException {
		List<File> fileList = new ArrayList<File>();
		fileList.add(srcFile);
		compress(fileList, destFile, DEFAULT_ENCODE, filter, false, null);
	}

	/**
	 * 将一组文件或目录压缩成一个zip文件
	 * @param fileList 文件或目录列表
	 * @param destFile 目标文件
	 * @throws IOException
	 */
	public static void compress(List<File> fileList, File destFile) throws IOException {
		compress(fileList, destFile, DEFAULT_ENCODE, null, false, null);
	}

	/**
	 * 将一组文件或目录压缩成一个zip文件
	 * @param fileList 文件或目录列表
	 * @param destFile 目标文件
	 * @param filter 过滤文件(正则表达式)
	 * @throws IOException
	 */
	public static void compress(List<File> fileList, File destFile, List<String> filter) throws IOException {
		compress(fileList, destFile, DEFAULT_ENCODE, filter, false, null);
	}

	/**
	 * 将一组文件或目录压缩成一个zip文件
	 * @param fileList 文件或目录列表
	 * @param destFile 目标文件
	 * @param encoding 文件名称编码
	 * @param filter 过滤文件(正则表达式)
	 * @param ignoreError 忽略错误
	 * @param comment 压缩包注释
	 * @throws IOException
	 */
	public static void compress(List<File> fileList, File destFile, String encoding, List<String> filter,
			boolean ignoreError, String comment) throws IOException {
		if (destFile.isDirectory()) {
			String name = destFile.getName() + ZIP;
			destFile = new File(destFile, name);
			if (filter != null) {
				filter = new ArrayList<String>();
			}
			filter.add(name);
		}
		if (isEmpty(encoding)) {
			encoding = GBK_ENCODE;
		}
		ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(destFile));
		try {
			zos.setEncoding(encoding);
			for (File file : fileList) {
				if (fileList.size() == 1) {
					compress(zos, file, "", filter, ignoreError, comment);
				} else {
					compress(zos, file, file.getName(), filter, ignoreError, comment);
				}
			}
		} finally {
			close(zos);
		}
	}

	/**
	 * 添加文件到压缩包中
	 * @param zipFile
	 * @param file
	 * @param entryPath
	 * @throws IOException
	 */
	public static void addFileToZip(File zipFile, File file, String entryPath) throws IOException {
		addFileToZip(zipFile, new File[] { file }, new String[] { entryPath });
	}

	/**
	 * 添加文件到压缩包中
	 * @param zipFile
	 * @param files
	 * @param entryPaths
	 * @throws IOException
	 */
	public static void addFileToZip(File zipFile, File[] files, String[] entryPaths) throws IOException {
		File tempFile = File.createTempFile(zipFile.getName(), null);
		tempFile.delete();
		boolean renameOk = zipFile.renameTo(tempFile);
		if (!renameOk) {
			throw new RuntimeException(
					"could not rename the file " + zipFile.getAbsolutePath() + " to " + tempFile.getAbsolutePath());
		}
		addFileToZip(tempFile, zipFile, DEFAULT_ENCODE, files, entryPaths);
	}

	/**
	 * 添加文件到压缩包中
	 * @param srcZip
	 * @param destZip
	 * @param encoding 文件名称编码
	 * @param files
	 * @param entryPaths
	 * @throws IOException
	 */
	public static void addFileToZip(File srcZip, File destZip, String encoding, File[] files, String[] entryPaths)
			throws IOException {
		byte[] buffer = new byte[BUFFER_SIZE];
		if (isEmpty(encoding)) {
			encoding = GBK_ENCODE;
		}
		ZipFile zipFile = new ZipFile(srcZip, encoding);
		ZipOutputStream out = new ZipOutputStream(new FileOutputStream(destZip));
		try {
			out.setEncoding(encoding);
			Enumeration<?> e = zipFile.getEntries();
			while (e.hasMoreElements()) {
				ZipEntry entry = (ZipEntry) e.nextElement();
				String name = entry.getName();
				boolean notInFiles = true;

				String entryName = "";
				for (int i = 0; i < files.length; i++) {
					File file = files[i];
					if (entryPaths != null && i < entryPaths.length) {
						entryName = entryPaths[i];
					}
					if ((entryName + "/" + file.getName()).equals(name)) {
						notInFiles = false;
						break;
					}
				}
				if (notInFiles) {
					out.putNextEntry(new ZipEntry(name));
					InputStream is = zipFile.getInputStream(entry);
					int length = 0;
					while ((length = is.read(buffer, 0, BUFFER_SIZE)) != -1) {
						out.write(buffer, 0, length);
					}
				}
			}
			String entryName = "";
			for (int i = 0; i < files.length; i++) {
				File file = files[i];
				if (entryPaths != null && i < entryPaths.length) {
					entryName = entryPaths[i];
				}
				if (file.isDirectory()) {
					compress(out, file, entryName, null, false, "");
				} else {
					compress(out, file, entryName + "/" + file.getName(), null, false, "");
				}
			}
		} finally {
			close(zipFile);
			close(out);
		}
	}

	/**
	 * 将一组文件或目录压缩到一个Zip输出流中
	 * @param out zip输出流
	 * @param srcFile 文件或目录
	 * @param name 压缩包中的名称
	 * @param filter 过滤文件(正则表达式)
	 * @param ignoreError 忽略错误
	 * @param comment 压缩包注释
	 * @throws IOException
	 */
	private static void compress(ZipOutputStream out, File srcFile, String name, List<String> filter, boolean ignoreError,
			String comment) throws IOException {
		if (srcFile.exists() == false) {
			throw new IOException(srcFile.getAbsolutePath() + " file does not exist");
		}
		try {
			if (srcFile.isDirectory()) {
				File[] files = srcFile.listFiles();
				name = name.length() == 0 ? "" : name + "/";
				if (!isFilter(name, filter) && name.length() != 0) {
					out.putNextEntry(new ZipEntry(name));
				}
				for (File file : files) {
					compress(out, file, name + file.getName(), filter, ignoreError, comment);
				}
			} else {
				if (!isFilter(name, filter)) {
					name = name.length() == 0 ? srcFile.getName() : name;
					ZipEntry zipEntry = new ZipEntry(name);
					if (comment != null) {
						zipEntry.setComment(comment);
					}
					out.putNextEntry(zipEntry);
					FileInputStream in = null;
					try {
						in = new FileInputStream(srcFile);
						byte[] buffer = new byte[BUFFER_SIZE];
						int length;
						while ((length = in.read(buffer)) != -1) {
							out.write(buffer, 0, length);
						}
						out.flush();
					} finally {
						close(in);
					}
				}
			}
		} catch (IOException e) {
			if (ignoreError) {
				logger.warn("compress zip ignore error: ");
				logger.warn(e.getMessage(), e);
			} else {
				throw e;
			}
		}
	}

	/**
	 * 指定的名称是否被过滤
	 * @param base
	 * @param filter 过滤列表(正则表达式)
	 * @return
	 */
	private static boolean isFilter(String base, List<String> filter) {
		if (filter != null && !filter.isEmpty()) {
			for (int i = 0; i < filter.size(); i++) {
				Pattern pat = Pattern.compile(filter.get(i));
				Matcher mat = pat.matcher(base);
				if (mat.find()) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * 将一个zip文件解压到一个目录
	 * @param srcPath 压缩包路径
	 * @param destDir 解压位置
	 * @throws IOException
	 */
	public static void uncompress(String srcPath, String destDir) throws IOException {
		uncompress(new File(srcPath), new File(destDir), DEFAULT_ENCODE);
	}

	/**
	 * 将一个zip文件解压到一个目录
	 * @param srcFile 压缩包文件
	 * @param destDir 解压位置
	 * @param encoding 文件名称编码
	 * @throws IOException
	 */
	public static void uncompress(File srcFile, File destDir, String encoding) throws IOException {
		if (!srcFile.exists()) {
			throw new IOException(srcFile.getAbsolutePath() + " file does not exist");
		}
		if (destDir.isFile()) {
			throw new IOException(srcFile.getAbsolutePath() + " is not a folder");
		}
		if (isEmpty(encoding)) {
			encoding = GBK_ENCODE;
		}
		ZipFile zipFile = null;
		/*-
		try {
			zipFile = new ZipFile(srcFile, encoding);
			Enumeration<?> e = zipFile.getEntries();
			while (e.hasMoreElements()) {
				ZipEntry zipEntry = (ZipEntry) e.nextElement();
				String name = zipEntry.getName();
				if (!java.nio.charset.Charset.forName(GBK_ENCODE).newEncoder().canEncode(name)) {
					encoding = UTF8_ENCODE;
					break;
				}
			}
		} finally {
			close(zipFile);
		}
		*/
		try {
			zipFile = new ZipFile(srcFile, encoding);
			Enumeration<?> e = zipFile.getEntries();
			while (e.hasMoreElements()) {
				ZipEntry zipEntry = (ZipEntry) e.nextElement();
				// 会把目录作为一个file读出一次，所以只建立目录就可以，之下的文件还会被迭代到。
				if (zipEntry.isDirectory()) {
					File f = new File(destDir, zipEntry.getName());
					f.mkdirs();
				} else {
					File f = new File(destDir, zipEntry.getName());
					f.getParentFile().mkdirs();
					InputStream is = null;
					FileOutputStream fos = null;
					try {
						is = zipFile.getInputStream(zipEntry);
						fos = new FileOutputStream(f);
						int length = 0;
						byte[] buffer = new byte[BUFFER_SIZE];
						while ((length = is.read(buffer)) != -1) {
							fos.write(buffer, 0, length);
						}
						fos.flush();
					} finally {
						close(fos);
						close(is);
					}
				}
			}
		} catch (Exception e) {
			throw new IOException(e.getMessage(), e);
		} finally {
			close(zipFile);
		}
	}

	/**
	 * 读取压缩包的注释
	 * @param filePath
	 * @return
	 * @throws IOException
	 */
	public static String getZipComment(String filePath) throws IOException {
		return getZipComment(filePath, DEFAULT_ENCODE);
	}

	/**
	 * 读取压缩包的注释
	 * @param filePath
	 * @param encoding
	 * @return
	 * @throws IOException
	 */
	public static String getZipComment(String filePath, String encoding) throws IOException {
		ZipFile zipFile = null;
		if (isEmpty(encoding)) {
			encoding = GBK_ENCODE;
		}
		try {
			zipFile = new ZipFile(filePath, encoding);
			Enumeration<?> e = zipFile.getEntries();
			String comment = "";
			while (e.hasMoreElements()) {
				ZipEntry ze = (ZipEntry) e.nextElement();
				comment = ze.getComment();
				if (comment != null && !comment.equals("") && !comment.equals("null")) {
					break;
				}
			}
			return comment;
		} finally {
			close(zipFile);
		}
	}

	public static boolean isEmpty(String text) {
		return text == null || text.trim().length() < 1;
	}

	private static void close(Closeable object) {
		if (object != null) {
			try {
				object.close();
			} catch (IOException e) {
				logger.warn(e.getMessage(), e);
			}
		}
	}

	private static void close(ZipFile zipFile) {
		if (zipFile != null) {
			try {
				zipFile.close();
			} catch (IOException e) {
				logger.warn(e.getMessage(), e);
			}
		}
	}
}